package org.bouncycastle.jsse.provider;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.net.ssl.SSLEngine;
import java.net.Socket;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.Provider;
import java.security.cert.*;
import java.util.*;

class ProvX509TrustManagerImpl implements ProvX509TrustManager {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private Set<X509Certificate> getTrustedCerts(Set<TrustAnchor> trustAnchors) {
        Set<X509Certificate> result = new HashSet<X509Certificate>(trustAnchors.size());
        for (TrustAnchor trustAnchor : trustAnchors) {
            if (trustAnchor != null) {
                X509Certificate trustedCert = trustAnchor.getTrustedCert();
                if (trustedCert != null) {
                    result.add(trustedCert);
                }
            }
        }
        return result;
    }

    private final Provider pkixProvider;
    private final Set<X509Certificate> trustedCerts;
    private final PKIXParameters baseParameters;

    ProvX509TrustManagerImpl(Provider pkixProvider, Set<TrustAnchor> trustAnchors) throws InvalidAlgorithmParameterException {
        this.pkixProvider = pkixProvider;
        this.trustedCerts = getTrustedCerts(trustAnchors);
        this.baseParameters = new PKIXBuilderParameters(trustAnchors, new X509CertSelector());
        this.baseParameters.setRevocationEnabled(false);
    }

    ProvX509TrustManagerImpl(Provider pkixProvider, PKIXParameters baseParameters) throws InvalidAlgorithmParameterException {
        this.pkixProvider = pkixProvider;
        this.trustedCerts = getTrustedCerts(baseParameters.getTrustAnchors());
        if (baseParameters instanceof PKIXBuilderParameters) {
            this.baseParameters = baseParameters;
        } else {
            this.baseParameters = new PKIXBuilderParameters(baseParameters.getTrustAnchors(), baseParameters.getTargetCertConstraints());
            this.baseParameters.setCertStores(baseParameters.getCertStores());
            this.baseParameters.setRevocationEnabled(baseParameters.isRevocationEnabled());
            this.baseParameters.setCertPathCheckers(baseParameters.getCertPathCheckers());
            this.baseParameters.setDate(baseParameters.getDate());
            this.baseParameters.setAnyPolicyInhibited(baseParameters.isAnyPolicyInhibited());
            this.baseParameters.setPolicyMappingInhibited(baseParameters.isPolicyMappingInhibited());
            this.baseParameters.setExplicitPolicyRequired(baseParameters.isExplicitPolicyRequired());
        }
    }

    @Override
    public void checkClientTrusted(X509Certificate[] x509Certificates, String authType) throws CertificateException {
        // TODO: need to confirm cert and client identity match  需要确认证书与客户身份匹配
        // TODO: need to make sure authType makes sense.   需要确保authType有意义。
        validatePath(x509Certificates);
    }

    @Override
    public void checkServerTrusted(X509Certificate[] x509Certificates, String authType) throws CertificateException {
        // TODO: need to confirm cert and server identity match
        // TODO: need to make sure authType makes sense.
        validatePath(x509Certificates);
    }

    @Override
    public void checkClientTrusted(X509Certificate[] x509Certificates, String authType, Socket socket) throws CertificateException {
        // TODO: need to confirm cert and client identity match
        // TODO: need to make sure authType makes sense.
        validatePath(x509Certificates);
    }

    @Override
    public void checkServerTrusted(X509Certificate[] x509Certificates, String authType, Socket socket) throws CertificateException {
        // TODO: need to confirm cert and server identity match
        // TODO: need to make sure authType makes sense.
        validatePath(x509Certificates);
    }

    @Override
    public void checkClientTrusted(X509Certificate[] x509Certificates, String authType, SSLEngine sslEngine) throws CertificateException {
        // TODO: need to confirm cert and client identity match
        // TODO: need to make sure authType makes sense.
        validatePath(x509Certificates);
    }

    @Override
    public void checkServerTrusted(X509Certificate[] x509Certificates, String authType, SSLEngine sslEngine) throws CertificateException {
        // TODO: need to confirm cert and server identity match
        // TODO: need to make sure authType makes sense.
        validatePath(x509Certificates);
    }

    @Override
    public X509Certificate[] getAcceptedIssuers() {
        return trustedCerts.toArray(new X509Certificate[trustedCerts.size()]);
    }

    // 证书验证 这里依赖的 pkixProvider 为 BcTlsCryptoProvider.getPkixProvider()
    // 需要为 BC provider 否则解析不了 证书 抛异常 不能识别证书 bad certificate
    // 如果上述没有问题，可能是因为项目idea设置引用了 maven 的jsse provider，未改动BcTlsCryptoProvider.getPkixProvider()的jar包版本
    protected void validatePath(X509Certificate[] x509Certificates) throws CertificateException {

        if (x509Certificates == null || x509Certificates.length < 1) {
            throw new IllegalArgumentException("'x509Certificates' must be a chain of at least one certificate");
        }

        logger.debug(
                "validatePath certificate subjectDN: {} sn: {} issueDN: {}",
                x509Certificates[0].getSubjectDN().toString(),
                x509Certificates[0].getSerialNumber(),
                x509Certificates[0].getIssuerDN().toString()
        );

        X509Certificate eeCert = x509Certificates[0];
        if (trustedCerts.contains(eeCert)) {
            return;
        }

        try {
            CertStore certStore = CertStore.getInstance("Collection",
                    new CollectionCertStoreParameters(Arrays.asList(x509Certificates)), pkixProvider);

            CertPathBuilder pathBuilder = CertPathBuilder.getInstance("PKIX", pkixProvider);

            X509CertSelector constraints = (X509CertSelector) baseParameters.getTargetCertConstraints().clone();

            constraints.setCertificate(eeCert);

            PKIXBuilderParameters param = (PKIXBuilderParameters) baseParameters.clone();

            param.addCertStore(certStore);
            param.setTargetCertConstraints(constraints);

            PKIXCertPathValidatorResult result = (PKIXCertPathValidatorResult) pathBuilder.build(param);
            result.getTrustAnchor();
        } catch (GeneralSecurityException e) {
            throw new CertificateException("unable to process certificates: " + e.getMessage(), e);
        }
    }
}
