package org.bouncycastle.tls.crypto.impl.bc;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.crypto.sdf.SdfECKeyParameters;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfSM2Engine;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2EncryptUtils;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ECKeyParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.util.PrivateKeyFactory;
import org.bouncycastle.tls.*;
import org.bouncycastle.tls.crypto.TlsCertificate;
import org.bouncycastle.tls.crypto.TlsCryptoParameters;
import org.bouncycastle.tls.crypto.TlsSecret;
import org.bouncycastle.tls.crypto.TlsSigner;
import org.bouncycastle.tls.crypto.impl.AbstractTlsCrypto;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.PrivateKey;

/**
 * Credentialed class for generating signatures based on the use of primitives from the BC light-weight API.
 */
//GMSSL SUPPORT: : 2018/7/31 for ecc sm2 credentialed signer BcTlsSM2Signer && decryptor to decrypt pre master secret
public class BcDefaultTlsCredentialedECCSM2 extends DefaultTlsCredentialedSigner implements TlsCredentialedDecryptor {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    protected AbstractTlsCrypto crypto;

    //签名 private key
    protected PrivateKey signaturePrivateKey;
    //加密 private key
    protected PrivateKey encryptionPrivateKey;

    //签名证书在前 加密证书在后
    protected Certificate gmsslCertificate;

    private static TlsSigner makeSigner(AbstractTlsCrypto crypto, PrivateKey privateKey) {
        if (crypto instanceof BcTlsCrypto){
            PrivateKeyInfo privateKeyInfo = PrivateKeyInfo.getInstance(privateKey.getEncoded());
            if (privateKeyInfo == null) {
                throw new IllegalArgumentException("'privateKey' type not supported: " + privateKey.getClass().getName());
            }
            AsymmetricKeyParameter parameter;
            try {
                parameter = PrivateKeyFactory.createKey(privateKeyInfo);
            } catch (IOException e) {
                throw new IllegalArgumentException("'privateKey' type not supported: " + privateKey.getClass().getName());
            }
            TlsSigner signer;
            if (parameter instanceof ECPrivateKeyParameters) {
                signer = new BcTlsSM2Signer((BcTlsCrypto) crypto, parameter);
            } else {
                throw new IllegalArgumentException("'privateKey' type not supported: " + privateKey.getClass().getName());
            }
            return signer;
        } else if (crypto instanceof BcTlsCryptoSdf){
            if (!(privateKey instanceof SdfPrivateKey)){
                throw new IllegalArgumentException("'privateKey' type not supported: " + privateKey.getClass().getName());
            }
            return new BcTlsSM2SignerSdf((BcTlsCryptoSdf) crypto, new SdfECKeyParameters((SdfPrivateKey) privateKey));
        } else {
            throw new IllegalArgumentException("un supported: " + crypto.getClass().getName());
        }
    }

    //GMSSL SUPPORT:  签名证书在前 加密证书在后
    private static Certificate makeSignerCertificate(Certificate gmsslCertificate) {
        return new Certificate(new TlsCertificate[]{
                GMSSLUtils.getSignatureCertificate(gmsslCertificate),
        });
    }

    public BcDefaultTlsCredentialedECCSM2(TlsCryptoParameters cryptoParams, AbstractTlsCrypto crypto,
                                          Certificate gmsslCertificate, PrivateKey signaturePrivateKey, PrivateKey encryptionPrivateKey,
                                          SignatureAndHashAlgorithm signatureAndHashAlgorithm) {
        super(cryptoParams, makeSigner(crypto, signaturePrivateKey), makeSignerCertificate(gmsslCertificate), signatureAndHashAlgorithm);
        this.gmsslCertificate = gmsslCertificate;
        this.signaturePrivateKey = signaturePrivateKey;
        this.encryptionPrivateKey = encryptionPrivateKey;
        this.crypto = crypto;
    }

    //GMSSL SUPPORT:  use gmssl certificate
    @Override
    public Certificate getCertificate() {
        return gmsslCertificate;
    }

    //GMSSL SUPPORT: : 2018/8/2 decrypt pre master secret
    @Override
    public TlsSecret decrypt(TlsCryptoParameters cryptoParams, byte[] ciphertext) throws IOException {
        if (crypto instanceof BcTlsCrypto) {
            PrivateKeyInfo privateKeyInfo = PrivateKeyInfo.getInstance(encryptionPrivateKey.getEncoded());
            if (privateKeyInfo == null) {
                throw new IllegalArgumentException("'privateKey' type not supported: " + encryptionPrivateKey.getClass().getName());
            }
            AsymmetricKeyParameter parameter = PrivateKeyFactory.createKey(privateKeyInfo);
            return safeDecryptPreMasterSecret((ECKeyParameters) parameter, ciphertext);
        } else if (crypto instanceof BcTlsCryptoSdf){
            return safeDecryptPreMasterSecretSdf((BcTlsCryptoSdf) crypto, ciphertext);
        } else {
            throw new IllegalArgumentException("'crypto' type not supported: " + crypto.getClass().getName());
        }

    }

    protected TlsSecret safeDecryptPreMasterSecret(ECKeyParameters ecServerPrivateKey, byte[] encryptedPreMasterSecret) throws TlsFatalAlert {
        try {
            byte[] M = GMSSLSM2EncryptUtils.decryptASN1ByBC(ecServerPrivateKey, encryptedPreMasterSecret);
            GMSSLByteArrayUtils.printHexBinary(logger, "safeDecryptPreMasterSecret", M);
            return crypto.createSecret(M);
        } catch (Exception e) {
            //tls 中这里可以不抛出异常，这里考虑到排查问题，所以抛出异常
            throw new TlsFatalAlert(AlertDescription.decrypt_error, e);
        }
    }

    protected TlsSecret safeDecryptPreMasterSecretSdf(BcTlsCryptoSdf crypto, byte[] encryptedPreMasterSecret) throws TlsFatalAlert {
        try {
            SdfPrivateKey pPrivateKey = (SdfPrivateKey) this.signaturePrivateKey;
            SdfECKeyParameters parameters = new SdfECKeyParameters(pPrivateKey);
            SdfSM2Engine pSM2Engine = new SdfSM2Engine(crypto.getSdfCryptoType());
            pSM2Engine.init(false, parameters);
            byte[] M = pSM2Engine.decryptASN1(encryptedPreMasterSecret);
            GMSSLByteArrayUtils.printHexBinary(logger, "safeDecryptPreMasterSecret", M);
            pSM2Engine.release();
            return crypto.createSecret(M);
        } catch (Exception e) {
            //tls 中这里可以不抛出异常，这里考虑到排查问题，所以抛出异常
            throw new TlsFatalAlert(AlertDescription.decrypt_error, e);
        }
    }

    //GMSSL SUPPORT:  签名证书在前
    public TlsCertificate getSignatureCertificate() {
        return GMSSLUtils.getSignatureCertificate(gmsslCertificate);
    }

    //GMSSL SUPPORT:  加密证书在后
    public TlsCertificate getEncryptionCertificate() {
        return GMSSLUtils.getEncryptionCertificate(gmsslCertificate);
    }

}
