package org.bouncycastle.tls.crypto.impl.bc;

import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.crypto.macs.HMac;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.tls.TlsUtils;
import org.bouncycastle.tls.crypto.TlsSecret;
import org.bouncycastle.tls.crypto.impl.AbstractTlsCrypto;
import org.bouncycastle.tls.crypto.impl.AbstractTlsSecret;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.Strings;

/**
 * BC light-weight support class for handling TLS secrets and deriving key material and other secrets from them.
 */
public class BcTlsSecretSdf extends AbstractTlsSecret {
    protected final BcTlsCryptoSdf crypto;

    public BcTlsSecretSdf(BcTlsCryptoSdf crypto, byte[] data) {
        super(data);

        this.crypto = crypto;
    }

    @Override
    public synchronized TlsSecret deriveUsingPRF(int prfAlgorithm, String label, byte[] seed, int length) {
        checkAlive();
        byte[] labelSeed = Arrays.concatenate(Strings.toByteArray(label), seed);
        byte[] result = prf_1_2(prfAlgorithm, data, labelSeed, length);

//        PSM3Hmac psm3Hmac = new PSM3Hmac();
//        byte[] result = psm3Hmac.hmacHash(data, labelSeed);
//        byte[] r = new byte[length];
//        System.arraycopy(result, 0, r, 0, length);

        return crypto.adoptLocalSecret(result);
    }

    @Override
    protected AbstractTlsCrypto getCrypto() {
        return crypto;
    }

    protected void hmacHash(Digest digest, byte[] secret, int secretOff, int secretLen, byte[] seed, byte[] output) {
        // TODO: 2019/4/3 use pcie hsm sm3
//        HMac mac = new HMac(digest);
        HMac mac = new HMac(new SM3Digest());
//        HMacSdf mac = new HMacSdf(digest);
        mac.init(new KeyParameter(secret, secretOff, secretLen));

        byte[] a = seed;

        int macSize = mac.getMacSize();

        byte[] b1 = new byte[macSize];
        byte[] b2 = new byte[macSize];

        int pos = 0;
        while (pos < output.length) {
            mac.update(a, 0, a.length);
            mac.doFinal(b1, 0);
            a = b1;
            mac.update(a, 0, a.length);
            mac.update(seed, 0, seed.length);
            mac.doFinal(b2, 0);
            System.arraycopy(b2, 0, output, pos, Math.min(macSize, output.length - pos));
            pos += macSize;
        }
    }

    protected byte[] prf_1_2(int prfAlgorithm, byte[] secret, byte[] labelSeed, int length) {
        Digest digest = crypto.createDigest(TlsUtils.getHashAlgorithmForPRFAlgorithm(prfAlgorithm));
        byte[] result = new byte[length];
        hmacHash(digest, secret, 0, secret.length, labelSeed, result);
        return result;
    }

}
