package org.bouncycastle.tls;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import org.bouncycastle.jsse.provider.ProvTlsServer;
import org.bouncycastle.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.UUID;
import java.util.Vector;

public class TlsServerProtocol extends TlsProtocol {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    //GMSSL SUPPORT: SUPPORT: 2018/7/18 change to TlsServer
    protected ProvTlsServer tlsServer = null;
    TlsServerContextImpl tlsServerContext = null;

    protected TlsKeyExchange keyExchange = null;
    protected TlsCredentials serverCredentials = null;
    protected CertificateRequest certificateRequest = null;

    protected TlsHandshakeHash prepareFinishHash = null;

    /**
     * Constructor for non-blocking mode.<br>
     * <br>
     * When data is received, use {@link #offerInput(byte[])} to provide the received ciphertext,
     * then use {@link #readInput(byte[], int, int)} to read the corresponding cleartext.<br>
     * <br>
     * Similarly, when data needs to be sent, use {@link #writeApplicationData(byte[], int, int)} to
     * provide the cleartext, then use {@link #readOutput(byte[], int, int)} to get the
     * corresponding ciphertext.
     */
    public TlsServerProtocol() {
        super();
    }

    /**
     * Constructor for blocking mode.
     *
     * @param input  The stream of data from the client
     * @param output The stream of data to the client
     */
    public TlsServerProtocol(InputStream input, OutputStream output) {
        super(input, output);
    }

    /**
     * Receives a TLS handshake in the role of server.<br>
     * <br>
     * In blocking mode, this will not return until the handshake is complete.
     * In non-blocking mode, use {@link TlsPeer#notifyHandshakeComplete()} to
     * receive a callback when the handshake is complete.
     *
     * @param tlsServer
     * @throws IOException If in blocking mode and handshake was not successful.
     */
    public void accept(TlsServer tlsServer)
            throws IOException {
        if (tlsServer == null) {
            throw new IllegalArgumentException("'tlsServer' cannot be null");
        }
        if (this.tlsServer != null) {
            throw new IllegalStateException("'accept' can only be called once");
        }

        //todo change default to tls server
        this.tlsServer = (ProvTlsServer) tlsServer;

        this.securityParameters = new SecurityParameters();
        this.securityParameters.entity = ConnectionEnd.server;

        this.tlsServerContext = new TlsServerContextImpl(tlsServer.getCrypto(), securityParameters);

        //生成server端随机数
        this.securityParameters.serverRandom = createRandomBlock(tlsServer.shouldUseGMTUnixTime(), tlsServerContext);
        this.securityParameters.extendedPadding = tlsServer.shouldUseExtendedPadding();

        this.tlsServer.init(tlsServerContext);
        this.recordStream.init(tlsServerContext);

        this.recordStream.setRestrictReadVersion(false);

        if (blocking) {
            blockForHandshake();
        }
    }

    @Override
    protected void cleanupHandshake() {
        super.cleanupHandshake();

        this.keyExchange = null;
        this.serverCredentials = null;
        this.certificateRequest = null;
        this.prepareFinishHash = null;
    }

    @Override
    protected TlsContext getContext() {
        return tlsServerContext;
    }

    @Override
    AbstractTlsContext getContextAdmin() {
        return tlsServerContext;
    }

    @Override
    protected TlsPeer getPeer() {
        return tlsServer;
    }

    @Override
    protected void handleHandshakeMessage(short type, ByteArrayInputStream buf)
            throws IOException {
        switch (type) {
            case HandshakeType.client_hello: {
                switch (this.connection_state) {
                    case CS_START: {
                        //接收 client hello 消息
                        // logger.info("receive ClientHello message!");
                        receiveClientHelloMessage(buf);
                        this.connection_state = CS_CLIENT_HELLO;

                        // NOTE: Currently no server support for session resumption 目前没有服务器支持会话恢复
                        {
                            // logger.info("invalidate session!");
                            invalidateSession();//使session无效

                            //new session TlsSessionImpl
                            String uuid = UUID.randomUUID().toString();    //获取UUID并转化为String对象
                            uuid = uuid.replace("-", "");
                            byte[] sb = uuid.getBytes();
                            // logger.info("new session: {}", uuid);
                            this.tlsSession = TlsUtils.importSession(sb, null);
                            this.sessionParameters = null;
                        }

                        //发送 server hello 消息
                        sendServerHelloMessage();
                        this.connection_state = CS_SERVER_HELLO;
                        // logger.info("send ServerHello message done!");

                        //hello 消息完整性检查
                        recordStream.notifyHelloComplete();
                        // logger.info("notify hello complete!");

                        //server 补充内容
                        Vector serverSupplementalData = tlsServer.getServerSupplementalData();
                        if (serverSupplementalData != null) {
                            sendSupplementalDataMessage(serverSupplementalData);
                            // logger.info("send SupplementalData message!");
                        }
                        this.connection_state = CS_SERVER_SUPPLEMENTAL_DATA;

                        //server key exchange
                        // logger.info("init key exchange!");
                        this.keyExchange = tlsServer.getKeyExchange();
                        this.keyExchange.init(getContext());

                        //验证证书
                        // logger.info("validate credentials!");
                        this.serverCredentials = validateCredentials(tlsServer.getCredentials());

                        //获取证书
                        Certificate serverCertificate = null;

                        ByteArrayOutputStream endPointHash = new ByteArrayOutputStream();
                        if (this.serverCredentials == null) {
                            //跳过 Server Credentials 服务端证书
                            this.keyExchange.skipServerCredentials();
                            // logger.info("skip server credentials!");
                        } else {
                            //进行 Server Credentials 服务端证书
                            // logger.info("process server credentials!");
                            this.keyExchange.processServerCredentials(this.serverCredentials);

                            //DefaultTlsCredentialedSigner.getCertificate
                            serverCertificate = this.serverCredentials.getCertificate();
                            //发送 server certificate 消息
                            sendCertificateMessage(serverCertificate, endPointHash);
                            // logger.info("send Certificate message done!");
                        }
                        securityParameters.tlsServerEndPoint = endPointHash.toByteArray();
                        this.connection_state = CS_SERVER_CERTIFICATE;

                        // TODO[RFC 3546] Check whether empty certificates is possible, allowed, or excludes CertificateStatus
                        if (serverCertificate == null || serverCertificate.isEmpty()) {
                            this.allowCertificateStatus = false;
                        }

                        if (this.allowCertificateStatus) {
                            //这个是 null 一定不会发送
                            CertificateStatus certificateStatus = tlsServer.getCertificateStatus();
                            if (certificateStatus != null) {
                                //发送证书状态消息
                                sendCertificateStatusMessage(certificateStatus);
                                // logger.info("send CertificateStatus message done!");
                            }
                        }

                        this.connection_state = CS_CERTIFICATE_STATUS;

                        byte[] serverKeyExchange = this.keyExchange.generateServerKeyExchange();
                        if (serverKeyExchange != null) {
                            //发送 server key exchange 消息
                            sendServerKeyExchangeMessage(serverKeyExchange);
                            // logger.info("send ServerKeyExchange message done!");
                        }
                        this.connection_state = CS_SERVER_KEY_EXCHANGE;

                        if (this.serverCredentials != null) {
                            this.certificateRequest = tlsServer.getCertificateRequest();
                            if (this.certificateRequest != null) {
                                if (TlsUtils.isTLSv12(getContext()) == (certificateRequest.getSupportedSignatureAlgorithms() == null)) {
                                    throw new TlsFatalAlert(AlertDescription.internal_error);
                                }

                                this.certificateRequest = TlsUtils.validateCertificateRequest(this.certificateRequest, this.keyExchange);

                                //发送 certificate request 证书验证消息
                                sendCertificateRequestMessage(certificateRequest);
                                // logger.info("send CertificateRequest message done!");

                                TlsUtils.trackHashAlgorithms(this.recordStream.getHandshakeHash(), this.certificateRequest.getSupportedSignatureAlgorithms());
                            }
                        }
                        this.connection_state = CS_CERTIFICATE_REQUEST;

                        //发送 server hello done 消息
                        sendServerHelloDoneMessage();
                        this.connection_state = CS_SERVER_HELLO_DONE;
                        // logger.info("send ServerHelloDone message done!");

                        boolean forceBuffering = false;
                        TlsUtils.sealHandshakeHash(getContext(), this.recordStream.getHandshakeHash(), forceBuffering);

                        break;
                    }
                    case CS_END: {
                        refuseRenegotiation();
                        break;
                    }
                    default:
                        throw new TlsFatalAlert(AlertDescription.unexpected_message);
                }
                break;
            }
            case HandshakeType.supplemental_data: {
                // logger.info("receive ClientSupplementalData message!");
                switch (this.connection_state) {
                    case CS_SERVER_HELLO_DONE: {
                        tlsServer.processClientSupplementalData(readSupplementalDataMessage(buf));
                        this.connection_state = CS_CLIENT_SUPPLEMENTAL_DATA;
                        break;
                    }
                    default:
                        throw new TlsFatalAlert(AlertDescription.unexpected_message);
                }
                break;
            }
            case HandshakeType.certificate: {
                // logger.info("receive Certificate message!");
                switch (this.connection_state) {
                    case CS_SERVER_HELLO_DONE: {
                        tlsServer.processClientSupplementalData(null);
                        // NB: Fall through to next case label
                    }
                    case CS_CLIENT_SUPPLEMENTAL_DATA: {
                        if (this.certificateRequest == null) {
                            throw new TlsFatalAlert(AlertDescription.unexpected_message);
                        }
                        receiveCertificateMessage(buf);
                        this.connection_state = CS_CLIENT_CERTIFICATE;
                        break;
                    }
                    default:
                        throw new TlsFatalAlert(AlertDescription.unexpected_message);
                }
                break;
            }
            case HandshakeType.client_key_exchange: {
                // logger.info("receive ClientKeyExchange message!");
                switch (this.connection_state) {
                    case CS_SERVER_HELLO_DONE: {
                        tlsServer.processClientSupplementalData(null);
                        // NB: Fall through to next case label
                    }
                    case CS_CLIENT_SUPPLEMENTAL_DATA: {
                        if (this.certificateRequest == null) {
                            this.keyExchange.skipClientCredentials();
                        } else {
                            if (TlsUtils.isTLSv12(getContext())) {
                                /*
                                 * RFC 5246 If no suitable certificate is available, the client MUST send a
                                 * certificate message containing no certificates.
                                 *
                                 * NOTE: In previous RFCs, this was SHOULD instead of MUST.
                                 */
                                throw new TlsFatalAlert(AlertDescription.unexpected_message);
                            } else {
                                notifyClientCertificate(Certificate.EMPTY_CHAIN);
                            }
                        }
                        // NB: Fall through to next case label
                    }
                    case CS_CLIENT_CERTIFICATE: {
                        receiveClientKeyExchangeMessage(buf);
                        this.connection_state = CS_CLIENT_KEY_EXCHANGE;
                        break;
                    }
                    default:
                        throw new TlsFatalAlert(AlertDescription.unexpected_message);
                }
                break;
            }
            case HandshakeType.certificate_verify: {
                // logger.info("receive CertificateVerify message!");
                switch (this.connection_state) {
                    case CS_CLIENT_KEY_EXCHANGE: {
                        /*
                         * RFC 5246 7.4.8 This message is only sent following a client certificate that has
                         * signing capability (i.e., all certificates except those containing fixed
                         * Diffie-Hellman parameters).
                         */
                        if (!expectCertificateVerifyMessage()) {
                            throw new TlsFatalAlert(AlertDescription.unexpected_message);
                        }

                        receiveCertificateVerifyMessage(buf);
                        this.connection_state = CS_CERTIFICATE_VERIFY;

                        break;
                    }
                    default:
                        throw new TlsFatalAlert(AlertDescription.unexpected_message);
                }
                break;
            }
            case HandshakeType.finished: {
                if (logger.isDebugEnabled()){
                    logger.debug("receive Finished message!");
                }
                switch (this.connection_state) {
                    case CS_CLIENT_KEY_EXCHANGE: {
                        if (expectCertificateVerifyMessage()) {
                            throw new TlsFatalAlert(AlertDescription.unexpected_message);
                        }
                        // NB: Fall through to next case label
                    }
                    case CS_CERTIFICATE_VERIFY: {
                        processFinishedMessage(buf);
                        this.connection_state = CS_CLIENT_FINISHED;

                        if (this.expectSessionTicket) {
                            sendNewSessionTicketMessage(tlsServer.getNewSessionTicket());
                        }
                        this.connection_state = CS_SERVER_SESSION_TICKET;

                        sendChangeCipherSpecMessage();
                        sendFinishedMessage();
                        this.connection_state = CS_SERVER_FINISHED;

                        completeHandshake();
                        break;
                    }
                    default:
                        throw new TlsFatalAlert(AlertDescription.unexpected_message);
                }
                break;
            }
            case HandshakeType.hello_request:
            case HandshakeType.hello_verify_request:
            case HandshakeType.server_hello:
            case HandshakeType.server_key_exchange:
            case HandshakeType.certificate_request:
            case HandshakeType.server_hello_done:
            case HandshakeType.session_ticket:
            default:
                logger.error("receive other error message {}!", type);
                throw new TlsFatalAlert(AlertDescription.unexpected_message);
        }
    }

    @Override
    protected void handleAlertWarningMessage(short alertDescription)
            throws IOException {
        super.handleAlertWarningMessage(alertDescription);

        switch (alertDescription) {
            case AlertDescription.no_certificate: {
                throw new TlsFatalAlert(AlertDescription.unexpected_message);
            }
        }
    }

    protected void notifyClientCertificate(Certificate clientCertificate)
            throws IOException {
        if (certificateRequest == null) {
            throw new IllegalStateException();
        }

        if (peerCertificate != null) {
            throw new TlsFatalAlert(AlertDescription.unexpected_message);
        }

        this.peerCertificate = clientCertificate;

        if (clientCertificate.isEmpty()) {
            this.keyExchange.skipClientCredentials();
        } else {

            /*
             * TODO RFC 5246 7.4.6. If the certificate_authorities list in the certificate request
             * message was non-empty, one of the certificates in the certificate chain SHOULD be
             * issued by one of the listed CAs.
             */

            this.keyExchange.processClientCertificate(clientCertificate);
        }

        /*
         * RFC 5246 7.4.6. If the client does not send any certificates, the server MAY at its
         * discretion either continue the handshake without client authentication, or respond with a
         * fatal handshake_failure alert. Also, if some aspect of the certificate chain was
         * unacceptable (e.g., it was not signed by a known, trusted CA), the server MAY at its
         * discretion either continue the handshake (considering the client unauthenticated) or send
         * a fatal alert.
         */
        this.tlsServer.notifyClientCertificate(clientCertificate);
    }

    protected void receiveCertificateMessage(ByteArrayInputStream buf)
            throws IOException {
        Certificate clientCertificate = Certificate.parse(getContext(), buf, null);

        assertEmpty(buf);

        notifyClientCertificate(clientCertificate);
    }

    protected void receiveCertificateVerifyMessage(ByteArrayInputStream buf)
            throws IOException {
        if (certificateRequest == null) {
            throw new IllegalStateException();
        }

        TlsContext context = getContext();

        if (TlsUtils.isGMSSLv11(context.getServerVersion())) {
            GMSSLUtils.verifyCertificateVerify(context, certificateRequest, peerCertificate, buf,
                    prepareFinishHash);
        } else {
            DigitallySigned clientCertificateVerify = DigitallySigned.parse(context, buf);

            assertEmpty(buf);

            TlsUtils.verifyCertificateVerify(context, certificateRequest, peerCertificate, clientCertificateVerify,
                    prepareFinishHash);
        }
    }

    protected void receiveClientHelloMessage(ByteArrayInputStream buf)
            throws IOException {
        // 读取 client ssl version
        ProtocolVersion client_version = TlsUtils.readVersion(buf);
        // 默认为 国密SSL
        // 360浏览器需要在尝试 TLS SSL 时返回国密SSL 0101 的 alert 即150101 才会发送 国密SSL 请求
        // 以此为标记 服务器只支持国密SSL链接
        recordStream.setWriteVersion(ProtocolVersion.GMSSLv11);
        if (logger.isInfoEnabled()){
            logger.info("client hello version: " + client_version);
        }

        //不支持 DTLS 协议
        if (client_version.isDTLS()) {
            throw new TlsFatalAlert(AlertDescription.illegal_parameter);
        }

        //读取 client random 随机数
        byte[] client_random = TlsUtils.readFully(32, buf);

        /*
         * TODO RFC 5077 3.4. If a ticket is presented by the client, the server MUST NOT attempt to
         * use the Session ID in the ClientHello for stateful session resumption.
         * 如果客户端提供了 ticket ，服务器就不能尝试在ClientHello中使用会话ID进行有状态会话恢复。
         */
        //读取 session id
        byte[] sessionID = TlsUtils.readOpaque8(buf);
        if (sessionID.length > 32) {
            throw new TlsFatalAlert(AlertDescription.illegal_parameter);
        }

        /*
         * TODO RFC 5246 7.4.1.2. If the session_id field is not empty (implying a session
         * resumption request), this vector MUST include at least the cipher_suite from that
         * session.
         * 如果 session id 字段不是空的（意味着会话恢复请求），这个会话必须至少包含这个密码套件。
         */
        //读取 加密套件 长度
        int cipher_suites_length = TlsUtils.readUint16(buf);
        if (cipher_suites_length < 2 || (cipher_suites_length & 1) != 0) {
            throw new TlsFatalAlert(AlertDescription.decode_error);
        }
        //读取 提供的 加密套件
        this.offeredCipherSuites = TlsUtils.readUint16Array(cipher_suites_length / 2, buf);

        /*
         * TODO RFC 5246 7.4.1.2. If the session_id field is not empty (implying a session
         * resumption request), it MUST include the compression_method from that session.
         * 如果sessionid字段不是空的（意味着会话恢复请求），它必须包含来自该会话的压缩方法。
         */
        //读取 压缩方法 长度
        int compression_methods_length = TlsUtils.readUint8(buf);
        if (compression_methods_length < 1) {
            throw new TlsFatalAlert(AlertDescription.illegal_parameter);
        }
        //读取 压缩方法
        this.offeredCompressionMethods = TlsUtils.readUint8Array(compression_methods_length, buf);

        /*
         * TODO RFC 3546 2.3 If [...] the older session is resumed, then the server MUST ignore
         * extensions appearing in the client hello, and send a server hello containing no
         * extensions.
         * 旧的会话被恢复，然后服务器必须忽略在client hello中出现的扩展，并发送一个包含no的server hello扩展。
         */

        //读取 客户端 扩展
        this.clientExtensions = readExtensions(buf);

        /*
         * TODO[session-hash]
         *
         * draft-ietf-tls-session-hash-04 4. Clients and servers SHOULD NOT accept handshakes
         * that do not use the extended master secret [..]. (and see 5.2, 5.3)
         * 客户和服务器不应该接受不使用扩展的主秘钥的握手
         */
        this.securityParameters.extendedMasterSecret = TlsExtensionsUtils.hasExtendedMasterSecretExtension(clientExtensions);

        getContextAdmin().setClientVersion(client_version);

        tlsServer.notifyClientVersion(client_version);
        tlsServer.notifyFallback(Arrays.contains(offeredCipherSuites, CipherSuite.TLS_FALLBACK_SCSV));

        securityParameters.clientRandom = client_random;

        tlsServer.notifyOfferedCipherSuites(offeredCipherSuites);
        tlsServer.notifyOfferedCompressionMethods(offeredCompressionMethods);

        /*
         * RFC 5746 3.6. Server Behavior: Initial Handshake
         */
        {
            /*
             * RFC 5746 3.4. The client MUST include either an empty "renegotiation_info" extension,
             * or the TLS_EMPTY_RENEGOTIATION_INFO_SCSV signaling cipher suite value in the
             * ClientHello. Including both is NOT RECOMMENDED.
             */

            /*
             * When a ClientHello is received, the server MUST check if it includes the
             * TLS_EMPTY_RENEGOTIATION_INFO_SCSV SCSV. If it does, set the secure_renegotiation flag
             * to TRUE.
             */
            //GMSSL SUPPORT: 2018/7/18 gmssl client not support tls empty renegotiation info scsv
            if (Arrays.contains(offeredCipherSuites, CipherSuite.TLS_EMPTY_RENEGOTIATION_INFO_SCSV)) {
                this.secure_renegotiation = true;
            }

            /*
             * The server MUST check if the "renegotiation_info" extension is included in the
             * ClientHello.
             */
            byte[] renegExtData = TlsUtils.getExtensionData(clientExtensions, EXT_RenegotiationInfo);
            //GMSSL SUPPORT: 2018/7/18 gmssl not support client and server extention
            if (renegExtData != null && !client_version.isGMSSL()) {
                /*
                 * If the extension is present, set secure_renegotiation flag to TRUE. The
                 * server MUST then verify that the length of the "renegotiated_connection"
                 * field is zero, and if it is not, MUST abort the handshake.
                 */
                this.secure_renegotiation = true;

                if (!Arrays.constantTimeAreEqual(renegExtData, createRenegotiationInfo(TlsUtils.EMPTY_BYTES))) {
                    throw new TlsFatalAlert(AlertDescription.handshake_failure);
                }
            }
        }

        //通知重新协商密钥
        tlsServer.notifySecureRenegotiation(this.secure_renegotiation);

        if (clientExtensions != null) {
            // NOTE: Validates the padding extension data, if present
            TlsExtensionsUtils.getPaddingExtension(clientExtensions);

            tlsServer.processClientExtensions(clientExtensions);
        }
    }

    protected void receiveClientKeyExchangeMessage(ByteArrayInputStream buf)
            throws IOException {
        keyExchange.processClientKeyExchange(buf);

        assertEmpty(buf);

        this.prepareFinishHash = recordStream.prepareToFinish();
        this.securityParameters.sessionHash = TlsUtils.getCurrentPRFHash(prepareFinishHash);

        //建立主密钥
        establishMasterSecret(getContext(), keyExchange);

        recordStream.setPendingConnectionState(getPeer().getCompression(), getPeer().getCipher());
    }

    protected void sendCertificateRequestMessage(CertificateRequest certificateRequest)
            throws IOException {
        HandshakeMessage message = new HandshakeMessage(HandshakeType.certificate_request);

        certificateRequest.encode(message);

        message.writeToRecordStream();
    }

    protected void sendCertificateStatusMessage(CertificateStatus certificateStatus)
            throws IOException {
        HandshakeMessage message = new HandshakeMessage(HandshakeType.certificate_status);

        certificateStatus.encode(message);

        message.writeToRecordStream();
    }

    protected void sendNewSessionTicketMessage(NewSessionTicket newSessionTicket)
            throws IOException {
        if (newSessionTicket == null) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }

        HandshakeMessage message = new HandshakeMessage(HandshakeType.session_ticket);

        newSessionTicket.encode(message);

        message.writeToRecordStream();
    }

    protected void sendServerHelloMessage()
            throws IOException {
        //HandshakeMessage extends ByteArrayOutputStream
        HandshakeMessage message = new HandshakeMessage(HandshakeType.server_hello);
        {
            //GMSSL SUPPORT 根据 client version 确定 server version
            ProtocolVersion server_version = tlsServer.getServerVersion();
            if (!server_version.isEqualOrEarlierVersionOf(getContext().getClientVersion())) {
                throw new TlsFatalAlert(AlertDescription.internal_error);
            }

            recordStream.setReadVersion(server_version);
            recordStream.setWriteVersion(server_version);
            recordStream.setRestrictReadVersion(true);
            getContextAdmin().setServerVersion(server_version);

            TlsUtils.writeVersion(server_version, message);
        }

        message.write(this.securityParameters.serverRandom);

        /*
         * The server may return an empty session_id to indicate that the session will not be cached
         * and therefore cannot be resumed.
         */
        TlsUtils.writeOpaque8(tlsSession.getSessionID(), message);

        //GMSSL SUPPORT 选取加密套件
        int selectedCipherSuite = tlsServer.getSelectedCipherSuite();
        if (!Arrays.contains(offeredCipherSuites, selectedCipherSuite)
                || selectedCipherSuite == CipherSuite.TLS_NULL_WITH_NULL_NULL
                || CipherSuite.isSCSV(selectedCipherSuite)
                || !TlsUtils.isValidCipherSuiteForVersion(selectedCipherSuite, getContext().getServerVersion())) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }
        securityParameters.cipherSuite = selectedCipherSuite;

        //GMSSL SUPPORT 选取压缩算法
        short selectedCompressionMethod = tlsServer.getSelectedCompressionMethod();
        if (!Arrays.contains(offeredCompressionMethods, selectedCompressionMethod)) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }
        securityParameters.compressionAlgorithm = selectedCompressionMethod;

        //写入选取的加密套件压缩算法
        TlsUtils.writeUint16(selectedCipherSuite, message);
        TlsUtils.writeUint8(selectedCompressionMethod, message);

        //GMSSL SUPPORT 处理扩展
        this.serverExtensions = tlsServer.getServerExtensions();

        /*
         * RFC 5746 3.6. Server Behavior: Initial Handshake
         * 服务器行为:最初的握手
         * 重新协商密钥
         */
        if (this.secure_renegotiation) {
            byte[] renegExtData = TlsUtils.getExtensionData(this.serverExtensions, EXT_RenegotiationInfo);
            boolean noRenegExt = (null == renegExtData);

            if (noRenegExt) {
                /*
                 * Note that sending a "renegotiation_info" extension in response to a ClientHello
                 * containing only the SCSV is an explicit exception to the prohibition in RFC 5246,
                 * Section 7.4.1.4, on the server sending unsolicited extensions and is only allowed
                 * because the client is signaling its willingness to receive the extension via the
                 * TLS_EMPTY_RENEGOTIATION_INFO_SCSV SCSV.
                 * 请注意，发送“重新谈判信息”的扩展是为了响应ClientHello
                 * 只包含SCSV是RFC 5246中禁止的一个明显的例外，
                 * 在服务器上发送未经请求的扩展，并且只允许
                 * 因为客户端表示愿意通过以下方式接收扩展
                 * TLS_EMPTY_RENEGOTIATION_INFO_SCSV SCSV。
                 */

                /*
                 * If the secure_renegotiation flag is set to TRUE, the server MUST include an empty
                 * "renegotiation_info" extension in the ServerHello message.
                 * 如果secure重新谈判标志被设置为TRUE，则服务器必须包含一个空
                 *“重新谈判信息”在ServerHello消息中的扩展。
                 */
                this.serverExtensions = TlsExtensionsUtils.ensureExtensionsInitialised(serverExtensions);
                this.serverExtensions.put(EXT_RenegotiationInfo, createRenegotiationInfo(TlsUtils.EMPTY_BYTES));
            }
        }

        if (securityParameters.isExtendedMasterSecret()) {
            this.serverExtensions = TlsExtensionsUtils.ensureExtensionsInitialised(serverExtensions);
            TlsExtensionsUtils.addExtendedMasterSecretExtension(serverExtensions);
        }

        /*
         * TODO RFC 3546 2.3 If [...] the older session is resumed, then the server MUST ignore
         * extensions appearing in the client hello, and send a server hello containing no
         * extensions.
         * 如果[…]旧的会话被恢复，然后服务器必须忽略在客户端hello中出现的扩展，并发送一个包含no的服务器hello
         */

        if (this.serverExtensions != null) {
            this.securityParameters.encryptThenMAC = TlsExtensionsUtils.hasEncryptThenMACExtension(serverExtensions);

            this.securityParameters.maxFragmentLength = processMaxFragmentLengthExtension(clientExtensions,
                    serverExtensions, AlertDescription.internal_error);

            this.securityParameters.truncatedHMac = TlsExtensionsUtils.hasTruncatedHMacExtension(serverExtensions);

            /*
             * TODO It's surprising that there's no provision to allow a 'fresh' CertificateStatus to be sent in
             * a session resumption handshake.
             */
            this.allowCertificateStatus = !resumedSession
                    && TlsUtils.hasExpectedEmptyExtensionData(serverExtensions, TlsExtensionsUtils.EXT_status_request,
                    AlertDescription.internal_error);

            this.expectSessionTicket = !resumedSession
                    && TlsUtils.hasExpectedEmptyExtensionData(serverExtensions, TlsProtocol.EXT_SessionTicket,
                    AlertDescription.internal_error);

            writeExtensions(message, serverExtensions);
        }

        securityParameters.prfAlgorithm = getPRFAlgorithm(getContext(), securityParameters.getCipherSuite());

        /*
         * RFC 5246 7.4.9. Any cipher suite which does not explicitly specify verify_data_length has
         * a verify_data_length equal to 12. This includes all existing cipher suites.
         * 任何未显式指定verify_data_length的密码套件verify_data_length等于12。这包括所有现有的密码套件。
         */
        securityParameters.verifyDataLength = 12;

        //应用最大片段长度扩展
        applyMaxFragmentLengthExtension();

        message.writeToRecordStream();
    }

    protected void sendServerHelloDoneMessage()
            throws IOException {
        byte[] message = new byte[4];
        TlsUtils.writeUint8(HandshakeType.server_hello_done, message, 0);
        TlsUtils.writeUint24(0, message, 1);

        writeHandshakeMessage(message, 0, message.length);
    }

    protected void sendServerKeyExchangeMessage(byte[] serverKeyExchange)
            throws IOException {
        HandshakeMessage message = new HandshakeMessage(HandshakeType.server_key_exchange, serverKeyExchange.length);

        message.write(serverKeyExchange);

        message.writeToRecordStream();
    }

    protected boolean expectCertificateVerifyMessage() {
        return peerCertificate != null && !peerCertificate.isEmpty() && keyExchange.requiresCertificateVerify();
    }
}
