package org.bouncycastle.tls;

import org.bouncycastle.tls.crypto.TlsCryptoParameters;
import org.bouncycastle.tls.crypto.TlsSigner;
import org.bouncycastle.tls.crypto.TlsStreamSigner;
import org.bouncycastle.tls.crypto.impl.TlsImplUtils;

import java.io.IOException;

/**
 * Container class for generating signatures that carries the signature type, parameters, public key certificate and public key's associated signer object.
 */
public class DefaultTlsCredentialedSigner
        implements TlsCredentialedSigner
{
    protected TlsCryptoParameters cryptoParams;
    protected Certificate certificate;
    protected SignatureAndHashAlgorithm signatureAndHashAlgorithm;

    protected TlsSigner signer;

    public DefaultTlsCredentialedSigner(TlsCryptoParameters cryptoParams, TlsSigner signer, Certificate certificate,
                                        SignatureAndHashAlgorithm signatureAndHashAlgorithm)
    {
        if (certificate == null)
        {
            throw new IllegalArgumentException("'certificate' cannot be null");
        }
        if (certificate.isEmpty())
        {
            throw new IllegalArgumentException("'certificate' cannot be empty");
        }
        if (signer == null)
        {
            throw new IllegalArgumentException("'signer' cannot be null");
        }

        this.signer = signer;

        this.cryptoParams = cryptoParams;
        this.certificate = certificate;
        this.signatureAndHashAlgorithm = signatureAndHashAlgorithm;
    }

    //GMSSL SUPPORT 需要添加双证支持
    @Override
    public Certificate getCertificate()
    {
        return certificate;
    }

    @Override
    public byte[] generateRawSignature(byte[] hash)
            throws IOException
    {
        return signer.generateRawSignature(getEffectiveAlgorithm(), hash);
    }

    @Override
    public SignatureAndHashAlgorithm getSignatureAndHashAlgorithm()
    {
        return signatureAndHashAlgorithm;
    }

    @Override
    public TlsStreamSigner getStreamSigner() throws IOException
    {
        return signer.getStreamSigner(getEffectiveAlgorithm());
    }

    protected SignatureAndHashAlgorithm getEffectiveAlgorithm()
    {
        SignatureAndHashAlgorithm algorithm = null;
        if (TlsImplUtils.isTLSv12(cryptoParams) || TlsImplUtils.isGMSSLv11(cryptoParams))
        {
            algorithm = getSignatureAndHashAlgorithm();
            if (algorithm == null)
            {
                throw new IllegalStateException("'signatureAndHashAlgorithm' cannot be null for (D)TLS 1.2+");
            }
        }
        return algorithm;
    }
}
