package org.bouncycastle.tls.crypto.impl.bc;

import com.xdja.pki.gmssl.crypto.sdf.SdfECKeyParameters;
import org.bouncycastle.asn1.ASN1Encoding;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.x509.Certificate;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.tls.*;
import org.bouncycastle.tls.crypto.TlsCertificate;
import org.bouncycastle.tls.crypto.TlsCryptoException;
import org.bouncycastle.tls.crypto.TlsVerifier;
import org.bouncycastle.util.Arrays;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;

/**
 * Implementation class for a single X.509 certificate based on the BC light-weight API.
 */
public class BcTlsCertificateSdf implements TlsCertificate {
    public static BcTlsCertificateSdf convert(BcTlsCryptoSdf crypto, TlsCertificate certificate) throws IOException {
        if (certificate instanceof BcTlsCertificateSdf) {
            return (BcTlsCertificateSdf) certificate;
        }

        return new BcTlsCertificateSdf(crypto, certificate.getEncoded());
    }

    public static Certificate parseCertificate(byte[] encoding) throws IOException {
        try {
            return Certificate.getInstance(encoding);
        } catch (IllegalArgumentException e) {
            throw new TlsCryptoException("unable to decode certificate: " + e.getMessage(), e);
        }
    }

    protected final BcTlsCryptoSdf crypto;
    protected final Certificate certificate;

    protected ECPublicKeyParameters ecPublicKey = null;

    public BcTlsCertificateSdf(BcTlsCryptoSdf crypto, byte[] encoding) throws IOException {
        this(crypto, parseCertificate(encoding));
    }

    public BcTlsCertificateSdf(BcTlsCryptoSdf crypto, Certificate certificate) {
        this.crypto = crypto;
        this.certificate = certificate;
    }

    @Override
    public TlsVerifier createVerifier(short signatureAlgorithm) throws IOException {
        validateKeyUsage(KeyUsage.digitalSignature);

        switch (signatureAlgorithm) {
            case SignatureAlgorithm.sm2:
                return new BcTlsSM2VerifierSdf(crypto, getPublicKey());

            default:
                throw new TlsFatalAlert(AlertDescription.certificate_unknown);
        }
    }

    @Override
    public short getClientCertificateType() throws IOException {
        AsymmetricKeyParameter publicKey = getPublicKey();
        if (publicKey.isPrivate()) {
            throw new TlsFatalAlert(AlertDescription.internal_error);
        }

        try {
            /*
             * ECDSA-capable public key; the certificate MUST allow the key to be used for signing
             * with the hash algorithm that will be employed in the certificate verify message; the
             * public key MUST use a curve and point format supported by the server.
             */
            if (publicKey instanceof SdfECKeyParameters) {
                validateKeyUsage(KeyUsage.digitalSignature);
                // TODO Check the curve and point format
                return ClientCertificateType.ecdsa_sign;
            }

            // TODO Add support for ClientCertificateType.*_fixed_*
        } catch (IOException e) {
            throw e;
        } catch (Exception e) {
            throw new TlsFatalAlert(AlertDescription.unsupported_certificate, e);
        }

        throw new TlsFatalAlert(AlertDescription.unsupported_certificate);
    }

    @Override
    public byte[] getEncoded() throws IOException {
        return certificate.getEncoded(ASN1Encoding.DER);
    }

    @Override
    public byte[] getExtension(ASN1ObjectIdentifier extensionOID) throws IOException {
        Extensions extensions = certificate.getTBSCertificate().getExtensions();
        if (extensions != null) {
            Extension extension = extensions.getExtension(extensionOID);
            if (extension != null) {
                return Arrays.clone(extension.getExtnValue().getOctets());
            }
        }
        return null;
    }

    @Override
    public BigInteger getSerialNumber() {
        return certificate.getSerialNumber().getValue();
    }

    @Override
    public String getSigAlgOID() {
        return certificate.getSignatureAlgorithm().getAlgorithm().getId();
    }

    @Override
    public TlsCertificate useInRole(int connectionEnd, int keyExchangeAlgorithm) throws IOException {
        throw new TlsFatalAlert(AlertDescription.certificate_unknown);
    }

    protected AsymmetricKeyParameter getPublicKey() throws IOException {
        SubjectPublicKeyInfo keyInfo = certificate.getSubjectPublicKeyInfo();
        try {
            PublicKey publicKey = KeyFactory.getInstance("EC", "BC").generatePublic(new X509EncodedKeySpec(keyInfo.getEncoded()));
            return new SdfECKeyParameters((ECPublicKey) publicKey);
        } catch (RuntimeException | NoSuchAlgorithmException | InvalidKeySpecException | NoSuchProviderException e) {
            throw new TlsFatalAlert(AlertDescription.unsupported_certificate, e);
        }
    }

    protected void validateKeyUsage(int keyUsageBits) throws IOException {
        Extensions exts = certificate.getTBSCertificate().getExtensions();
        if (exts != null) {
            KeyUsage ku = KeyUsage.fromExtensions(exts);
            if (ku != null) {
                int bits = ku.getBytes()[0] & 0xff;
                if ((bits & keyUsageBits) != keyUsageBits) {
                    throw new TlsFatalAlert(AlertDescription.certificate_unknown);
                }
            }
        }
    }

    //GMSSL SUPPORT add x509
    public X509Certificate getX509Certificate() throws IOException {
        try {
            /*
             * NOTE: We want to restrict 'encoding' to a binary BER encoding, but
             * CertificateFactory.generateCertificate claims to require DER encoding, and also
             * supports Base64 encodings (in PEM format), which we don't support.
             *
             * Re-encoding validates as BER and produces DER.
             */
            byte[] derEncoding = this.getEncoded();

            ByteArrayInputStream input = new ByteArrayInputStream(derEncoding);
            X509Certificate certificate = (X509Certificate) CertificateFactory.getInstance("X.509", "BC").generateCertificate(input);
            if (input.available() != 0) {
                throw new IOException("Extra data detected in stream");
            }
            return certificate;
        } catch (GeneralSecurityException e) {
            throw new TlsCryptoException("unable to decode certificate", e);
        }
    }
}
