package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.asn1.x509.SubjectInformationAccess;
import org.bouncycastle.asn1.*;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;
import java.util.jar.Attributes;

public class GMSSLExtensionUtils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLExtensionUtils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *               证书 CRL 都需要的 Extension 扩展项生成                             *
     ************************************************************************************/


    /**
     * 获取 授权信息访问 扩展
     *
     * @param ldapUrl 签发者证书 ldap 地址
     * @return 授权信息访问扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genAuthorityInfoAccessExtension(String ldapUrl) throws IOException {
        ASN1OctetString instance = new DEROctetString(ldapUrl.getBytes());
        AuthorityInformationAccess issueInfo = new AuthorityInformationAccess(new AccessDescription(
                AccessDescription.id_ad_caIssuers,
                new GeneralName(GeneralName.uniformResourceIdentifier, instance)));
        return new Extension(Extension.authorityInfoAccess, true, issueInfo.getEncoded());
    }

    /************************************************************************************
     *                          证书 Extension 扩展项生成                               *
     ************************************************************************************/


    /**
     * 获取 主体信息访问 扩展
     *
     * @param ldapUrl 主体证书 ldap 地址
     * @return 主体信息访问扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genSubjectInfoAccessExtension(String ldapUrl) throws IOException {
        ASN1OctetString instance = new DEROctetString(ldapUrl.getBytes());
        SubjectInformationAccess subjectInfo = new SubjectInformationAccess(
                new AccessDescription(
                        SubjectInformationAccess.id_ad_caRepository,
                        new GeneralName(GeneralName.uniformResourceIdentifier, instance)
                )
        );
        return new Extension(Extension.subjectInfoAccess, true, subjectInfo.getEncoded());
    }

    /**
     * 获取 一般 自签名根证书 密钥用途 扩展
     *
     * @return 一般自签名根证书 密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genRootCertKeyUsageExtension() throws IOException {
        KeyUsage keyUsage = new KeyUsage(KeyUsage.cRLSign | KeyUsage.keyCertSign);
        return new Extension(Extension.keyUsage, true, keyUsage.getEncoded());
    }

    /**
     * 获取 一般签名证书 密钥用途 扩展
     *
     * @return 一般签名证书 密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genSignatureCertKeyUsageExtension() throws IOException {
        KeyUsage keyUsage = new KeyUsage(KeyUsage.digitalSignature | KeyUsage.nonRepudiation);
        return new Extension(Extension.keyUsage, true, keyUsage.getEncoded());
    }

    /**
     * 获取 OCSP 签名证书 扩展密钥用途 扩展
     *
     * @return OCSP 签名证书 扩展密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genOCSPCertExtendedKeyUsageExtension() throws IOException {
        return new Extension(Extension.extendedKeyUsage, true, new ExtendedKeyUsage(KeyPurposeId.id_kp_OCSPSigning).getEncoded());
    }

    /**
     * 获取 一般加密证书 密钥用途 扩展
     *
     * @return 一般加密证书 密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genEncryptCertKeyUsageExtension() throws IOException {
        KeyUsage keyUsage = new KeyUsage(KeyUsage.keyEncipherment | KeyUsage.dataEncipherment | KeyUsage.keyAgreement);
        return new Extension(Extension.keyUsage, true, keyUsage.getEncoded());
    }

    /**
     * 获取 签发者 公钥信息 扩展
     *
     * @param rootCert 签发者证书
     * @return 签发者 公钥信息 扩展
     * @throws IOException                  获取二进制编码异常
     * @throws CertificateEncodingException 生成 签发者公钥信息失败
     * @throws NoSuchAlgorithmException     org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils 扩展工具初始化失败
     */
    public static Extension genAuthorityKeyIdentifierExtension(X509Certificate rootCert) throws NoSuchAlgorithmException, CertificateEncodingException, IOException {
        JcaX509ExtensionUtils extUtils = new JcaX509ExtensionUtils();
        AuthorityKeyIdentifier authorityKeyIdentifier = extUtils.createAuthorityKeyIdentifier(rootCert);
        return new Extension(Extension.authorityKeyIdentifier, false, authorityKeyIdentifier.getEncoded());
    }

    /**
     * 获取 主体 公钥信息 扩展
     *
     * @param publicKey 主体公钥信息
     * @return 主体公钥信息扩展
     * @throws IOException              获取二进制编码异常
     * @throws NoSuchAlgorithmException org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils 扩展工具初始化失败
     */
    public static Extension genSubjectKeyIdentifierExtension(PublicKey publicKey) throws IOException, NoSuchAlgorithmException {
        JcaX509ExtensionUtils extUtils = new JcaX509ExtensionUtils();
        SubjectKeyIdentifier subjectKeyIdentifier = extUtils.createSubjectKeyIdentifier(publicKey);
        return new Extension(Extension.subjectKeyIdentifier, false, subjectKeyIdentifier.getEncoded());
    }

    /**
     * 获取 CA证书 标识扩展项
     *
     * @param pathLenConstraint 签发路径长度
     * @return CA证书 标识扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genBasicConstraintsExtension(int pathLenConstraint) throws IOException {
        return new Extension(Extension.basicConstraints, true, new BasicConstraints(pathLenConstraint).getEncoded());
    }

    /**
     * 获取 全量 CRL 发布点扩展项
     *
     * @param crlLdapUrl 全量 CRL ldap 发布点
     * @return CRL 发布点扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genCRLDistributionPointsExtension(String crlLdapUrl) throws IOException {
        DistributionPointName dpn = genDistributionPointName(crlLdapUrl);
        CRLDistPoint crlDistPoint = new CRLDistPoint(new DistributionPoint[]{new DistributionPoint(dpn, null, null)});
        return new Extension(Extension.cRLDistributionPoints, true, crlDistPoint.getEncoded());
    }

    /**
     * 获取 最新（增量） CRL 发布点扩展项
     *
     * @param drlLdapUrl 最新（增量） CRL ldap 发布点
     * @return CRL 发布点扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genFreshestCRL(String drlLdapUrl) throws IOException {
        DistributionPointName dpn = genDistributionPointName(drlLdapUrl);
        CRLDistPoint crlDistPoint = new CRLDistPoint(new DistributionPoint[]{new DistributionPoint(dpn, null, null)});
        return new Extension(Extension.freshestCRL, true, crlDistPoint.getEncoded());
    }

    /**
     * 从证书中读取扩展项
     *
     * @param cert X509证书
     * @return 证书扩展项列表
     */
    public static List<Extension> getCertificateExtensions(X509Certificate cert) throws CertificateEncodingException, IOException {
        List<Extension> extensions = new ArrayList<>();
        X509CertificateHolder holder = new X509CertificateHolder(cert.getEncoded());
        List list = holder.getExtensionOIDs();
        for (Object e : list) {
            extensions.add(holder.getExtension((ASN1ObjectIdentifier) e));
        }
        return extensions;
    }


    /**
     * 根据 RFC 3039 定义的几个属性
     * 1. 选取 title 填入 CommonName 内容
     * 2. 选取 countryOfCitizenship 填入国别码 CN
     *
     * @param title 标识组织位置或个人身份的信息
     * @param critical 是否是必须的
     * @return Extension
     * @throws IOException subjectDirectoryAttributes.getEncoded() 抛出
     */
    public static Extension genSubjectDirectoryAttributesExtension(String title, boolean critical) throws IOException {
        Vector<Attribute> attributes = new Vector<>();
        if (title != null) {
            attributes.add(makeAttribute(BCStyle.T, title));
        }
        attributes.add(makeAttribute(BCStyle.COUNTRY_OF_CITIZENSHIP, "CN"));
        SubjectDirectoryAttributes subjectDirectoryAttributes = new SubjectDirectoryAttributes(attributes);
        //getEncoded will throw IOException
        return new Extension(Extension.subjectDirectoryAttributes, critical, subjectDirectoryAttributes.getEncoded());
    }

    public static Attribute makeAttribute(ASN1ObjectIdentifier oid, String value) {
        DERSet valueSet = new DERSet(new DEROctetString(value.getBytes()));
        return new Attribute(oid, valueSet);
    }

    public static void parseAttribute(Attribute attribute, ASN1ObjectIdentifier oid, String valueName, StringBuilder sb) {
        if (oid.equals(attribute.getAttrType())) {
            ASN1Set set = attribute.getAttrValues();
            ASN1OctetString valueInstance = DEROctetString.getInstance(set.getObjectAt(0));
            String value = new String(valueInstance.getOctets());
            sb.append(valueName).append("=").append(value).append(",");
        }
    }

    /**
     * 获取证书扩展项 SubjectDirectoryAttributes 并格式化
     * @param octets SubjectDirectoryAttributes 的二进制
     * @return 格式化的 string  例如 [Title=user0,CountryOfCitizenship=CN] [Title=中文,CountryOfCitizenship=CN]
     */
    public static String parseSubjectDirectoryAttributesExtensions(byte[] octets) {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        SubjectDirectoryAttributes subjectDirectoryAttributes = SubjectDirectoryAttributes.getInstance(octets);
        Vector attributes = subjectDirectoryAttributes.getAttributes();
        for (Object attr : attributes) {
            Attribute attribute = (Attribute) attr;
            parseAttribute(attribute, BCStyle.T, "Title", sb);
            parseAttribute(attribute, BCStyle.COUNTRY_OF_CITIZENSHIP, "CountryOfCitizenship", sb);
        }
        sb.deleteCharAt(sb.length()-1);
        sb.append("]");
        return sb.toString();
    }

    /************************************************************************************
     *                              CRL Extension 扩展项生成                            *
     ************************************************************************************/
    /**
     * 如果 CRL 为 ARL ，则需要添加此扩展项标识 该 CRL 为 ARL
     *
     * @param arlLdapUrl ARL 对应的 LDAP 存储地址 使用在分块CRL中，必须与证书中发布点扩展值一致
     * @return 标识 ARL 扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genARLExtension(String arlLdapUrl) throws IOException {
        DistributionPointName dpn = genDistributionPointName(arlLdapUrl);
        ReasonFlags reasonFlags = new ReasonFlags(ReasonFlags.cACompromise);
        IssuingDistributionPoint point = new IssuingDistributionPoint(dpn, false, true, reasonFlags, false, false);
        return new Extension(Extension.issuingDistributionPoint, true, point.getEncoded());
    }

    /**
     * 如果 CRL 为 DRL ，则需要添加此扩展项标识 该 CRL 为 DRL
     *
     * @param baseNumber DRL 对应的全量序列号
     * @return 标识 DRL 扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genDRLExtension(int baseNumber) throws IOException {
        CRLNumber basecrlnum = new CRLNumber(BigInteger.valueOf(baseNumber));
        return new Extension(Extension.deltaCRLIndicator, true, basecrlnum.getEncoded());
    }


    public static DistributionPointName genDistributionPointName(String ldapUrl) {
        DEROctetString instance = new DEROctetString(ldapUrl.getBytes());
        GeneralName gn = new GeneralName(GeneralName.uniformResourceIdentifier, instance);
        GeneralNames gns = new GeneralNames(gn);
        return new DistributionPointName(0, gns);
    }
}
