package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfSM2Key;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.*;

public class GMSSLSM2KeyUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                      密钥对相关                                  *
     ************************************************************************************/
    /**
     * 生成 SM2 密钥对
     *
     * @return SM2 密钥对
     * @throws NoSuchProviderException            找不到相应的 Provider
     * @throws NoSuchAlgorithmException           找不到相应的算法
     * @throws InvalidAlgorithmParameterException KeyPairGenerator 初始化异常
     */
    public static KeyPair generateSM2KeyPairByBC() throws NoSuchProviderException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        return GMSSLX509Utils.generateSM2KeyPair();
    }

    /**
     * 生成 SDF 相关的 SM2 私钥
     *
     * @param index    密钥访问索引
     * @param password 密钥访问控制密码
     * @return SDF 相关的 SM2私钥
     */
    public static SdfPrivateKey genSdfPrivateKey(int index, String password) {
        return new SdfPrivateKey(index, password.getBytes());
    }

    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByYunhsm(int index) throws SdfSDKException {
        SdfSM2Key sdfSM2Key = new SdfSM2Key(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfSM2Key.exportSignPublicKey(index);
        sdfSM2Key.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByPcie(int index) throws SdfSDKException {
        SdfSM2Key sdfSM2Key = new SdfSM2Key(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfSM2Key.exportSignPublicKey(index);
        sdfSM2Key.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByYunhsm(int index) throws SdfSDKException {
        SdfSM2Key sdfSM2Key = new SdfSM2Key(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfSM2Key.exportEncryptPublicKey(index);
        sdfSM2Key.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByPcie(int index) throws SdfSDKException {
        SdfSM2Key sdfSM2Key = new SdfSM2Key(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfSM2Key.exportEncryptPublicKey(index);
        sdfSM2Key.release();
        return publicKey;
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return getPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);

    }

    /**
     * 外部校验密钥索引和私钥访问控制码  PCIE卡
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromPCIE(int index, String password) {
        return getPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromSdf(SdfCryptoType sdfCryptoType, int index, String password) {
        try {
            SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
            sdfSDK.init();
            sdfSDK.getPrivateKeyAccessRight(index, password.getBytes());
            sdfSDK.release();
            return true;
        } catch (SdfSDKException e) {
            e.printStackTrace();
            return false;
        }
    }

}
