package com.xdja.pki.gmssl.operator.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentSigner;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

import java.security.PrivateKey;
import java.security.Security;

public class GMSSLSdfContentSignerUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                             ContentSigner 生成                                   *
     ************************************************************************************/

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByBC(GMSSLSdfContentSignatureAlgorithm signatureAlgorithm, PrivateKey privateKey) throws OperatorCreationException {
        //bc
//        AlgorithmIdentifier sigAlgId = new DefaultSignatureAlgorithmIdentifierFinder().find("SM3withSM2");
//        AlgorithmIdentifier digAlgId = new DefaultDigestAlgorithmIdentifierFinder().find(sigAlgId);
//        BcContentSignerBuilder contentSignerBuilder = new BcECContentSignerBuilder(sigAlgId, digAlgId);
//        AsymmetricKeyParameter privateKeyParameter = GMSSLX509Utils.convertECPublicKeyParameters(issuerKey);
//        ContentSigner bcCcontentSigner = contentSignerBuilder.build(privateKeyParameter);
        JcaContentSignerBuilder contentSignerBuilder = new JcaContentSignerBuilder(signatureAlgorithm.getId());
        return contentSignerBuilder.build(privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByYunhsm(PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.YUNHSM, privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByPcie(PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.PCIE, privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerBySDF(SdfCryptoType sdfCryptoType, PrivateKey privateKey) throws OperatorCreationException {
        if (!(privateKey instanceof SdfPrivateKey)) {
            throw new OperatorCreationException(" un support private key type " + privateKey.getClass());
        } else {
            return new SdfContentSigner(sdfCryptoType, (SdfPrivateKey) privateKey);
        }
    }
}
