package com.xdja.pki.gmssl.operator.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentVerifier;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;

import java.security.PublicKey;
import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class GMSSLSdfContentVerifierUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                             ContentVerifier 生成                                   *
     ************************************************************************************/

    /**
     * BC 生成 ContentVerifier 内容验签实体类
     *
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param publicKey          验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByBC(GMSSLSdfContentSignatureAlgorithm signatureAlgorithm, PublicKey publicKey) throws OperatorCreationException {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        return verifierProvider.get(signatureAlgorithm.convertAlgorithmIdentifier());
    }

    /**
     * BC 生成 ContentVerifier 内容验签实体类
     *
     * @param algorithmIdentifier 签名算法 org.bouncycastle.asn1.x509.AlgorithmIdentifier 签名算法
     * @param publicKey           验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByBC(AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey) throws OperatorCreationException {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        return verifierProvider.get(algorithmIdentifier);
    }

    /**
     * 密码机 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey           验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByYunhsm(PublicKey publicKey) throws OperatorCreationException {
        return generateContentVerifierBySdf(SdfCryptoType.YUNHSM, publicKey);
    }

    /**
     * PCIE 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey           验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByPcie(PublicKey publicKey) throws OperatorCreationException {
        return generateContentVerifierBySdf(SdfCryptoType.PCIE, publicKey);
    }

    /**
     * SDF 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey           验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey) throws OperatorCreationException {
        return new SdfContentVerifier(sdfCryptoType, (ECPublicKey) publicKey);
    }

}
