package com.xdja.pki.gmssl.sdf.yunhsm.utils;


import com.alibaba.fastjson.JSONObject;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDK;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmExceptionEnum;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmInfoEntry;
import org.apache.commons.io.FileUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.security.KeyStore;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.List;

public class GMSSLYunHsmUtils {
    private static Logger logger = LoggerFactory.getLogger(GMSSLYunHsmUtils.class);

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 初始化密码机 无需输入密码机路径
     *
     * @throws IOException 读写文件时产生的异常
     */
    public static void resetYunHsm() throws IOException {
        String yunHsmPath = getYunHsmPath();
        resetYunHsm(yunHsmPath);
    }

    /**
     * 初始化密码机
     *
     * @param yunhsmPath 密码机安装路径
     * @throws IOException 读写文件时产生的异常
     */
    public static void resetYunHsm(String yunhsmPath) throws IOException {
        String bakPath = yunhsmPath + File.separator + "confBak" + File.separator;
        String sourcePath = yunhsmPath + File.separator + "conf" + File.separator;
        File bakfile = new File(bakPath);
        if (!bakfile.exists()) {
            logger.info("密码机未进行任何配置，无需初始化");
            return;
        }
        GMSSLX509Utils.deleteDirectory(sourcePath);
        File file = new File(sourcePath);
        file.mkdir();
        GMSSLX509Utils.copyDir(bakPath, sourcePath);
        GMSSLX509Utils.deleteDirectory(bakPath);
        logger.info("密码机初始化成功");
    }

    /**
     * 获取配置信息 不需要传路径
     *
     * @throws IOException 读写配置文件产生的异常
     */
    public static YunHsmInfoEntry getYunHsmInfo() throws IOException {
        String yunHsmPath = getYunHsmPath();
        return getYunHsmInfo(yunHsmPath);
    }

    /**
     * 根据密码机位置获取配置信息
     *
     * @throws IOException 读写配置文件产生的异常
     */
    public static YunHsmInfoEntry getYunHsmInfo(String yunhsmPath) throws IOException {
        String confPath = yunhsmPath + File.separator + "conf" + File.separator + "yunhsmsdk.conf";
        YunHsmInfoEntry yunHsmInfoEntry = new YunHsmInfoEntry();
        File file = new File(confPath);
        String jsonString = FileUtils.readFileToString(file, "UTF-8");
        //总目录
        JSONObject dataJson = JSONObject.parseObject(jsonString);
        //Certificate目录下
        JSONObject certificate = dataJson.getJSONObject("Certificate");
        //SignatureCertificate
        JSONObject softCert = certificate.getJSONObject("SoftCert");
        JSONObject signatureCertificate = softCert.getJSONObject("SignatureCertificate");

        yunHsmInfoEntry.setSignCertPassword(signatureCertificate.getString("pin"));
        String signCertName = signatureCertificate.getString("file");
        yunHsmInfoEntry.setSignCertName(signCertName.substring(signCertName.indexOf("sign")));

        //EncryptCertificate
        JSONObject encryptCertificate = softCert.getJSONObject("EncryptCertificate");
        yunHsmInfoEntry.setEncCertPassword(encryptCertificate.getString("pin"));
        String encCertName = encryptCertificate.getString("file");
        yunHsmInfoEntry.setEncCertName(encCertName.substring(encCertName.indexOf("enc")));

        JSONObject hsm = dataJson.getJSONObject("hsm");
        yunHsmInfoEntry.setServerIp(hsm.getString("ip"));
        yunHsmInfoEntry.setServerPort(Integer.valueOf(hsm.getString("port")));

        JSONObject ssl = dataJson.getJSONObject("ssl");
        String caCertName = ssl.getString("CertificatePath");
        //setCaCertName
        yunHsmInfoEntry.setCaCertName(caCertName.substring(caCertName.indexOf("trust")));

        yunHsmInfoEntry.setYunHsmType("信大捷安服务器密码机");
        return yunHsmInfoEntry;
    }

    /**
     * 配置密码机 无需输入密码机地址
     *
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxPath     签名证书地址    （pfx格式证书）
     * @param encPfxPath      加密证书地址    （pfx格式证书）
     * @param trustP7bPath    密码机CA证书链  （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(String ip, int port, String signPfxPassword, String encPfxPassword,
                                                                     String signPfxPath, String encPfxPath, String trustP7bPath) throws Exception {
        String yunHsmPath = getYunHsmPath();
        return initYunHsmConfigAndTestConnect(yunHsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxPath, encPfxPath, trustP7bPath);

    }

    /**
     * 配置密码机 无需输入密码机地址
     *
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxStream   签名证书文件流        （pfx格式证书）
     * @param encPfxStream    加密证书文件流        （pfx格式证书）
     * @param trustP7bStream  密码机CA证书链文件流   （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(String ip, int port, String signPfxPassword, String encPfxPassword,
                                                                     FileInputStream signPfxStream, FileInputStream encPfxStream, FileInputStream trustP7bStream) throws Exception {
        String yunHsmPath = getYunHsmPath();
        return initYunHsmConfigAndTestConnect1(yunHsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
    }

    /**
     * 配置密码机
     *
     * @param yunhsmPath      密码机安装路径
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxPath     签名证书地址    （pfx格式证书）
     * @param encPfxPath      加密证书地址    （pfx格式证书）
     * @param trustP7bPath    密码机CA证书链  （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword,
                                                                     String signPfxPath, String encPfxPath, String trustP7bPath) throws Exception {
        //第一次时候进行备份
        backUpConf(yunhsmPath);
        //存储证书到keystore
        YunHsmExceptionEnum checkCert = storeCert(signPfxPassword, encPfxPassword, signPfxPath, encPfxPath, trustP7bPath, yunhsmPath);
        if (checkCert.id != YunHsmExceptionEnum.NORMAL.id) {
            return checkCert;
        }
        //改写配置文件
        writeConf(yunhsmPath, ip, port, signPfxPassword, encPfxPassword);
        //测试ip和端口
        YunHsmExceptionEnum checkConnect = testConnect(ip, port);
        if (checkConnect.id != YunHsmExceptionEnum.NORMAL.id) {
            return checkConnect;
        }
        return YunHsmExceptionEnum.NORMAL;
    }

    /**
     * 配置密码机
     *
     * @param yunhsmPath      密码机安装路径
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxStream   签名证书文件流    （pfx格式证书）
     * @param encPfxStream    加密证书文件流    （pfx格式证书）
     * @param trustP7bStream  密码机CA证书链文件流  （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect1(String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword,
                                                                     FileInputStream signPfxStream, FileInputStream encPfxStream, FileInputStream trustP7bStream) throws Exception {
        //第一次时候进行备份
        backUpConf(yunhsmPath);
        //存储证书到keystore
        YunHsmExceptionEnum checkCert = storeCert(signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream, yunhsmPath);
        if (checkCert.id != YunHsmExceptionEnum.NORMAL.id) {
            return checkCert;
        }
        //改写配置文件
        writeConf(yunhsmPath, ip, port, signPfxPassword, encPfxPassword);
        //测试ip和端口
        YunHsmExceptionEnum checkConnect = testConnect(ip, port);
        if (checkConnect.id != YunHsmExceptionEnum.NORMAL.id) {
            return checkConnect;
        }
        return YunHsmExceptionEnum.NORMAL;
    }

    private static void deleteConDir(String yunHsmPath) {
        File confDir = new File(yunHsmPath + File.separator + "conf");
        if (confDir.exists()) {
            GMSSLX509Utils.deleteDirectory(yunHsmPath + File.separator + "conf");
        }
        confDir.mkdir();
    }

    private static String getYunHsmPath() {
        String os = System.getProperty("os.name");
        String yunhsmPath;
        if (os.startsWith("Windows")) {
            yunhsmPath = "C:\\Program Files (x86)\\yunhsmsdk";
        } else {
            yunhsmPath = "/usr/local/yunhsmsdk";
        }
        return yunhsmPath;
    }

    /**
     * 备份配置文件
     *
     * @param path 密码机安装路径
     */
    private static void backUpConf(String path) throws IOException {
        String bakPath = path + File.separator + "confBak" + File.separator;
        File bakFile = new File(bakPath);
        if (bakFile.exists() && bakFile.isDirectory()) {
            return;
        }
        String sourcePath = path + File.separator + "conf" + File.separator;
        GMSSLX509Utils.copyDir(sourcePath, bakPath);
        logger.info("备份成功");
    }

    /**
     * 存储证书
     *
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxStream   签名证书    （pfx格式证书）
     * @param encPfxStream    加密证书    （pfx格式证书）
     * @param trustP7bStream  密码机CA证书链  （p7b格式证书链）
     */
    private static YunHsmExceptionEnum storeCert(String signPfxPassword, String encPfxPassword, FileInputStream signPfxStream, FileInputStream encPfxStream,
                                                 FileInputStream trustP7bStream, String yunHsmPath) throws Exception {
        //验证给的密码是否是正确的keyStore密码

        YunHsmInfoEntry yunHsmInfo = GMSSLYunHsmUtils.getYunHsmInfo(yunHsmPath);
        if (signPfxStream == null) {
            String signCertName = yunHsmInfo.getSignCertName();
            File signPfx = new File(yunHsmPath + File.separator + "conf" + File.separator + signCertName);
            signPfxStream = new FileInputStream(signPfx);
        }
        if (encPfxStream == null) {
            String encCertName = yunHsmInfo.getEncCertName();
            File signPfx = new File(yunHsmPath + File.separator + "conf" + File.separator + encCertName);
            encPfxStream = new FileInputStream(signPfx);
        }
        if (trustP7bStream == null) {
            String caCertName = yunHsmInfo.getCaCertName();
            File trustP7b = new File(yunHsmPath + File.separator + "conf" + File.separator + caCertName);
            trustP7bStream = new FileInputStream(trustP7b);
        }

        KeyStore signKse;
        KeyStore encKse;
        try {
            signKse = KeyStore.getInstance("pkcs12", BouncyCastleProvider.PROVIDER_NAME);
            signKse.load(signPfxStream, signPfxPassword.toCharArray());
            logger.info("签名PFX证书解析成功");
        } catch (Exception e) {
            logger.error("签名证书保护口令不能打开加密证书");
            return YunHsmExceptionEnum.SIGN_PASSWORD_IS_ERROR;
        }
        try {
            encKse = KeyStore.getInstance("pkcs12", BouncyCastleProvider.PROVIDER_NAME);
            encKse.load(encPfxStream, encPfxPassword.toCharArray());
            logger.info("加密PFX证书解析成功");
        } catch (Exception e) {
            logger.error("加密证书保护口令不能打开加密证书");
            return YunHsmExceptionEnum.ENC_PASSWORD_IS_ERROR;
        }
        String debugPath = yunHsmPath + File.separator + "confDebug" + File.separator;
        File file = new File(debugPath);
        file.mkdir();
        //在调试目录进行存储证书
        GMSSLKeyStoreUtils.saveGMSSLPfx(signKse, signPfxPassword, debugPath, "sign_" + signPfxPassword);
        GMSSLKeyStoreUtils.saveGMSSLPfx(encKse, encPfxPassword, debugPath, "enc_" + encPfxPassword);
        GMSSLX509Utils.copyFile(trustP7bStream, debugPath + "trust_chain.p7b");
        //获取p7b和pfx中的证书
        List<X509Certificate> trustList;
        try {
            trustList = GMSSLX509Utils.getCertsByCertChain(new FileInputStream(new File(debugPath + "trust_chain.p7b")));
        } catch (Exception e) {
            logger.error("解析密码机CA证书链失败");
            return YunHsmExceptionEnum.OPEN_TRAIN_CERT_P7b_IS_ERROR;
        }

        List<X509Certificate> signList = GMSSLX509Utils.readCertificatesFromP12(debugPath + "sign_" + signPfxPassword + ".pfx", signPfxPassword.toCharArray());
        List<X509Certificate> encList = GMSSLX509Utils.readCertificatesFromP12(debugPath + "enc_" + encPfxPassword + ".pfx", encPfxPassword.toCharArray());
        //keystore 验签
        for (int i = 0; i < signList.size(); i++) {
            if (!GMSSLX509Utils.verifyCert(signList.get(i), trustList)) {
                GMSSLX509Utils.deleteDirectory(debugPath);
                logger.error("第 " + (i + 1) + "个签名证书验签失败");
                return YunHsmExceptionEnum.SIGN_CERT_VERIFY_IS_ERROR;
            }
        }
        logger.info("签名证书验签全部通过");
        for (int i = 0; i < encList.size(); i++) {
            if (!GMSSLX509Utils.verifyCert(encList.get(i), trustList)) {
                GMSSLX509Utils.deleteDirectory(debugPath);
                logger.error("第" + (i + 1) + "个加密证书验签失败");
                return YunHsmExceptionEnum.ENC_CERT_VERIFY_IS_ERROR;
            }
        }
        logger.info("加密证书验签全部通过");
        String path = yunHsmPath + File.separator + "conf" + File.separator;

        //删除原来的配置文件目录，并创建新的目录
        deleteConDir(yunHsmPath);
        //复制文件
        GMSSLX509Utils.copyDir(debugPath, path);
        //删除调试文件夹
        GMSSLX509Utils.deleteDirectory(debugPath);
        logger.debug("调试文件夹删除成功");
        logger.info("证书替换成功");
        return YunHsmExceptionEnum.NORMAL;
    }


    /**
     * 存储证书
     *
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxPath     签名证书地址    （pfx格式证书）
     * @param encPfxPath      加密证书地址    （pfx格式证书）
     * @param trustP7bPath    密码机CA证书链  （p7b格式证书链）
     */
    private static YunHsmExceptionEnum storeCert(String signPfxPassword, String encPfxPassword, String signPfxPath, String
            encPfxPath,
                                                 String trustP7bPath, String yunHsmPath) throws Exception {
        File signPfx = new File(signPfxPath);
        FileInputStream signPfxStream = new FileInputStream(signPfx);

        File encPfx = new File(encPfxPath);
        FileInputStream encPfxStream = new FileInputStream(encPfx);

        File trustP7b = new File(trustP7bPath);
        FileInputStream trustP7bStream = new FileInputStream(trustP7b);

        return storeCert(signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream, yunHsmPath);

    }

    /**
     * 写配置文件
     *
     * @param yunhsmPath      密码机安装路径
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     */
    private static void writeConf(String yunhsmPath, String ip, int port, String signPfxPassword, String
            encPfxPassword) throws IOException {

        String os = System.getProperty("os.name");

        String resoucePath = yunhsmPath + File.separator + "confBak" + File.separator + "yunhsmsdk.conf";

        String confPath = yunhsmPath + File.separator + "conf" + File.separator + "yunhsmsdk.conf";


        BufferedReader br = new BufferedReader(new FileReader(resoucePath));// 读取原始json文件

        File file = new File(resoucePath);
        String jsonString = FileUtils.readFileToString(file, "UTF-8");
        //总目录
        JSONObject dataJson = JSONObject.parseObject(jsonString);
        //Certificate目录下
        JSONObject certificate = dataJson.getJSONObject("Certificate");
        //SignatureCertificate
        JSONObject softCert = certificate.getJSONObject("SoftCert");
        JSONObject signatureCertificate = softCert.getJSONObject("SignatureCertificate");

        signatureCertificate.put("pin", signPfxPassword);
        //EncryptCertificate
        JSONObject encryptCertificate = softCert.getJSONObject("EncryptCertificate");

        encryptCertificate.put("pin", encPfxPassword);

        JSONObject hsm = dataJson.getJSONObject("hsm");
        hsm.put("ip", ip);
        hsm.put("port", port);
        JSONObject ssl = dataJson.getJSONObject("ssl");


        if (os.startsWith("Windows")) {
            signatureCertificate.put("file", yunhsmPath + "\\conf\\sign_" + signPfxPassword + ".pfx");
            encryptCertificate.put("file", yunhsmPath + "\\conf\\enc_" + encPfxPassword + ".pfx");
            ssl.put("CertificatePath", yunhsmPath + "\\conf\\trust_chain.p7b");
        } else {
            //  if (os.equals("Linux")) {
            signatureCertificate.put("file", yunhsmPath + "/conf/sign_" + signPfxPassword + ".pfx");
            encryptCertificate.put("file", yunhsmPath + "/conf/enc_" + encPfxPassword + ".pfx");
            ssl.put("CertificatePath", yunhsmPath + "/conf/trust_chain.p7b");
        }

        String ws = dataJson.toString();
        BufferedWriter bw = new BufferedWriter(new FileWriter(confPath));// 输出新的json文件
        String s = formatJson(ws);
        bw.write(s);
        bw.flush();
        br.close();
        bw.close();
        logger.info("配置文件改写成功");
    }

    /**
     * 密码机链接测试功能接口
     */
    public static YunHsmExceptionEnum testConnect(String host, int port) {
        try {
            if (!isHostConnectable(host, port)) {
                return YunHsmExceptionEnum.TELNET_PORT_FAILURE;
            }
            YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();
            sdfSDK.init();
            sdfSDK.getDeviceInfo();
            sdfSDK.release();
            logger.info("connect yunhsm is successful");

        } catch (SdfSDKException e) {
            logger.error("open device is failure");
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
        return YunHsmExceptionEnum.NORMAL;
    }


    /**
     * 密码机链接测试功能接口
     */
    public static boolean testConnect() {
        try {
            YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();
            sdfSDK.init();
            sdfSDK.getDeviceInfo();
            sdfSDK.release();
            logger.info("connect yunhsm is successful");
            return true;
        } catch (SdfSDKException e) {
            logger.error("open device is failure");
            return false;
        }
    }

    /**
     * 测试端口连通性
     */
    private static boolean isHostConnectable(String host, int port) {
        Socket socket = new Socket();
        try {
            socket.connect(new InetSocketAddress(host, port));
        } catch (IOException e) {
            return false;
        } finally {
            try {
                socket.close();
            } catch (IOException ignored) {
            }
        }
        return true;
    }


    private static String formatJson(String jsonStr) {
        if (null == jsonStr || "".equals(jsonStr)) return "";
        StringBuilder sb = new StringBuilder();
        char last = '\0';
        char current = '\0';
        int indent = 0;
        for (int i = 0; i < jsonStr.length(); i++) {
            last = current;
            current = jsonStr.charAt(i);
            switch (current) {
                case '{':
                case '[':
                    sb.append(current);
                    sb.append('\n');
                    indent++;
                    addIndentBlank(sb, indent);
                    break;
                case '}':
                case ']':
                    sb.append('\n');
                    indent--;
                    addIndentBlank(sb, indent);
                    sb.append(current);
                    break;
                case ',':
                    sb.append(current);
                    if (last != '\\') {
                        sb.append('\n');
                        addIndentBlank(sb, indent);
                    }
                    break;
                default:
                    sb.append(current);
            }
        }

        return sb.toString();
    }

    private static void addIndentBlank(StringBuilder sb, int indent) {
        for (int i = 0; i < indent; i++) {
            sb.append('\t');
        }
    }
}
