package com.xdja.pki.gmssl.tomcat.utils;


import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;

public class GMSSLTomcatUtils {
    private static Logger logger = LoggerFactory.getLogger(GMSSLTomcatUtils.class);

    /**
     * CA服务器初始化tomcat调用接口
     *
     * @param caCert             CA根证书
     * @param caServerSignCert   CA服务器签名证书
     * @param caServerEncCert    CA服务器加密证书
     * @param caServerSignPriKey CA服务器签名私钥
     * @param caServerEncPriKey  CA服务器加密私钥
     * @param tomcatPath         tomcat绝对路径
     * @param port               需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByBC(X509Certificate caCert, X509Certificate caServerSignCert, X509Certificate caServerEncCert,
                                         PrivateKey caServerSignPriKey, PrivateKey caServerEncPriKey, String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        saveKeyStore(caCert, caServerSignCert, caServerEncCert, caServerSignPriKey, caServerEncPriKey, tomcatPath, String.valueOf(port));
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLV1.1", port);

    }

    /**
     * CA服务器初始化tomcat调用接口
     *
     * @param caCert                     CA根证书
     * @param caServerSignCert           CA服务器签名证书
     * @param caServerEncCert            CA服务器加密证书
     * @param caServerPrivateKeyIndex    CA服务器证书密钥控制索引
     * @param caServerPrivateKeyPassword CA服务器证书密钥控制访问码
     * @param tomcatPath                 tomcat绝对路径
     * @param port                       需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByYunHsm(X509Certificate caCert, X509Certificate caServerSignCert, X509Certificate caServerEncCert,
                                             int caServerPrivateKeyIndex, String caServerPrivateKeyPassword,
                                             String tomcatPath, int port) throws Exception {

        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(caServerPrivateKeyIndex, caServerPrivateKeyPassword.getBytes());
        saveKeyStore(caCert, caServerSignCert, caServerEncCert, sdfPrivateKey, sdfPrivateKey, tomcatPath, String.valueOf(port));
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLSDFYUNHSMV1.1", port);
    }

    /**
     * CA服务器初始化tomcat调用接口
     *
     * @param caCert                     CA根证书
     * @param caServerSignCert           CA服务器签名证书
     * @param caServerEncCert            CA服务器加密证书
     * @param caServerPrivateKeyIndex    CA服务器证书密钥控制索引
     * @param caServerPrivateKeyPassword CA服务器证书密钥控制访问码
     * @param tomcatPath                 tomcat绝对路径
     * @param port                       需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByPCIE(X509Certificate caCert, X509Certificate caServerSignCert, X509Certificate caServerEncCert,
                                           int caServerPrivateKeyIndex, String caServerPrivateKeyPassword,
                                           String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(caServerPrivateKeyIndex, caServerPrivateKeyPassword.getBytes());
        saveKeyStore(caCert, caServerSignCert, caServerEncCert, sdfPrivateKey, sdfPrivateKey, tomcatPath, String.valueOf(port));
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLSDFPCIEV1.1", port);
    }

    /**
     * 开tomcat  http 服务端口
     *
     * @param tomcatPath tomcat绝对路径
     * @param port       需要Http服务通道端口号k
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpPort(String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.addTomcatHttpPort(tomcatPath, port);
    }

    /**
     * 关闭tomcat服务端口 http和https端口公用一个方法
     *
     * @param tomcatPath tomcat绝对路径
     * @param port       需要关闭的通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void closeTomcatPort(String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.closeTomcatPort(tomcatPath, port);
    }

    public static boolean restartTomcat(String tomcatPath) {
        if (!shutdownTomcat(tomcatPath)) {
            return false;
        }
        return startupTomcat(tomcatPath);
    }

    public static boolean restartTomcat(String tomcatPath, String shutdownShellPath) {
        if (!run(tomcatPath, "shutdown.sh " + tomcatPath, true, shutdownShellPath)) {
            return false;
        }
        return startupTomcat(tomcatPath);
    }


    public static boolean shutdownTomcat(String tomcatPath) {
        return run(tomcatPath, "shutdown", false);
    }

    public static boolean startupTomcat(String tomcatPath) {
        return run(tomcatPath, "startup", false);
    }

    private static void saveKeyStore(X509Certificate caCert, X509Certificate caServerSignCert, X509Certificate caServerEncCert, PrivateKey caServerSignPriKey, PrivateKey caServerEncPriKey, String path, String port) throws Exception {
        path = path + File.separator + "conf" + File.separator + "cert" + File.separator + port + File.separator;
        File pathf = new File(path);
        if (!pathf.exists()) {
            pathf.mkdirs();
        }
        String password = "password";
        String signAlias = "sign";
        String encAlias = "enc";

        KeyStore serverKeyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStore(password, caCert,
                signAlias, caServerSignPriKey, caServerSignCert,
                encAlias, caServerEncPriKey, caServerEncCert);
        KeyStore trustKeyStore = GMSSLKeyStoreUtils.generateGMSSLTrustStore(caCert);
        GMSSLX509Utils.writeCertificateToPem(path, "ca", caCert);
        GMSSLX509Utils.writeObjectToPem(path, "ca_public", caCert.getPublicKey());
        GMSSLX509Utils.writeCertificateToCer(path, "ca", caCert);

        GMSSLX509Utils.writeCertificateToPem(path, "sign", caServerSignCert);
        GMSSLX509Utils.writeObjectToPem(path, "sign_public", caServerSignCert.getPublicKey());
        if (caServerSignPriKey instanceof SdfPrivateKey) {
            GMSSLX509Utils.writeSdfPrivateKey(path, "ca_server_key", caServerSignPriKey);
//            FileWriter fileWritter = new FileWriter(path + "ca_server_key.pem", false);
//            fileWritter.write(caServerSignPriKey.toString());
//            fileWritter.close();
        } else {
            GMSSLX509Utils.writeObjectToPem(path, "sign_private", caServerSignPriKey);
            GMSSLX509Utils.writeObjectToPem(path, "enc_private", caServerEncPriKey);
            GMSSLX509Utils.writeKeyStoreToP12(serverKeyStore, "password".toCharArray(), path, "sign");
            GMSSLX509Utils.writeKeyStoreToP12(serverKeyStore, "password".toCharArray(), path, "enc");
        }
        GMSSLX509Utils.writeCertificateToCer(path, "sign", caServerSignCert);
        GMSSLX509Utils.writeCertificateToPem(path, "enc", caServerEncCert);
        GMSSLX509Utils.writeCertificateToCer(path, "enc", caServerEncCert);
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(serverKeyStore, "password", path, "server");
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(trustKeyStore, "password", path, "trust");
        GMSSLX509Utils.writeKeyStoreToP12(trustKeyStore, "password".toCharArray(), path, "trust");

    }


    /**
     * 启动Tomcat程序
     *
     * @param tomcatPath
     * @param shName
     */
    private static boolean run(String tomcatPath, String shName, boolean isUseShutdownSh, String... shPath) {
        Runtime rt = Runtime.getRuntime();
        Process ps = null;
        try {
            String os = System.getProperty("os.name");
            if (shName.contains("shut")) {
                logger.info("=================开始关闭Tomcat=================");
            } else {
                logger.info("=================开始启动Tomcat=================");
            }
            if (os.startsWith("Windows")) {
                ps = rt.exec("cmd /c " + tomcatPath + File.separator + "bin" + File.separator + shName + ".bat", null, new File(tomcatPath));
            } else {
                if (isUseShutdownSh) {
                    ps = rt.exec("sh " + shPath[0] + File.separator + shName, null, new File(shPath[0]));
                } else {
                    ps = rt.exec("sh " + tomcatPath + File.separator + "bin" + File.separator + shName + ".sh", null, new File(tomcatPath));
                }

            }
            InputStream is = ps.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            ps.waitFor();
            is.close();
            reader.close();
            ps.destroy();
            logger.info("=================完成=================");
            return true;
        } catch (Exception e) {
            logger.error(shName + " tomcat is error ", e);
            return false;
        }
    }
}