package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentVerifierProvider;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentSignatureAlgorithm;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentSignerUtils;
import com.xdja.pki.gmssl.x509.utils.bean.CRLEntry;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.CRLNumber;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.CertException;
import org.bouncycastle.cert.X509CRLHolder;
import org.bouncycastle.cert.X509v2CRLBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CRLConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.util.BigIntegers;

import java.io.IOException;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CRLException;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.List;

public class GMSSLCRLUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                 CRL 生成                                         *
     ************************************************************************************/

    /**
     * 生成 CRL 默认 sm3_with_sm2 签名，使用 BC 软实现签名
     *
     * @param issueCert  签发者证书
     * @param issuerKey  签发者私钥信息
     * @param crlNumber  CRL 序列号
     * @param nextUpdate 下次更新时间
     * @param extensions 扩展项列表
     * @param revoked    CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRLByBC(
            X509Certificate issueCert, PrivateKey issuerKey,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByBC(GMSSLSdfContentSignatureAlgorithm.SM3_WITH_SM2, issuerKey);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    /**
     * 生成 CRL 默认 sm3_with_sm2 签名，使用 密码机 签名
     *
     * @param issueCert         签发者证书
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param crlNumber         CRL 序列号
     * @param nextUpdate        下次更新时间
     * @param extensions        扩展项列表
     * @param revoked           CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRLByYunhsm(
            X509Certificate issueCert, int issuerKeyIndex, String issuerKeyPassword,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByYunhsm(sdfPrivateKey);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    /**
     * 生成 CRL 默认 sm3_with_sm2 签名，默认使用 PCIE 签名
     *
     * @param issueCert         签发者证书
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param crlNumber         CRL 序列号
     * @param nextUpdate        下次更新时间
     * @param extensions        扩展项列表
     * @param revoked           CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常userCertificateSerial
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRLByPCIE(
            X509Certificate issueCert, int issuerKeyIndex, String issuerKeyPassword,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByPcie(sdfPrivateKey);
        return generateCRL(issueCert, contentSigner, crlNumber, thisUpdate, nextUpdate, extensions, revoked);
    }

    /**
     * 生成 CRL
     *
     * @param issueCert     签发者证书
     * @param contentSigner 内容签名者
     * @param crlNumber     CRL 序列号
     * @param nextUpdate    下次更新时间
     * @param extensions    扩展项列表
     * @param revoked       CRL实体 列表
     * @return X509CRL
     * @throws IOException  添加扩展项异常
     * @throws CRLException CRL生成异常
     */
    public static X509CRL generateCRL(
            X509Certificate issueCert, ContentSigner contentSigner,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions, List<CRLEntry> revoked
    ) throws IOException, CRLException {
        X509v2CRLBuilder builder = new X509v2CRLBuilder(new X500Name(issueCert.getSubjectDN().getName()), thisUpdate);

        builder.addExtension(Extension.cRLNumber, false, new CRLNumber(crlNumber));
        builder.setNextUpdate(nextUpdate);

        //extensions
        if (extensions != null){
            for (Extension extension : extensions) {
                builder.addExtension(extension);
            }
        }

        if (revoked != null) {
            for (CRLEntry entry : revoked) {
                builder.addCRLEntry(new BigInteger(entry.getUserCertificateSerial(), 16), entry.getRevocationDate(), entry.getReason());
            }
        }

        X509CRLHolder crlHolder = builder.build(contentSigner);
        return new JcaX509CRLConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCRL(crlHolder);
    }

    public static X509CRL decodeCRLByBC(PublicKey publicKey, byte[] crlEncode) throws OperatorCreationException, CRLException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        return decodeCRL(contentVerifierProvider, crlEncode);
    }

    public static X509CRL decodeCRLByYunhsm(PublicKey publicKey, byte[] crlEncode) throws CRLException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new SdfContentVerifierProvider(SdfCryptoType.YUNHSM, publicKey);
        return decodeCRL(contentVerifierProvider, crlEncode);
    }

    public static X509CRL decodeCRLByPcie(PublicKey publicKey, byte[] crlEncode) throws CRLException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new SdfContentVerifierProvider(SdfCryptoType.PCIE, publicKey);
        return decodeCRL(contentVerifierProvider, crlEncode);
    }

    public static X509CRL decodeCRLBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] crlEncode) throws CRLException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new SdfContentVerifierProvider(sdfCryptoType, publicKey);
        return decodeCRL(contentVerifierProvider, crlEncode);
    }

    public static X509CRL decodeCRL(ContentVerifierProvider contentVerifierProvider, byte[] crlEncode) throws IOException, CertException, CRLException {
        X509CRLHolder holder = new X509CRLHolder(crlEncode);
        holder.isSignatureValid(contentVerifierProvider);
        return new JcaX509CRLConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCRL(holder);
    }
}
