package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentSignatureAlgorithm;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentSignerUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentVerifierUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.ContentVerifier;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.PKCS10CertificationRequestBuilder;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.*;
import java.security.spec.InvalidKeySpecException;

public class GMSSLP10Utils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLP10Utils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                 P10 生成 解析                                    *
     ************************************************************************************/
    /**
     * 生成 P10 证书申请  默认 sm3_with_sm2 签名，使用 BC 签名
     *
     * @param subject    主体 string DN
     * @param publicKey  主体公钥信息
     * @param privateKey 主体私钥信息
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10SignByBC(String subject, PublicKey publicKey, PrivateKey privateKey) throws OperatorCreationException {
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByBC(GMSSLSdfContentSignatureAlgorithm.SM3_WITH_SM2, privateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    /**
     * 生成 P10 证书申请  默认 sm3_with_sm2 签名，使用 密码机 签名
     *
     * @param subject            主体 string DN
     * @param publicKey          主体公钥信息
     * @param privateKeyIndex    主体私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param privateKeyPassword 主体私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10SignByYunhsm(
            String subject, PublicKey publicKey,
            int privateKeyIndex, String privateKeyPassword
    ) throws OperatorCreationException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByYunhsm(sdfPrivateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    /**
     * 生成 P10 证书申请  默认 sm3_with_sm2 签名，使用 密码机 签名
     *
     * @param subject            主体 string DN
     * @param publicKey          主体公钥信息
     * @param privateKeyIndex    主体私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param privateKeyPassword 主体私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10SignByPcie(
            String subject, PublicKey publicKey,
            int privateKeyIndex, String privateKeyPassword
    ) throws OperatorCreationException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByPcie(sdfPrivateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    public static PKCS10CertificationRequest generateP10(String subject, ContentSigner contentSigner, PublicKey publicKey) throws OperatorCreationException {
        PKCS10CertificationRequestBuilder requestBuilder = new JcaPKCS10CertificationRequestBuilder(new X500Name(subject), publicKey);

        PKCS10CertificationRequest pkcs10 = requestBuilder.build(contentSigner);
        GMSSLByteArrayUtils.printHexBinary(logger, "generate p10 signature", pkcs10.getSignature());

        return pkcs10;
    }

    /**
     * P10 证书申请 编码成二进制
     *
     * @param pkcs10CertificationRequest org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @return 二进制编码
     * @throws IOException 编码异常
     */
    public static byte[] encodeP10(PKCS10CertificationRequest pkcs10CertificationRequest) throws IOException {
        return pkcs10CertificationRequest.getEncoded();
    }


    /**
     * P10 证书申请解码
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10(String base64P10) throws IOException {
        return new PKCS10CertificationRequest(GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10(byte[] p10) throws IOException {
        return new PKCS10CertificationRequest(p10);
    }

    /**
     * P10 证书申请解码 使用 BC 验签
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByBC(String base64P10) throws IOException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(GMSSLByteArrayUtils.base64Decode(base64P10));
        decodeP10VerifyByBC(pkcs10CertificationRequest);
        return new PKCS10CertificationRequest(GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码 使用 BC 验签
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByBC(byte[] p10) throws IOException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(p10);
        decodeP10VerifyByBC(pkcs10CertificationRequest);
        return new PKCS10CertificationRequest(p10);
    }

    private static void decodeP10VerifyByBC(PKCS10CertificationRequest pkcs10CertificationRequest) throws IOException {
        SubjectPublicKeyInfo subjectPublicKeyInfo = pkcs10CertificationRequest.getSubjectPublicKeyInfo();
        boolean verify;
        try {
            verify = pkcs10CertificationRequest.isSignatureValid(new JcaContentVerifierProviderBuilder().setProvider("BC").build(subjectPublicKeyInfo));
        } catch (PKCSException | OperatorCreationException e) {
            throw new IOException("verify error", e);
        }
        if (!verify) {
            throw new IOException("can not verify");
        }
    }

    /**
     * P10 证书申请解码 使用 密码机 验签
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByYunhsm(String base64P10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.YUNHSM, GMSSLByteArrayUtils.base64Decode(base64P10));
    }

    /**
     * P10 证书申请解码 使用 密码机 验签
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByYunhsm(byte[] p10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.YUNHSM, p10);
    }


    /**
     * P10 证书申请解码 使用 PCIE 验签
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByPCIE(String base64P10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.PCIE, GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码 使用 PCIE 验签
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByPCIE(byte[] p10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.PCIE, p10);
    }


    /**
     * P10 证书申请解码 使用 SDF 验签
     *
     * @param sdfCryptoType SDF 密码算法类型 密码机 PCIE
     * @param base64P10     Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyBySDF(SdfCryptoType sdfCryptoType, String base64P10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(base64P10);
        decodeP10VerifyBySDF(pkcs10CertificationRequest, sdfCryptoType);
        return new PKCS10CertificationRequest(GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码 使用 SDF 验签
     *
     * @param sdfCryptoType SDF 密码算法类型 密码机 PCIE
     * @param p10           二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyBySDF(SdfCryptoType sdfCryptoType, byte[] p10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(p10);
        decodeP10VerifyBySDF(pkcs10CertificationRequest, sdfCryptoType);
        return new PKCS10CertificationRequest(p10);
    }

    private static void decodeP10VerifyBySDF(PKCS10CertificationRequest pkcs10CertificationRequest, SdfCryptoType sdfCryptoType) throws InvalidKeySpecException, NoSuchAlgorithmException, IOException, SdfSDKException {
        SubjectPublicKeyInfo subjectPublicKeyInfo = pkcs10CertificationRequest.getSubjectPublicKeyInfo();
        PublicKey publicKey = GMSSLX509Utils.convertSM2PublicKey(subjectPublicKeyInfo);
        ContentVerifierProvider provider = new ContentVerifierProvider() {
            public boolean hasAssociatedCertificate() {
                return false;
            }

            public X509CertificateHolder getAssociatedCertificate() {
                return null;
            }

            public ContentVerifier get(AlgorithmIdentifier algorithm) throws OperatorCreationException {
                return GMSSLSdfContentVerifierUtils.generateContentVerifierBySdf(sdfCryptoType, publicKey);
            }
        };
        boolean verify;
        try {
            verify = pkcs10CertificationRequest.isSignatureValid(provider);
        } catch (PKCSException e) {
            throw new SdfSDKException("verify error", e);
        }
        if (!verify) {
            throw new IOException("can not verify");
        }
    }
}
