package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.asn1.crypto.ASN1SM2Cipher;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ECKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.Arrays;


import java.io.IOException;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class GMSSLSM2EncryptUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                      加密解密                                    *
     *                   GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3             *
     ************************************************************************************/

    /**
     * 使用 BC 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptByBC(PublicKey publicKey, String base64Data) throws CryptoException, IOException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPublicKeyParameters(publicKey);
        byte[] cipher = encryptByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(cipher);
    }

    /**
     * 使用 密码机 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptByYunhsm(PublicKey publicKey, String base64Data) throws InvalidCipherTextException, SdfSDKException {
        return encryptBySdf(SdfCryptoType.YUNHSM, publicKey, base64Data);
    }

    /**
     * 使用 PCIE 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptByPcie(PublicKey publicKey, String base64Data) throws InvalidCipherTextException, SdfSDKException {
        return encryptBySdf(SdfCryptoType.PCIE, publicKey, base64Data);
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sdfCipher = encryptBySdf(sdfCryptoType, publicKey, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(sdfCipher);
    }

    /**
     * 使用 BC 进行解密
     *
     * @param privateKey 加密者私钥
     * @param base64Data BASE64 编码 加密消息 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 加密原文
     */
    public static String decryptByBC(PrivateKey privateKey, String base64Data) throws IOException, InvalidCipherTextException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw InvalidCipherTextException
        byte[] out = decryptByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(out);
    }

    /**
     * 使用 密码机 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 明文
     */
    public static String decryptByYunhsm(int privateKeyIndex, String privateKeyPassword, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        return decryptBySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 PCIE 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 明文
     */
    public static String decryptByPcie(int privateKeyIndex, String privateKeyPassword, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        return decryptBySdf(SdfCryptoType.PCIE, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 SDF 进行解密
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 明文
     */
    public static String decryptBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] out = decryptBySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(out);
    }


    /************************************************************************************
     *                                      加密解密                                    *
     *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
     ************************************************************************************/

    /**
     * 使用 BC 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     */
    public static String encryptASN1ByBC(PublicKey publicKey, String base64Data) throws CryptoException, IOException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw IOException
        byte[] cipher = encryptASN1ByBC(publicKey, data);
        return GMSSLByteArrayUtils.base64Encode(cipher);
    }

    /**
     * 使用 BC 进行加密
     *
     * @param publicKey 加密者公钥
     * @param data      BASE64 编码 待加密消息
     * @return 二进制 编码 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     */
    public static byte[] encryptASN1ByBC(PublicKey publicKey, byte[] data) throws CryptoException, IOException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPublicKeyParameters(publicKey);
        byte[] cipher = encryptASN1ByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
        return cipher;
    }

    /**
     * 使用 密码机 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文
     */
    public static String encryptASN1ByYunhsm(PublicKey publicKey, String base64Data) throws CryptoException, SdfSDKException {
        return encryptASN1BySdf(SdfCryptoType.YUNHSM, publicKey, base64Data);
    }

    /**
     * 使用 PCIE 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文
     */
    public static String encryptASN1ByPcie(PublicKey publicKey, String base64Data) throws CryptoException, SdfSDKException {
        return encryptASN1BySdf(SdfCryptoType.PCIE, publicKey, base64Data);
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文
     */
    public static String encryptASN1BySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sdfCipher = encryptASN1BySdf(sdfCryptoType, publicKey, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(sdfCipher);
    }

    /**
     * 使用 BC 进行解密
     *
     * @param privateKey 加密者私钥
     * @param base64Data BASE64 编码 加密消息
     * @return BASE64 编码 加密原文
     */
    public static String decryptASN1ByBC(PrivateKey privateKey, String base64Data) throws IOException, InvalidCipherTextException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw InvalidCipherTextException
        byte[] out = decryptASN1ByBC(privateKey, data);
        return GMSSLByteArrayUtils.base64Encode(out);
    }

    /**
     * 使用 BC 进行解密
     *
     * @param privateKey 加密者私钥
     * @return 加密原文
     */
    public static byte[] decryptASN1ByBC(PrivateKey privateKey, byte[] data) throws IOException, InvalidCipherTextException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
        //will throw InvalidCipherTextException
        byte[] out = decryptASN1ByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        return out;
    }

    /**
     * 使用 密码机 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文
     * @return BASE64 编码 明文
     */
    public static String decryptASN1ByYunhsm(int privateKeyIndex, String privateKeyPassword, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        return decryptASN1BySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 PCIE 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文
     * @return BASE64 编码 明文
     */
    public static String decryptASN1ByPcie(int privateKeyIndex, String privateKeyPassword, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        return decryptASN1BySdf(SdfCryptoType.PCIE, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 SDF 进行解密
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文
     * @return BASE64 编码 明文
     */
    public static String decryptASN1BySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data) throws SdfSDKException, InvalidCipherTextException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] out = decryptASN1BySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(out);
    }

    /**
     * SM2 加密 输出 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     *
     * @param keyParameter 密钥配置
     * @param secureRandom 随机数
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 输出 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @throws InvalidCipherTextException
     */
    public static byte[] encryptByBC(ECKeyParameters keyParameter, SecureRandom secureRandom, byte[] input, int inOff, int length) throws InvalidCipherTextException {
        SM2Engine sm2Engine = new SM2Engine();
        sm2Engine.init(true, new ParametersWithRandom(keyParameter, secureRandom));
        return sm2Engine.processBlock(input, inOff, length);
    }

    /**
     * SM2 解密 输入 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 明文
     * @throws InvalidCipherTextException
     */
    public static byte[] decryptByBC(ECKeyParameters keyParameter, byte[] input, int inOff, int length) throws InvalidCipherTextException {
        SM2Engine sm2Engine = new SM2Engine();
        sm2Engine.init(false, keyParameter);
        return sm2Engine.processBlock(input, inOff, length);
    }

    /**
     * SM2 加密 输出 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 输出 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     * @throws InvalidCipherTextException
     */
    public static byte[] encryptASN1ByBC(ECKeyParameters keyParameter, SecureRandom secureRandom, byte[] input, int inOff, int length) throws InvalidCipherTextException, IOException {
        SM2Engine sm2Engine = new SM2Engine();
        //这里用来生成密文，与rsa不同的是不需要用pkcs#1结构体，传入参数为：公钥+随机数
        sm2Engine.init(true, new ParametersWithRandom(keyParameter, secureRandom));

        byte[] ciphertext = sm2Engine.processBlock(input, inOff, length);

        ECCurve curve = keyParameter.getParameters().getCurve();
        int curveLength = (curve.getFieldSize() + 7) / 8;

        byte[] c1 = new byte[curveLength * 2 + 1];
        System.arraycopy(ciphertext, inOff, c1, 0, c1.length);

        ECPoint c1P = curve.decodePoint(c1);
        byte[] xb = c1P.getXCoord().getEncoded();
        byte[] yb = c1P.getYCoord().getEncoded();

        SM3Digest digest = new SM3Digest();

        byte[] c2 = new byte[length];
        System.arraycopy(ciphertext, ciphertext.length - length - digest.getDigestSize(), c2, 0, length);

        byte[] c3 = new byte[digest.getDigestSize()];
        System.arraycopy(ciphertext, ciphertext.length - digest.getDigestSize(), c3, 0, digest.getDigestSize());

        ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(xb, yb, c3, c2);
        byte[] result = asn1SM2Cipher.toASN1Primitive().getEncoded();

        return result;
    }

    /**
     * SM2 解密 输入 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @return 明文
     * @throws InvalidCipherTextException
     */
    public static byte[] decryptASN1ByBC(ECKeyParameters keyParameter, byte[] input) throws InvalidCipherTextException {
        return decryptASN1ByBC(keyParameter, input, 0, input.length);
    }

    /**
     * SM2 解密 输入 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 明文
     * @throws InvalidCipherTextException
     */
    public static byte[] decryptASN1ByBC(ECKeyParameters keyParameter, byte[] input, int inOff, int length) throws InvalidCipherTextException {
        byte[] sm2CipherASN1 = new byte[length];
        System.arraycopy(input, inOff, sm2CipherASN1, 0, length);

        ASN1SM2Cipher asn1 = ASN1SM2Cipher.getInstance(sm2CipherASN1);

        //generate c1
        ECCurve curve = keyParameter.getParameters().getCurve();
        BigInteger x = asn1.getxCoordinate();
        BigInteger y = asn1.getyCoordinate();
        ECPoint point = curve.createPoint(x, y);
        byte[] c1 = point.getEncoded(false);

        //get c2
        byte[] c2 = asn1.getCipherText();

        //get c3
        byte[] c3 = asn1.getHash();

        byte[] ciphertext = Arrays.concatenate(c1, c2, c3);

        SM2Engine sm2Engine = new SM2Engine();
        //这里如果要是 decrypt 解密的话 需要传入 private key
        sm2Engine.init(false, keyParameter);

        byte[] M = sm2Engine.processBlock(ciphertext, 0, ciphertext.length);

        return M;
    }

    /************************************************************************************
     *                                      加密解密                                    *
     *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
     ************************************************************************************/


    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType SDF 密码类型 密码机、PCIE
     * @param publicKey     加密者公钥
     * @param input         待加密消息
     * @param inOff         输入数据 起始位置
     * @param length        输入数据 长度
     * @return GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static byte[] encryptBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] input, int inOff, int length) throws SdfSDKException, InvalidCipherTextException {
        //will throw SdfSDKException
        SdfSM2Engine sdfSM2Engine = new SdfSM2Engine(sdfCryptoType);
        sdfSM2Engine.init(true, new SdfECKeyParameters((ECPublicKey) publicKey));
        //will throw InvalidCipherTextException
        byte[] sdfCipher = sdfSM2Engine.processBlock(input, inOff, length);
        //will throw SdfSDKException
        sdfSM2Engine.release();
        return sdfCipher;
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType      SDF 密码类型 密码机、PCIE
     * @param privateKeyIndex    SDF私钥索引
     * @param privateKeyPassword SDF私钥访问控制码
     * @param input              密文 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     * @param inOff              输入数据 起始位置
     * @param length             输入数据 长度
     * @return 明文
     */
    public static byte[] decryptBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] input, int inOff, int length) throws SdfSDKException, InvalidCipherTextException {
        //will throw SdfSDKException
        SdfSM2Engine sdfSM2EngineDecrypt = new SdfSM2Engine(sdfCryptoType);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        sdfSM2EngineDecrypt.init(false, new SdfECKeyParameters(sdfPrivateKey));
        //will throw InvalidCipherTextException
        byte[] out = sdfSM2EngineDecrypt.processBlock(input, inOff, length);
        //will throw SdfSDKException
        sdfSM2EngineDecrypt.release();
        return out;
    }


    /************************************************************************************
     *                                      加密解密                                    *
     *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
     ************************************************************************************/

    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType SDF 密码类型 密码机、PCIE
     * @param publicKey     加密者公钥
     * @param input         待加密消息
     * @param inOff         输入数据 起始位置
     * @param length        输入数据 长度
     * @return GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     */
    public static byte[] encryptASN1BySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] input, int inOff, int length) throws SdfSDKException, InvalidCipherTextException {
        //will throw SdfSDKException
        SdfSM2Engine sdfSM2Engine = new SdfSM2Engine(sdfCryptoType);
        sdfSM2Engine.init(true, new SdfECKeyParameters((ECPublicKey) publicKey));
        //will throw InvalidCipherTextException
        byte[] sdfCipher = sdfSM2Engine.processBlockASN1(input, inOff, length);
        //will throw SdfSDKException
        sdfSM2Engine.release();
        return sdfCipher;
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType      SDF 密码类型 密码机、PCIE
     * @param privateKeyIndex    SDF私钥索引
     * @param privateKeyPassword SDF私钥访问控制码
     * @param input              密文 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     * @param inOff              输入数据 起始位置
     * @param length             输入数据 长度
     * @return 明文
     */
    public static byte[] decryptASN1BySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] input, int inOff, int length) throws SdfSDKException, InvalidCipherTextException {
        //will throw SdfSDKException
        SdfSM2Engine sdfSM2EngineDecrypt = new SdfSM2Engine(sdfCryptoType);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        sdfSM2EngineDecrypt.init(false, new SdfECKeyParameters(sdfPrivateKey));
        //will throw InvalidCipherTextException
        byte[] out = sdfSM2EngineDecrypt.processBlockASN1(input, inOff, length);
        //will throw SdfSDKException
        sdfSM2EngineDecrypt.release();
        return out;
    }

}
