package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfContentVerifierProvider;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentSignatureAlgorithm;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfContentSignerUtils;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.cert.*;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;

import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.*;
import java.util.Date;
import java.util.List;

public class GMSSLCertUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                  证书生成                                        *
     ************************************************************************************/
    /**
     * 生成证书 默认 sm3_with_sm2 签名，默认 BC 软实现签名
     *
     * @param issuer     签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject    证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial     证书序列号
     * @param notBefore  证书有效期开始时间
     * @param notAfter   证书有效期结束时间
     * @param subjectKey 证书主体 公钥信息
     * @param issuerKey  证书签发者 私钥信息
     * @param extensions 证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCertByBC(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey,
            List<Extension> extensions
    ) throws CertIOException, OperatorCreationException, CertificateException {
        return generateCertByBC(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                issuerKey, subjectKey,
                extensions
        );
    }

    public static X509Certificate generateCertByBC(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey issuerKey, PublicKey subjectKey,
            List<Extension> extensions
    ) throws CertIOException, OperatorCreationException, CertificateException {
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByBC(GMSSLSdfContentSignatureAlgorithm.SM3_WITH_SM2, issuerKey);
        return generateCert(
                issuer, subject,
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    /**
     * 生成证书 默认 sm3_with_sm2 签名，默认使用密码机签名
     *
     * @param issuer            签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject           证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial            证书序列号
     * @param notBefore         证书有效期开始时间
     * @param notAfter          证书有效期结束时间
     * @param subjectKey        证书主体 公钥信息
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param extensions        证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCertByYunhsm(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey,
            List<Extension> extensions
    ) throws CertIOException, OperatorCreationException, CertificateException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        return generateCertByYunhsm (
                issuer, subject,
                serial, notBefore, notAfter,
                sdfPrivateKey, subjectKey,
                extensions
        );
    }

    public static X509Certificate generateCertByYunhsm(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PrivateKey sdfPrivateKey, PublicKey subjectKey,
            List<Extension> extensions
    ) throws CertIOException, OperatorCreationException, CertificateException {
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByYunhsm(sdfPrivateKey);
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, contentSigner,
                extensions
        );
    }

    /**
     * 生成证书 默认 sm3_with_sm2 签名，默认使用密码机签名
     *
     * @param issuer            签发者 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject           证书主体 string DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param serial            证书序列号
     * @param notBefore         证书有效期开始时间
     * @param notAfter          证书有效期结束时间
     * @param subjectKey        证书主体 公钥信息
     * @param issuerKeyIndex    证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param issuerKeyPassword 证书签发者 私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param extensions        证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCertByPcie(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            int issuerKeyIndex, String issuerKeyPassword, PublicKey subjectKey,
            List<Extension> extensions
    ) throws CertIOException, OperatorCreationException, CertificateException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(issuerKeyIndex, issuerKeyPassword);
        ContentSigner contentSigner = GMSSLSdfContentSignerUtils.generateContentSignerByPcie(sdfPrivateKey);
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, contentSigner, extensions
        );
    }

    /**
     * 生成证书 X500Name DN
     *
     * @param issuer        签发者 X500Name DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject       证书主体 X500Name DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param subject       证书主体 X500Name DN 符合RFC 4519规范 例如:CN=USER1,O=XDJA,C=CN
     * @param contentSigner 内容签名者
     * @param serial        证书序列号
     * @param notBefore     证书有效期开始时间
     * @param notAfter      证书有效期结束时间
     * @param subjectKey    证书主体 公钥信息
     * @param extensions    证书扩展项列表
     * @return X509证书
     * @throws CertIOException      添加扩展项异常
     * @throws CertificateException 证书生成异常
     */
    public static X509Certificate generateCert(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PublicKey subjectKey, ContentSigner contentSigner,
            List<Extension> extensions
    ) throws CertIOException, CertificateException {

        //subject public key info
        SubjectPublicKeyInfo subjectPublicKeyInfo = GMSSLX509Utils.convertSubjectPublicKeyInfo(subjectKey);

        //builder
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(issuer, serial, notBefore, notAfter, subject, subjectPublicKeyInfo);

        if (extensions != null){
            //extensions
            for (Extension extension : extensions) {
                //add extension error throw CertIOException
                builder.addExtension(extension);
            }
        }

        //signer
        X509CertificateHolder holder = builder.build(contentSigner);

        //gen certificate getCertificate throw CertificateException
        return new JcaX509CertificateConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCertificate(holder);
    }

    public static X509Certificate decodeCertByBC(PublicKey publicKey, byte[] certEncode) throws OperatorCreationException, CertificateException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        return decodeCert(contentVerifierProvider, certEncode);
    }

    public static X509Certificate decodeCertByYunhsm(PublicKey publicKey, byte[] certEncode) throws CertificateException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new SdfContentVerifierProvider(SdfCryptoType.YUNHSM, publicKey);
        return decodeCert(contentVerifierProvider, certEncode);
    }

    public static X509Certificate decodeCertByPcie(PublicKey publicKey, byte[] certEncode) throws CertificateException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new SdfContentVerifierProvider(SdfCryptoType.PCIE, publicKey);
        return decodeCert(contentVerifierProvider, certEncode);
    }

    public static X509Certificate decodeCertBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] certEncode) throws CertificateException, IOException, CertException {
        ContentVerifierProvider contentVerifierProvider = new SdfContentVerifierProvider(sdfCryptoType, publicKey);
        return decodeCert(contentVerifierProvider, certEncode);
    }

    public static X509Certificate decodeCert(ContentVerifierProvider contentVerifierProvider, byte[] certEncode) throws IOException, CertException, CertificateException {
        X509CertificateHolder holder = new X509CertificateHolder(certEncode);
        holder.isSignatureValid(contentVerifierProvider);
        return new JcaX509CertificateConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCertificate(holder);
    }

}
