package com.xdja.pki.gmssl.crypto.utils;


import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfSM3Digest;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfECCPublicKey;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.Strings;

import java.math.BigInteger;
import java.security.*;
import java.security.interfaces.ECPublicKey;


public class GMSSLSM3DigestUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 使用 BC 摘要算法
     *
     * @param base64Data BASE64 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static String digestByBC(String base64Data) {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] digest = digestByBC(data);
        return GMSSLByteArrayUtils.base64Encode(digest);
    }

    /**
     * 使用 BC 摘要算法
     *
     * @param data 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static byte[] digestByBC(byte[] data) {
//        MessageDigest instance = MessageDigest.getInstance("SM3", BouncyCastleProvider.PROVIDER_NAME);
//        byte[] result = instance.digest(data);
//        return GMSSLByteArrayUtils.base64Encode(result);
        SM3Digest sm3Digest = new SM3Digest();
        sm3Digest.update(data, 0, data.length);
        byte[] out = new byte[sm3Digest.getDigestSize()];
        sm3Digest.doFinal(out, 0);
        return out;
    }

    /**
     * 使用 BC 先把公钥进行摘要计算 在把得到的结果的需要进行摘要计算的数据合并进行 摘要算法
     *
     * @param publicKey  SM2公钥
     * @param base64Data BASE64 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static String digestByBCWithPublicKey(ECPublicKey publicKey, String base64Data) {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] digest = digestByBCWithPublicKey(publicKey, data);
        return GMSSLByteArrayUtils.base64Encode(digest);
    }

    /**
     * 使用 BC  先把公钥进行摘要计算 在把得到的结果的需要进行摘要计算的数据合并进行 摘要算法
     *
     * @param publicKey SM2公钥
     * @param data      原始数据
     * @return BASE64 编码 摘要数据
     */
    public static byte[] digestByBCWithPublicKey(ECPublicKey publicKey, byte[] data) {
        SM3Digest digest = new SM3Digest();
        byte[] userID = GMSSLByteArrayUtils.hexDecode("31323334353637383132333435363738"); //default value
        byte[] x = publicKey.getW().getAffineX().toByteArray();
        byte[] y = publicKey.getW().getAffineY().toByteArray();

        int len = userID.length * 8;
        digest.update((byte) (len >> 8 & 0xFF));
        digest.update((byte) (len & 0xFF));
        digest.update(userID, 0, userID.length);

        //组装 Z 信息
        ECParameterSpec ecParameterSpec = ECNamedCurveTable.getParameterSpec(GMSSLX509Utils.ECC_SM2_NAME);
        ECCurve curve = ecParameterSpec.getCurve();
        //得到椭圆心
        ECPoint G = ecParameterSpec.getG();

        addFieldElement(digest, curve.getA().toBigInteger());
        addFieldElement(digest, curve.getB().toBigInteger());
        addFieldElement(digest, G.getAffineXCoord().toBigInteger());
        addFieldElement(digest, G.getAffineYCoord().toBigInteger());
        addFieldElement(digest, BigIntegers.fromUnsignedByteArray(x));
        addFieldElement(digest, BigIntegers.fromUnsignedByteArray(y));

        byte[] z = new byte[digest.getDigestSize()];

        digest.doFinal(z, 0);

        digest.update(z, 0, z.length);
        digest.update(data, 0, data.length);

        byte[] out = new byte[digest.getDigestSize()];
        digest.doFinal(out, 0);


        return out;
    }

    private static void addFieldElement(Digest digest, BigInteger p) {
        byte[] in = GMSSLByteArrayUtils.asUnsignedByteArray32(p);
        digest.update(in, 0, in.length);
    }

    /**
     * 使用 YUNHSM密码机 进行SM3摘要计算
     *
     * @param base64Data BASE64 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static String digestByYunhsm(String base64Data) throws SdfSDKException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] digest = digestByYunhsm(data);
        return GMSSLByteArrayUtils.base64Encode(digest);
    }

    /**
     * 使用 PCIE卡   进行SM3摘要计算
     *
     * @param base64Data BASE64 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static String digestByPcie(String base64Data) throws SdfSDKException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] digest = digestByPcie(data);
        return GMSSLByteArrayUtils.base64Encode(digest);
    }

    /**
     * 使用 YUNHSM密码机 进行SM3摘要计算
     *
     * @param data 原始数据
     * @return 摘要数据
     */
    public static byte[] digestByYunhsm(byte[] data) throws SdfSDKException {
        return digestBySdf(SdfCryptoType.YUNHSM, data);
    }

    /**
     * 使用 PCIE卡   进行SM3摘要计算
     *
     * @param data 原始数据
     * @return 摘要数据
     */
    public static byte[] digestByPcie(byte[] data) throws SdfSDKException {
        return digestBySdf(SdfCryptoType.PCIE, data);
    }

    /**
     * 使用 SDF模式    进行SM3摘要计算
     *
     * @param sdfCryptoType 使用的加密模式
     * @param data          原始数据
     * @return 摘要数据
     */
    public static byte[] digestBySdf(SdfCryptoType sdfCryptoType, byte[] data) throws SdfSDKException {
        SdfSM3Digest sm3Digest = new SdfSM3Digest(sdfCryptoType);
        sm3Digest.update(data, 0, data.length);
        byte[] out = new byte[sm3Digest.getDigestSize()];
        sm3Digest.doFinal(out, 0);
        return out;
    }

    /**
     * 使用 YUNHSM密码机 进行SM3摘要计算
     *
     * @param publicKey  签名值公钥
     * @param base64Data BASE64 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static String digestByYunhsmWithPublicKey(ECPublicKey publicKey, String base64Data) throws SdfSDKException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] digest = digestByYunhsmWithPublicKey(publicKey, data);
        return GMSSLByteArrayUtils.base64Encode(digest);
    }

    /**
     * 使用 PCIE卡 进行SM3摘要计算
     *
     * @param publicKey  BASE64编码 签名值公钥
     * @param base64Data BASE64编码 原始数据
     * @return BASE64 编码 摘要数据
     */
    public static String digestByPcieWithPublicKey(ECPublicKey publicKey, String base64Data) throws SdfSDKException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] digest = digestByPcieWithPublicKey(publicKey, data);
        return GMSSLByteArrayUtils.base64Encode(digest);
    }

    /**
     * 使用 YUNHSM密码机 进行SM3摘要计算
     *
     * @param publicKey 签名值公钥
     * @param data      原始数据
     * @return 摘要数据
     */
    public static byte[] digestByYunhsmWithPublicKey(ECPublicKey publicKey, byte[] data) throws SdfSDKException {
        return digestBySdfWithPublicKey(SdfCryptoType.YUNHSM, publicKey, data);
    }

    /**
     * 使用 PCIE卡 进行SM3摘要计算
     *
     * @param publicKey 签名值公钥
     * @param data      原始数据
     * @return 摘要数据
     */
    public static byte[] digestByPcieWithPublicKey(ECPublicKey publicKey, byte[] data) throws SdfSDKException {
        return digestBySdfWithPublicKey(SdfCryptoType.PCIE, publicKey, data);
    }

    /**
     * 使用 SDF模式 进行SM3摘要计算
     *
     * @param sdfCryptoType 使用的加密模式
     * @param data          原始数据
     * @param publicKey     签名值公钥
     * @return 摘要数据
     */
    public static byte[] digestBySdfWithPublicKey(SdfCryptoType sdfCryptoType, ECPublicKey publicKey, byte[] data) throws SdfSDKException {
        SdfECCPublicKey sdfECCPublicKey = SdfECCPublicKey.getInstance(publicKey);
        //userID 固定 为 1234567812345678
        byte[] pucID = Strings.toByteArray("1234567812345678");
        SdfSM3Digest sm3Digest = new SdfSM3Digest(sdfCryptoType, pucID, sdfECCPublicKey);
        sm3Digest.update(data, 0, data.length);
        byte[] out = new byte[sm3Digest.getDigestSize()];
        sm3Digest.doFinal(out, 0);
        return out;
    }
}
