package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfRSAKey;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPrivateKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPublicKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRsaKeyPair;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDK;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.BigIntegers;

import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPublicKeySpec;

public class GMSSLRSAKeyUtils {

    public static final String RSA_ALGORITHM_NAME = "RSA";
    public static final int RSA_KEY_SIZE_1024 = 1024;
    public static final int RSA_KEY_SIZE_2048 = 2048;

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                      密钥对相关                                  *
     ************************************************************************************/
    /**
     * 生成 RSA 密钥对
     *
     * @param keySize 密钥长度
     * @return RSA 密钥对
     * @throws NoSuchProviderException            找不到相应的 Provider
     * @throws NoSuchAlgorithmException           找不到相应的算法
     * @throws InvalidAlgorithmParameterException KeyPairGenerator 初始化异常
     */
    public static KeyPair generateKeyPairByBC(int keySize) throws NoSuchProviderException, NoSuchAlgorithmException {
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance(RSA_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
        kpGen.initialize(keySize);
        return kpGen.generateKeyPair();
    }

    /**
     * 生成 RSA 密钥对
     *
     * @param keySize 密钥长度
     * @return RSA 密钥对
     * @throws NoSuchProviderException  找不到相应的 Provider
     * @throws NoSuchAlgorithmException 找不到相应的算法
     * @throws InvalidKeySpecException  构建公钥或私钥失败
     * @throws SdfSDKException
     */
    public static KeyPair generateKeyPairByYunHsm(int keySize) throws SdfSDKException, InvalidKeySpecException, NoSuchProviderException, NoSuchAlgorithmException {
        YunhsmSdfSDK yunhsmSdfSDK = new YunhsmSdfSDK();
        SdfRsaKeyPair sdfRsaKeyPair = yunhsmSdfSDK.generateKeyPairRsa(keySize);
        SdfRSAPrivateKey rsaPrivateKey = sdfRsaKeyPair.getSdfRSAPrivateKey();
        RSAPrivateCrtKeySpec spec = new RSAPrivateCrtKeySpec(BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getM()),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getE()),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getD()),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPrime()[0]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPrime()[1]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPexp()[0]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPexp()[1]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getCoef()));
        KeyFactory keyFactory = KeyFactory.getInstance(RSA_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
        PrivateKey privateKey = keyFactory.generatePrivate(spec);
        SdfRSAPublicKey rsaPublicKey = sdfRsaKeyPair.getSdfRSAPublicKey();
        RSAPublicKeySpec spec1 = new RSAPublicKeySpec(BigIntegers.fromUnsignedByteArray(rsaPublicKey.getM()),
                BigIntegers.fromUnsignedByteArray(rsaPublicKey.getE()));//存储的就是这两个大整形数
        PublicKey pk = keyFactory.generatePublic(spec1);
        return new KeyPair(pk, privateKey);
    }

    /**
     * 生成 SDF 相关的 RSA 私钥
     *
     * @param index    密钥访问索引
     * @param password 密钥访问控制密码
     * @return SDF 相关的 RSA私钥
     */
    public static SdfPrivateKey genSdfPrivateKey(int index, String password) {
        return new SdfPrivateKey(index, password.getBytes());
    }

    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByYunhsm(int index) throws SdfSDKException {
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfRSAKey.exportSignPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByPcie(int index) throws SdfSDKException {
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfRSAKey.exportSignPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByYunhsm(int index) throws SdfSDKException {
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfRSAKey.exportEncryptPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByPcie(int index) throws SdfSDKException {
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfRSAKey.exportEncryptPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  PCIE卡
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromPCIE(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromSdf(SdfCryptoType sdfCryptoType, int index, String password) {
        try {
            SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
            sdfSDK.init();
            boolean result = sdfSDK.checkPrivateKeyAccessRight(index, password.getBytes());
            sdfSDK.release();
            return result;
        } catch (SdfSDKException e) {
            return false;
        }
    }

}
