package com.xdja.pki.gmssl.operator.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentSigner;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

import java.security.PrivateKey;
import java.security.Security;

public class GMSSLContentSignerUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                             ContentSigner 生成                                   *
     ************************************************************************************/

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param signatureAlgorithm 签名算法 GMSSLSignatureAlgorithm 枚举
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByBC(GMSSLSignatureAlgorithm signatureAlgorithm, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerByBC(signatureAlgorithm.getSigAlgName(), privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param signAlgName 签名算法 GMSSLSignatureAlgorithm 枚举
     * @param privateKey  签发者私钥
     * @return ContentSigner 内容签发体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByBC(String signAlgName, PrivateKey privateKey) throws OperatorCreationException {
        JcaContentSignerBuilder contentSignerBuilder = new JcaContentSignerBuilder(signAlgName);
        return contentSignerBuilder.build(privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByYunhsm(String signAlgName, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.YUNHSM, signAlgName, privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByPcie(String signAlgName, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.PCIE,  signAlgName, privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerBySDF(SdfCryptoType sdfCryptoType, String signAlgName, PrivateKey privateKey) throws OperatorCreationException {
        if (!(privateKey instanceof SdfPrivateKey)) {
            throw new OperatorCreationException(" un support private key type " + privateKey.getClass());
        } else {
            return new SdfContentSigner(sdfCryptoType, (SdfPrivateKey) privateKey);
        }
    }
}
