package com.xdja.pki.gmssl.sdf.yunhsm.utils;


import com.alibaba.fastjson.JSONObject;
import com.xdja.pki.gmssl.core.utils.GMSSLHttpUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.yunhsm.HsmConnectionPool;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDK;
import com.xdja.pki.gmssl.x509.utils.GMSSLX500NameUtils;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmExceptionEnum;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmInfoEntry;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.security.KeyStore;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.List;

public class GMSSLYunHsmUtils {
    private static Logger logger = LoggerFactory.getLogger(GMSSLYunHsmUtils.class);

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 配置密码机 无需输入密码机地址 会存储 配置文件
     *
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxPath     签名证书地址    （pfx格式证书）
     * @param encPfxPath      加密证书地址    （pfx格式证书）
     * @param trustP7bPath    密码机CA证书链  （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(
            String ip, int port, String signPfxPassword, String encPfxPassword,
            String signPfxPath, String encPfxPath, String trustP7bPath
    ) throws Exception {
        String yunHsmPath = getYunHsmPath();
        return initYunHsmConfigAndTestConnect(yunHsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxPath, encPfxPath, trustP7bPath);
    }

    /**
     * 配置密码机 会存储 配置文件
     * <p>
     * 需要传入密码机配置文件路径
     *
     * @param yunhsmPath      密码机安装路径
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxPath     签名证书地址    （pfx格式证书）
     * @param encPfxPath      加密证书地址    （pfx格式证书）
     * @param trustP7bPath    密码机CA证书链  （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(
            String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword,
            String signPfxPath, String encPfxPath, String trustP7bPath
    ) throws Exception {
        File signPfx = new File(signPfxPath);
        FileInputStream signPfxStream = new FileInputStream(signPfx);

        File encPfx = new File(encPfxPath);
        FileInputStream encPfxStream = new FileInputStream(encPfx);

        File trustP7b = new File(trustP7bPath);
        FileInputStream trustP7bStream = new FileInputStream(trustP7b);

        return initYunHsmConfigAndTestConnect(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
    }

    /**
     * 配置密码机 无需输入密码机地址 会存储 配置文件
     *
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxStream   签名证书文件流        （pfx格式证书）
     * @param encPfxStream    加密证书文件流        （pfx格式证书）
     * @param trustP7bStream  密码机CA证书链文件流   （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(
            String ip, int port, String signPfxPassword, String encPfxPassword,
            FileInputStream signPfxStream, FileInputStream encPfxStream, FileInputStream trustP7bStream
    ) throws Exception {
        String yunHsmPath = getYunHsmPath();
        return initYunHsmConfigAndTestConnect(yunHsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
    }

    /**
     * 初始化配置密码机并测试链接
     * <p>
     * 1. 测试链接
     * 2. 配置文件存储
     *
     * @param yunhsmPath      密码机安装路径
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxStream   签名证书文件流    （pfx格式证书）
     * @param encPfxStream    加密证书文件流    （pfx格式证书）
     * @param trustP7bStream  密码机CA证书链文件流  （p7b格式证书链）
     */
    public static YunHsmExceptionEnum initYunHsmConfigAndTestConnect(
            String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword,
            FileInputStream signPfxStream, FileInputStream encPfxStream, FileInputStream trustP7bStream
    ) throws Exception {
        InputStream signTestInput = null;
        InputStream encTestInput = null;
        InputStream trustTestInput = null;
        InputStream signInput = null;
        InputStream encInput = null;
        InputStream trustInput = null;

        if (signPfxStream != null){
            ByteArrayOutputStream signByteOut = new ByteArrayOutputStream();
            IOUtils.copy(signPfxStream, signByteOut);
            signTestInput = new ByteArrayInputStream(signByteOut.toByteArray());
            signInput = new ByteArrayInputStream(signByteOut.toByteArray());
        }

        if (encPfxStream != null){
            ByteArrayOutputStream encByteOut = new ByteArrayOutputStream();
            IOUtils.copy(encPfxStream, encByteOut);
            encTestInput = new ByteArrayInputStream(encByteOut.toByteArray());
            encInput = new ByteArrayInputStream(encByteOut.toByteArray());
        }

        if (trustP7bStream != null){
            ByteArrayOutputStream trustByteOut = new ByteArrayOutputStream();
            IOUtils.copy(trustP7bStream, trustByteOut);
            trustTestInput = new ByteArrayInputStream(trustByteOut.toByteArray());
            trustInput = new ByteArrayInputStream(trustByteOut.toByteArray());
        }


        // 写入配置文件 到 confTest 然后测试链接
        YunHsmExceptionEnum yunHsmExceptionEnum = testConnect(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, signTestInput, encTestInput, trustTestInput);
        if (yunHsmExceptionEnum != YunHsmExceptionEnum.NORMAL) {
            return yunHsmExceptionEnum;
        }

        String confPath = yunhsmPath + File.separator + "conf";

        // 写入新配置文件
        writeConf(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, confPath);

        YunHsmExceptionEnum checkCert = storeCert(signPfxPassword, encPfxPassword, signInput, encInput, trustInput, yunhsmPath, confPath);
        if (checkCert != YunHsmExceptionEnum.NORMAL) {
            logger.info("配置文件校验失败 {}", checkCert);
            return checkCert;
        }

        // 关闭密码机当前连接 如果要是有
        HsmConnectionPool instance = HsmConnectionPool.getInstance();
        if (instance != null) {
            logger.debug("关闭密码机当前连接");
            instance.releaseAllConnection();
        }

        instance = HsmConnectionPool.getInstance();
        if (instance != null) {
            logger.debug("重新打开密码机链接");
            instance.initConnectionPool();
        }
        logger.info("初始化配置文件成功");
        return YunHsmExceptionEnum.NORMAL;
    }


    public static YunHsmExceptionEnum testConnect(
            String ip, int port, String signPfxPassword, String encPfxPassword,
            FileInputStream signPfxStream, FileInputStream encPfxStream, FileInputStream trustP7bStream
    ) throws Exception {
        String yunhsmPath = getYunHsmPath();
        return testConnect(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
    }

    public static YunHsmExceptionEnum testConnect(
            String ip, int port, String signPfxPassword, String encPfxPassword,
            String signPfxPath, String encPfxPath, String trustP7bPath
    ) throws Exception {
        String yunhsmPath = getYunHsmPath();
        return testConnect(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxPath, encPfxPath, trustP7bPath);
    }

    public static YunHsmExceptionEnum testConnect(
            String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword,
            String signPfxPath, String encPfxPath, String trustP7bPath
    ) throws Exception {
        File signPfx = new File(signPfxPath);
        FileInputStream signPfxStream = new FileInputStream(signPfx);

        File encPfx = new File(encPfxPath);
        FileInputStream encPfxStream = new FileInputStream(encPfx);

        File trustP7b = new File(trustP7bPath);
        FileInputStream trustP7bStream = new FileInputStream(trustP7b);
        return testConnect(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream);
    }

    /**
     * 测试链接 支持传入配置文件
     *
     * @param yunhsmPath      密码机配置文件路径
     * @param ip              密码机服务IP
     * @param port            密码机服务Port
     * @param signPfxPassword 密码机服务 签名pfx 密码
     * @param encPfxPassword  密码机服务 加密pfx 密码
     * @param signPfxStream   密码机服务 签名pfx 文件流
     * @param encPfxStream    密码机服务 加密pfx 文件流
     * @param trustP7bStream  密码机服务 信任链 p7b 文件流
     * @return 链接是否正常 错误返回错误码
     * @throws Exception 异常信息
     */
    public static YunHsmExceptionEnum testConnect(
            String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword,
            InputStream signPfxStream, InputStream encPfxStream, InputStream trustP7bStream
    ) throws Exception {
        // 将传入的配置文件写入到 confTest
        // 验证并存储 签名证书、加密证书、信任证书链
        String confTest = yunhsmPath + File.separator + "confTest" + File.separator;
        YunHsmExceptionEnum checkCert = storeCert(signPfxPassword, encPfxPassword, signPfxStream, encPfxStream, trustP7bStream, yunhsmPath, confTest);
        if (checkCert != YunHsmExceptionEnum.NORMAL) {
            logger.info("配置文件校验失败 {}", checkCert);
            return checkCert;
        }

        writeConf(yunhsmPath, ip, port, signPfxPassword, encPfxPassword, confTest);

        // 测试链接 confTest
        return testConnectWithConf(ip, port, confTest);
    }


    /**
     * 测试链接 传入 ip port
     *
     * @param host IP 地址
     * @param port 端口号
     * @return 链接状态
     */
    public static YunHsmExceptionEnum testConnect(String host, int port) {
        try {
            logger.debug("开始测试链接 配置文件路径 host={} port={}", host, port);
            if (!GMSSLHttpUtils.isHostConnectivity(host, port)) {
                return YunHsmExceptionEnum.TELNET_PORT_FAILURE;
            }
            YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();
            sdfSDK.init();
            sdfSDK.testConnect();
            sdfSDK.release();
            logger.info("测试链接成功 host={} port={}", host, port);
        } catch (SdfSDKException e) {
            logger.error("测试链接失败，打开设备失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
        return YunHsmExceptionEnum.NORMAL;
    }

    /**
     * 测试链接 传入 ip port
     *
     * @param host     IP 地址
     * @param port     端口号
     * @param confPath 配置文件路径
     * @return 链接状态
     */
    public static YunHsmExceptionEnum testConnectWithConf(String host, int port, String confPath) {
        confPath = confPath + File.separator + "yunhsmsdk.conf";
        try {
            logger.debug("开始测试链接 配置文件路径 host={} port={} confPath={}", host, port, confPath);
            if (!GMSSLHttpUtils.isHostConnectivity(host, port)) {
                return YunHsmExceptionEnum.TELNET_PORT_FAILURE;
            }
            YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();
            sdfSDK.init();
            sdfSDK.testConnect(confPath);
            sdfSDK.release();
            logger.info("测试链接成功 host={} port={} confPath={}", host, port, confPath);
        } catch (SdfSDKException e) {
            logger.error("测试链接失败，打开设备失败，host={} port={} confPath={}", host, port, confPath, e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
        return YunHsmExceptionEnum.NORMAL;
    }


    /**
     * 测试链接
     *
     * @return 是否联通
     */
    public static boolean testConnect() {
        try {
            YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();
            sdfSDK.init();
            sdfSDK.testConnect();
            sdfSDK.release();
            logger.info("测试链接成功");
            return true;
        } catch (SdfSDKException e) {
            logger.error("测试链接失败，打开设备失败", e);
            return false;
        }
    }

    /**
     * 密码机恢复初始化 无需输入密码机路径
     *
     * @throws IOException 读写文件时产生的异常
     */
    public static void resetYunHsm() throws IOException {
        String yunHsmPath = getYunHsmPath();
        resetYunHsm(yunHsmPath);
    }

    /**
     * 密码机恢复初始化
     *
     * @param yunhsmPath 密码机安装路径
     * @throws IOException 读写文件时产生的异常
     */
    public static void resetYunHsm(String yunhsmPath) throws IOException {
        String bakPath = yunhsmPath + File.separator + "confBak" + File.separator;
        String sourcePath = yunhsmPath + File.separator + "conf" + File.separator;
        File bakfile = new File(bakPath);
        if (!bakfile.exists()) {
            logger.info("密码机未进行任何配置，无需初始化");
            return;
        }
        GMSSLX509Utils.deleteDirectory(sourcePath);
        File file = new File(sourcePath);
        file.mkdir();
        GMSSLX509Utils.copyDir(bakPath, sourcePath);
        GMSSLX509Utils.deleteDirectory(bakPath);
        logger.info("密码机初始化成功");
    }

    /**
     * 获取配置信息 不需要传路径
     *
     * @throws IOException 读写配置文件产生的异常
     */
    public static YunHsmInfoEntry getYunHsmInfo() throws IOException {
        String yunHsmPath = getYunHsmPath();
        return getYunHsmInfo(yunHsmPath);
    }

    /**
     * 根据密码机位置获取配置信息
     *
     * @throws IOException 读写配置文件产生的异常
     */
    public static YunHsmInfoEntry getYunHsmInfo(String yunhsmPath) throws IOException {
        String confPath = yunhsmPath + File.separator + "conf" + File.separator + "yunhsmsdk.conf";
        YunHsmInfoEntry yunHsmInfoEntry = new YunHsmInfoEntry();
        File file = new File(confPath);
        String jsonString = FileUtils.readFileToString(file, "UTF-8");
        //总目录
        JSONObject dataJson = JSONObject.parseObject(jsonString);
        //Certificate目录下
        JSONObject certificate = dataJson.getJSONObject("Certificate");
        //SignatureCertificate
        JSONObject softCert = certificate.getJSONObject("SoftCert");
        JSONObject signatureCertificate = softCert.getJSONObject("SignatureCertificate");

        yunHsmInfoEntry.setSignCertPassword(signatureCertificate.getString("pin"));
        String signCertName = signatureCertificate.getString("file");
        yunHsmInfoEntry.setSignCertName(signCertName.substring(signCertName.indexOf("sign")));

        //EncryptCertificate
        JSONObject encryptCertificate = softCert.getJSONObject("EncryptCertificate");
        yunHsmInfoEntry.setEncCertPassword(encryptCertificate.getString("pin"));
        String encCertName = encryptCertificate.getString("file");
        yunHsmInfoEntry.setEncCertName(encCertName.substring(encCertName.indexOf("enc")));

        JSONObject hsm = dataJson.getJSONObject("hsm");
        yunHsmInfoEntry.setServerIp(hsm.getString("ip"));
        yunHsmInfoEntry.setServerPort(Integer.valueOf(hsm.getString("port")));

        JSONObject ssl = dataJson.getJSONObject("ssl");
        String caCertName = ssl.getString("CertificatePath");
        //setCaCertName
        yunHsmInfoEntry.setCaCertName(caCertName.substring(caCertName.indexOf("trust")));

        yunHsmInfoEntry.setYunHsmType("信大捷安服务器密码机");
        return yunHsmInfoEntry;
    }

    private static void deleteConDir(String yunHsmPath) {
        File confDir = new File(yunHsmPath + File.separator + "conf");
        if (confDir.exists()) {
            GMSSLX509Utils.deleteDirectory(yunHsmPath + File.separator + "conf");
        }
        confDir.mkdir();
    }

    private static String getYunHsmPath() {
        String os = System.getProperty("os.name");
        String yunhsmPath;
        if (os.startsWith("Windows")) {
            yunhsmPath = "C:\\Program Files (x86)\\yunhsmsdk";
        } else {
            yunhsmPath = "/usr/local/yunhsmsdk";
        }
        return yunhsmPath;
    }

    /**
     * 备份配置文件
     *
     * @param path 密码机安装路径
     */
    private static void backUpConf(String path) throws IOException {
        String bakPath = path + File.separator + "confBak" + File.separator;
        File bakFile = new File(bakPath);
        String sourcePath = path + File.separator + "conf" + File.separator;
        logger.info("confFile 备份成功");
        if (bakFile.exists() && bakFile.isDirectory()) {
            return;
        }

        GMSSLX509Utils.copyDir(sourcePath, bakPath);
        logger.info("备份成功");
    }

    /**
     * 验证并存储 签名证书、加密证书、信任证书链
     *
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     * @param signPfxStream   签名证书    （pfx格式证书）
     * @param encPfxStream    加密证书    （pfx格式证书）
     * @param trustP7bStream  密码机CA证书链  （p7b格式证书链）
     */
    private static YunHsmExceptionEnum storeCert(
            String signPfxPassword, String encPfxPassword,
            InputStream signPfxStream, InputStream encPfxStream,
            InputStream trustP7bStream, String yunHsmPath, String writeConfPath
    ) throws Exception {
        YunHsmInfoEntry yunHsmInfo = GMSSLYunHsmUtils.getYunHsmInfo(yunHsmPath);
        if (signPfxStream == null) {
            String signCertName = yunHsmInfo.getSignCertName();
            File signPfx = new File(yunHsmPath + File.separator + "conf" + File.separator + signCertName);
            signPfxStream = new FileInputStream(signPfx);
        }
        if (encPfxStream == null) {
            String encCertName = yunHsmInfo.getEncCertName();
            File signPfx = new File(yunHsmPath + File.separator + "conf" + File.separator + encCertName);
            encPfxStream = new FileInputStream(signPfx);
        }
        if (trustP7bStream == null) {
            String caCertName = yunHsmInfo.getCaCertName();
            File trustP7b = new File(yunHsmPath + File.separator + "conf" + File.separator + caCertName);
            trustP7bStream = new FileInputStream(trustP7b);
        }
        ByteArrayOutputStream signByteOut = new ByteArrayOutputStream();
        IOUtils.copy(signPfxStream, signByteOut);
        ByteArrayOutputStream encByteOut = new ByteArrayOutputStream();
        IOUtils.copy(encPfxStream, encByteOut);
        ByteArrayOutputStream trustByteOut = new ByteArrayOutputStream();
        IOUtils.copy(trustP7bStream, trustByteOut);

        //验证 签名 pfx 密码
        KeyStore signKse;
        try {

            InputStream si = new ByteArrayInputStream(signByteOut.toByteArray());
            signKse = KeyStore.getInstance("pkcs12", BouncyCastleProvider.PROVIDER_NAME);
            signKse.load(si, signPfxPassword.toCharArray());
            logger.info("签名PFX 解析成功");
        } catch (Exception e) {
            logger.error("签名证书保护口令不能打开加密证书", e);
            return YunHsmExceptionEnum.SIGN_PASSWORD_IS_ERROR;
        }

        //验证 加密 pfx 密码
        KeyStore encKse;
        try {
            InputStream ei = new ByteArrayInputStream(encByteOut.toByteArray());
            encKse = KeyStore.getInstance("pkcs12", BouncyCastleProvider.PROVIDER_NAME);
            encKse.load(ei, encPfxPassword.toCharArray());
            logger.info("加密PFX 解析成功");
        } catch (Exception e) {
            logger.error("加密证书保护口令不能打开加密证书", e);
            return YunHsmExceptionEnum.ENC_PASSWORD_IS_ERROR;
        }

        //获取p7b中的证书
        List<X509Certificate> trustList;
        try {
            InputStream ti = new ByteArrayInputStream(trustByteOut.toByteArray());
            trustList = GMSSLX509Utils.getCertsByCertChain(ti);
        } catch (Exception e) {
            logger.error("解析密码机CA证书链失败", e);
            return YunHsmExceptionEnum.OPEN_TRAIN_CERT_P7b_IS_ERROR;
        }

        // 验证签名证书
        InputStream si = new ByteArrayInputStream(signByteOut.toByteArray());
        List<X509Certificate> signList = GMSSLX509Utils.readCertificatesFromP12(si, signPfxPassword.toCharArray());
        for (int i = 0; i < signList.size(); i++) {
            X509Certificate certificate = signList.get(i);
            // 校验签名证书用途
            boolean[] signKeyUsage = new boolean[]{true, true, false, false, false, false, false, false, false};
            boolean[] keyUsage = certificate.getKeyUsage();
            if (!Arrays.equals(signKeyUsage, keyUsage)){
                logger.error("签名证书密钥用途错误，应为 {}, 配置文件为 {}", signKeyUsage, keyUsage);
                return YunHsmExceptionEnum.SIGN_CERT_KEY_USAGE_IS_ERROR;
            }
            if (!GMSSLX509Utils.verifyCert(certificate, trustList)) {
                logger.error("第 {} 个签名证书 {} 验签失败", i, GMSSLX500NameUtils.getRFCStyleSubjectDN(certificate));
                return YunHsmExceptionEnum.SIGN_CERT_VERIFY_IS_ERROR;
            }
        }
        logger.info("签名证书验签全部通过");

        // 验证加密证书
        InputStream ei = new ByteArrayInputStream(encByteOut.toByteArray());
        List<X509Certificate> encList = GMSSLX509Utils.readCertificatesFromP12(ei, encPfxPassword.toCharArray());
        for (int i = 0; i < encList.size(); i++) {
            X509Certificate certificate = encList.get(i);
            // 校验加密证书用途
            boolean[] signKeyUsage = new boolean[]{false, false, true, true, true, false, false, false, false};
            boolean[] keyUsage = certificate.getKeyUsage();
            if (!Arrays.equals(signKeyUsage, keyUsage)){
                logger.error("加密证书密钥用途错误，应为 {}, 配置文件为 {}", signKeyUsage, keyUsage);
                return YunHsmExceptionEnum.ENC_CERT_KEY_USAGE_IS_ERROR;
            }
            if (!GMSSLX509Utils.verifyCert(encList.get(i), trustList)) {
                logger.error("第 {} 个加密证书 {} 验签失败", i, GMSSLX500NameUtils.getRFCStyleSubjectDN(certificate));
                return YunHsmExceptionEnum.ENC_CERT_VERIFY_IS_ERROR;
            }
        }
        logger.info("加密证书验签全部通过");

        //指定目录存储 相关文件
        File cp = new File(writeConfPath);
        if (!cp.exists()){
            cp.mkdirs();
        }
        String sign = "sign_" + signPfxPassword;
        String enc = "enc_" + encPfxPassword;
        String trust = "trust_chain.p7b";
        GMSSLKeyStoreUtils.saveGMSSLPfx(signKse, signPfxPassword, writeConfPath, sign);
        GMSSLKeyStoreUtils.saveGMSSLPfx(encKse, encPfxPassword, writeConfPath, enc);
        InputStream ti = new ByteArrayInputStream(trustByteOut.toByteArray());
        GMSSLX509Utils.copyFile(ti, writeConfPath + File.separator + trust);

        logger.info("签名PFX {}、加密PFX {}、信任证书链 {} 写入 {} 成功", sign, enc, trust, writeConfPath);
        return YunHsmExceptionEnum.NORMAL;
    }


    /**
     * 备份当前配置文件 写入新的配置文件
     *
     * @param yunhsmPath      密码机安装路径
     * @param ip              密码机服务ip地址
     * @param port            密码机服务端口
     * @param signPfxPassword 签名证书保护口令
     * @param encPfxPassword  加密证书保护口令
     */
    private static void writeConf(
            String yunhsmPath, String ip, int port, String signPfxPassword, String encPfxPassword, String writePath
    ) throws IOException {
        //备份当前密码机配置文件
        backUpConf(yunhsmPath);

        String os = System.getProperty("os.name");
        String resourcePath = yunhsmPath + File.separator + "confBak" + File.separator + "yunhsmsdk.conf";
        String confPath = writePath + File.separator + "yunhsmsdk.conf";

        BufferedReader br = new BufferedReader(new FileReader(resourcePath));// 读取原始json文件

        File file = new File(resourcePath);
        String jsonString = FileUtils.readFileToString(file, "UTF-8");
        //总目录
        JSONObject dataJson = JSONObject.parseObject(jsonString);
        //Certificate目录下
        JSONObject certificate = dataJson.getJSONObject("Certificate");
        //SignatureCertificate
        JSONObject softCert = certificate.getJSONObject("SoftCert");
        JSONObject signatureCertificate = softCert.getJSONObject("SignatureCertificate");

        signatureCertificate.put("pin", signPfxPassword);
        //EncryptCertificate
        JSONObject encryptCertificate = softCert.getJSONObject("EncryptCertificate");

        encryptCertificate.put("pin", encPfxPassword);

        JSONObject hsm = dataJson.getJSONObject("hsm");
        hsm.put("ip", ip);
        hsm.put("port", port);
        JSONObject ssl = dataJson.getJSONObject("ssl");


        if (os.startsWith("Windows")) {
            signatureCertificate.put("file", writePath + "\\sign_" + signPfxPassword + ".pfx");
            encryptCertificate.put("file", writePath + "\\enc_" + encPfxPassword + ".pfx");
            ssl.put("CertificatePath", writePath + "\\trust_chain.p7b");
        } else {
            //  if (os.equals("Linux")) {
            signatureCertificate.put("file", writePath + "/sign_" + signPfxPassword + ".pfx");
            encryptCertificate.put("file", writePath + "/enc_" + encPfxPassword + ".pfx");
            ssl.put("CertificatePath", writePath + "/trust_chain.p7b");
        }

        String ws = dataJson.toString();
        BufferedWriter bw = new BufferedWriter(new FileWriter(confPath));// 输出新的json文件
        String s = formatJson(ws);
        bw.write(s);
        bw.flush();
        br.close();
        bw.close();
        logger.info("配置文件写入 {} 成功", writePath);
    }

    private static String formatJson(String jsonStr) {
        if (null == jsonStr || "".equals(jsonStr)) return "";
        StringBuilder sb = new StringBuilder();
        char last = '\0';
        char current = '\0';
        int indent = 0;
        for (int i = 0; i < jsonStr.length(); i++) {
            last = current;
            current = jsonStr.charAt(i);
            switch (current) {
                case '{':
                case '[':
                    sb.append(current);
                    sb.append('\n');
                    indent++;
                    addIndentBlank(sb, indent);
                    break;
                case '}':
                case ']':
                    sb.append('\n');
                    indent--;
                    addIndentBlank(sb, indent);
                    sb.append(current);
                    break;
                case ',':
                    sb.append(current);
                    if (last != '\\') {
                        sb.append('\n');
                        addIndentBlank(sb, indent);
                    }
                    break;
                default:
                    sb.append(current);
            }
        }

        return sb.toString();
    }

    private static void addIndentBlank(StringBuilder sb, int indent) {
        for (int i = 0; i < indent; i++) {
            sb.append('\t');
        }
    }
}
