package com.xdja.pki.gmssl.tomcat.utils;


import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.keystore.utils.GMSSLKeyStoreUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

public class GMSSLTomcatUtils {
    private static Logger logger = LoggerFactory.getLogger(GMSSLTomcatUtils.class);

    /**
     * 服务器初始化tomcat调用接口
     *
     * @param trustCert        信任证书
     * @param serveSignCert    服务器签名证书
     * @param serverEncCert    服务器加密证书
     * @param serverSignPriKey 服务器签名私钥
     * @param serverEncPriKey  服务器加密私钥
     * @param tomcatPath       tomcat绝对路径
     * @param port             需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByBC(X509Certificate trustCert, X509Certificate serveSignCert, X509Certificate serverEncCert,
                                         PrivateKey serverSignPriKey, PrivateKey serverEncPriKey, String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        saveKeyStore(trustCert, serveSignCert, serverEncCert, serverSignPriKey, serverEncPriKey, tomcatPath, String.valueOf(port));
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLV1.1", port);

    }


    /**
     * 服务器初始化tomcat调用接口
     *
     * @param trustCert        信任证书
     * @param serveSignCert    服务器签名证书
     * @param serverEncCert    服务器加密证书
     * @param serverSignPriKey 服务器签名私钥
     * @param serverEncPriKey  服务器加密私钥
     * @param tomcatPath       tomcat绝对路径
     * @param port             需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByBC(List<X509Certificate> trustCert, X509Certificate serveSignCert, X509Certificate serverEncCert,
                                         PrivateKey serverSignPriKey, PrivateKey serverEncPriKey, String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        saveKeyStore(trustCert, serveSignCert, serverEncCert, serverSignPriKey, serverEncPriKey, tomcatPath, String.valueOf(port));
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLV1.1", port);

    }

    /**
     * 服务器初始化tomcat调用接口
     *
     * @param trustCert                  信任证书
     * @param serveSignCert              服务器签名证书
     * @param serverEncCert              服务器加密证书
     * @param caServerPrivateKeyIndex    服务器证书密钥控制索引
     * @param caServerPrivateKeyPassword 服务器证书密钥控制访问码
     * @param tomcatPath                 tomcat绝对路径
     * @param port                       需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByYunHsm(X509Certificate trustCert, X509Certificate serveSignCert, X509Certificate serverEncCert,
                                             int caServerPrivateKeyIndex, String caServerPrivateKeyPassword,
                                             String tomcatPath, int port) throws Exception {

        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(caServerPrivateKeyIndex, caServerPrivateKeyPassword.getBytes());
        saveKeyStore(trustCert, serveSignCert, serverEncCert, sdfPrivateKey, sdfPrivateKey, tomcatPath, String.valueOf(port));
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLSDFYUNHSMV1.1", port);
    }

    /**
     * 服务器初始化tomcat调用接口
     *
     * @param trustCerts                 信任证书
     * @param serveSignCert              服务器签名证书
     * @param serverEncCert              服务器加密证书
     * @param caServerPrivateKeyIndex    服务器证书密钥控制索引
     * @param caServerPrivateKeyPassword 服务器证书密钥控制访问码
     * @param tomcatPath                 tomcat绝对路径
     * @param port                       需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByYunHsm(List<X509Certificate> trustCerts, X509Certificate serveSignCert, X509Certificate serverEncCert,
                                             int caServerPrivateKeyIndex, String caServerPrivateKeyPassword,
                                             String tomcatPath, int port) throws Exception {

        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(caServerPrivateKeyIndex, caServerPrivateKeyPassword.getBytes());
        saveKeyStore(trustCerts, serveSignCert, serverEncCert, sdfPrivateKey, sdfPrivateKey, tomcatPath, String.valueOf(port));
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLSDFYUNHSMV1.1", port);
    }

    /**
     * 服务器初始化tomcat调用接口
     *
     * @param trustCert                  信任证书
     * @param serveSignCert              服务器签名证书
     * @param serverEncCert              服务器加密证书
     * @param caServerPrivateKeyIndex    服务器证书密钥控制索引
     * @param caServerPrivateKeyPassword 服务器证书密钥控制访问码
     * @param tomcatPath                 tomcat绝对路径
     * @param port                       需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByPCIE(X509Certificate trustCert, X509Certificate serveSignCert, X509Certificate serverEncCert,
                                           int caServerPrivateKeyIndex, String caServerPrivateKeyPassword,
                                           String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(caServerPrivateKeyIndex, caServerPrivateKeyPassword.getBytes());
        saveKeyStore(trustCert, serveSignCert, serverEncCert, sdfPrivateKey, sdfPrivateKey, tomcatPath, String.valueOf(port));
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLSDFPCIEV1.1", port);
    }

    /**
     * 服务器初始化tomcat调用接口
     *
     * @param trustCerts                 信任证书
     * @param serveSignCert              服务器签名证书
     * @param serverEncCert              服务器加密证书
     * @param caServerPrivateKeyIndex    服务器证书密钥控制索引
     * @param caServerPrivateKeyPassword 服务器证书密钥控制访问码
     * @param tomcatPath                 tomcat绝对路径
     * @param port                       需要开启的GMSSL通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpsPortByPCIE(List<X509Certificate> trustCerts, X509Certificate serveSignCert, X509Certificate serverEncCert,
                                           int caServerPrivateKeyIndex, String caServerPrivateKeyPassword,
                                           String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(caServerPrivateKeyIndex, caServerPrivateKeyPassword.getBytes());
        saveKeyStore(trustCerts, serveSignCert, serverEncCert, sdfPrivateKey, sdfPrivateKey, tomcatPath, String.valueOf(port));
        serverXmlResolver.addTomcatHttpsPort(tomcatPath, "GMSSLSDFPCIEV1.1", port);
    }

    /**
     * 开tomcat  http 服务端口
     *
     * @param tomcatPath tomcat绝对路径
     * @param port       需要Http服务通道端口号k
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void openHttpPort(String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.addTomcatHttpPort(tomcatPath, port);
    }

    /**
     * 关闭tomcat服务端口 http和https端口公用一个方法
     *
     * @param tomcatPath tomcat绝对路径
     * @param port       需要关闭的通道端口号
     * @throws Exception 生成KeyStore产生的各种异常以及读写证书和操作XML文件时产生的各种异常
     */
    public static void closeTomcatPort(String tomcatPath, int port) throws Exception {
        ServerXmlResolver serverXmlResolver = new ServerXmlResolver();
        serverXmlResolver.closeTomcatPort(tomcatPath, port);
    }

    public static boolean restartTomcat(String tomcatPath) {
        if (!shutdownTomcat(tomcatPath)) {
            return false;
        }
        return startupTomcat(tomcatPath);
    }

    public static boolean restartTomcat(String tomcatPath, String shutdownShellPath) {
        if (!run(tomcatPath, "shutdown.sh " + tomcatPath, true, shutdownShellPath)) {
            return false;
        }
        return startupTomcat(tomcatPath);
    }


    public static boolean shutdownTomcat(String tomcatPath) {
        return run(tomcatPath, "shutdown", false);
    }

    public static boolean startupTomcat(String tomcatPath) {
        return run(tomcatPath, "startup", false);
    }


    private static void saveKeyStore(X509Certificate trustCert, X509Certificate serveSignCert, X509Certificate serverEncCert, PrivateKey serverSignPriKey, PrivateKey serverEncPriKey, String path, String port) throws Exception {
        List<X509Certificate> certs = new ArrayList<>();
        certs.add(trustCert);
        saveKeyStore(certs, serveSignCert, serverEncCert, serverSignPriKey, serverEncPriKey, path, port);
    }

    private static void saveKeyStore(List<X509Certificate> trustCerts, X509Certificate serveSignCert, X509Certificate serverEncCert, PrivateKey serverSignPriKey, PrivateKey serverEncPriKey, String path, String port) throws Exception {

        path = path + File.separator + "conf" + File.separator + "cert" + File.separator + port + File.separator;
        File pathf = new File(path);
        if (!pathf.exists()) {
            pathf.mkdirs();
        }
        String password = "password";
        String signAlias = "sign";
        String encAlias = "enc";
        KeyStore serverKeyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStore(password, trustCerts.get(0),
                signAlias, serverSignPriKey, serveSignCert,
                encAlias, serverEncPriKey, serverEncCert);
        ;
        for (int i = 0; i < trustCerts.size(); i++) {
            if (GMSSLX509Utils.verifyCert(serveSignCert, trustCerts.get(i).getPublicKey())) {
                serverKeyStore = GMSSLKeyStoreUtils.generateGMSSLKeyStore(password, trustCerts.get(i),
                        signAlias, serverSignPriKey, serveSignCert,
                        encAlias, serverEncPriKey, serverEncCert);
            }
            GMSSLX509Utils.writeCertificateToPem(path, "ca_" + i, trustCerts.get(i));
            GMSSLX509Utils.writeObjectToPem(path, "ca_public_" + i, trustCerts.get(i).getPublicKey());
            GMSSLX509Utils.writeCertificateToCer(path, "ca_" + i, trustCerts.get(i));
        }

        KeyStore trustKeyStore = GMSSLKeyStoreUtils.generateGMSSLTrustStore(trustCerts);
        GMSSLX509Utils.writeCertificateToPem(path, "sign", serveSignCert);
        GMSSLX509Utils.writeObjectToPem(path, "sign_public", serveSignCert.getPublicKey());
        if (serverSignPriKey instanceof SdfPrivateKey) {
            GMSSLX509Utils.writeSdfPrivateKey(path, "ca_server_key", serverSignPriKey);
//            FileWriter fileWritter = new FileWriter(path + "ca_server_key.pem", false);
//            fileWritter.write(serverSignPriKey.toString());
//            fileWritter.close();
        } else {
            GMSSLX509Utils.writeObjectToPem(path, "sign_private", serverSignPriKey);
            GMSSLX509Utils.writeObjectToPem(path, "enc_private", serverEncPriKey);
            GMSSLX509Utils.writeKeyStoreToP12(serverKeyStore, "password".toCharArray(), path, "sign");
            GMSSLX509Utils.writeKeyStoreToP12(serverKeyStore, "password".toCharArray(), path, "enc");
            GMSSLX509Utils.writeKeyStoreToP12(trustKeyStore, "password".toCharArray(), path, "trust");
        }
        GMSSLX509Utils.writeCertificateToCer(path, "sign", serveSignCert);
        GMSSLX509Utils.writeCertificateToPem(path, "enc", serverEncCert);
        GMSSLX509Utils.writeCertificateToCer(path, "enc", serverEncCert);
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(serverKeyStore, "password", path, "server");
        GMSSLKeyStoreUtils.saveGMSSLKeyStore(trustKeyStore, "password", path, "trust");


    }


    /**
     * 启动Tomcat程序
     *
     * @param tomcatPath
     * @param shName
     */
    private static boolean run(String tomcatPath, String shName, boolean isUseShutdownSh, String... shPath) {
        Runtime rt = Runtime.getRuntime();
        Process ps = null;
        try {
            String os = System.getProperty("os.name");
            if (shName.contains("shut")) {
                logger.info("=================开始关闭Tomcat=================");
            } else {
                logger.info("=================开始启动Tomcat=================");
            }
            if (os.startsWith("Windows")) {
                ps = rt.exec("cmd /c " + tomcatPath + File.separator + "bin" + File.separator + shName + ".bat", null, new File(tomcatPath));
            } else {
                if (isUseShutdownSh) {
                    ps = rt.exec("sh " + shPath[0] + File.separator + shName, null, new File(shPath[0]));
                } else {
                    ps = rt.exec("sh " + tomcatPath + File.separator + "bin" + File.separator + shName + ".sh", null, new File(tomcatPath));
                }

            }
            InputStream is = ps.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            ps.waitFor();
            is.close();
            reader.close();
            ps.destroy();
            logger.info("=================完成=================");
            return true;
        } catch (Exception e) {
            logger.error(shName + " tomcat is error ", e);
            return false;
        }
    }
}