package com.xdja.pki.gmssl.crypto.utils;


import com.xdja.pki.gmssl.core.utils.GMSSLBCCipherUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class GMSSLECIESEncryptUtils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLECIESEncryptUtils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public static byte[] encrypt(GMSSLCryptoType cryptoType, PublicKey publicKey, byte[] data) throws Exception {
        switch (cryptoType) {
            case BC:
                return GMSSLBCCipherUtils.encryptData(GMSSLBCCipherUtils.ECIES, publicKey, data);
            case YUNHSM:
                return encrypt(SdfCryptoType.YUNHSM, publicKey, data);
            case PCIE:
                return encrypt(SdfCryptoType.PCIE, publicKey, data);
            default:
                throw new Exception("un support this crypto" + cryptoType);
        }

    }

    public static String encrypt(GMSSLCryptoType cryptoType, PublicKey publicKey, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(encrypt(cryptoType, publicKey, GMSSLByteArrayUtils.base64Decode(base64Data)));
    }


    public static byte[] decrypt(GMSSLCryptoType cryptoType, PrivateKey privateKey, byte[] cipher) throws Exception {
        if (privateKey instanceof SdfPrivateKey && cryptoType == GMSSLCryptoType.BC) {
            throw new Exception("privateKey is sdfPrivateKey, but crypto type is bc");
        }
        switch (cryptoType) {
            case YUNHSM:
                return decrypt(SdfCryptoType.YUNHSM, privateKey, cipher);
            case BC:
                return GMSSLBCCipherUtils.decryptData(GMSSLBCCipherUtils.ECIES, privateKey, cipher);
            case PCIE:
                return decrypt(SdfCryptoType.PCIE, privateKey, cipher);
            default:
                throw new Exception("un support this crypto" + cryptoType);
        }
    }

    public static String decrypt(GMSSLCryptoType cryptoType, PrivateKey privateKey, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(decrypt(cryptoType, privateKey, GMSSLByteArrayUtils.base64Decode(base64Data)));

    }

    private static byte[] encrypt(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] data) throws Exception {
        SdfECEngine sdfECEngine = new SdfECEngine(sdfCryptoType.getSdfSDK(), GMSSLECUtils.NISTp256);
        sdfECEngine.init(true, new SdfECKeyParameters((ECPublicKey) publicKey));
        byte[] sdfCipher = sdfECEngine.encrypt(data);
        sdfECEngine.release();
        return sdfCipher;
    }

    private static byte[] decrypt(SdfCryptoType sdfCryptoType, PrivateKey privateKey, byte[] cipher) throws Exception {
        SdfECEngine sdfECEngine = new SdfECEngine(sdfCryptoType.getSdfSDK(), GMSSLECUtils.NISTp256);
        sdfECEngine.init(false, new SdfECKeyParameters((SdfPrivateKey) privateKey));
        byte[] data = sdfECEngine.decrypt(cipher);
        sdfECEngine.release();
        return data;
    }

}
