package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.hsm.api.alg.AlgId;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfECKey;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfECCKeyPair;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDK;
import org.bouncycastle.asn1.gm.GMNamedCurves;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.sec.SECObjectIdentifiers;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.*;

/**
 * @ClassName GMSSLECKeyUtils
 * @Description NIST密钥对
 * @Date 2020/1/13 14:05
 * @Author FengZhen
 */
public class GMSSLECKeyUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public static final String SM2p256 = GMNamedCurves.getName(GMObjectIdentifiers.sm2p256v1);
    public static final String NISTp256 = NISTNamedCurves.getName(SECObjectIdentifiers.secp256r1);
    /************************************************************************************
     *                                      密钥对相关                                  *
     ************************************************************************************/
    /**
     * 生成 EC 密钥对
     *
     * @return EC 密钥对
     */
    public static KeyPair generateECKeyPairByBC(String stdName) throws Exception {
        return GMSSLECUtils.generateKeyPair(stdName);
    }

    /**
     * 生成 EC 密钥对
     *
     * @return EC 密钥对
     */
    public static SdfECCKeyPair generateECKeyPairByYunHsm(String stdName) throws Exception {
        YunhsmSdfSDK sdfSDK = new YunhsmSdfSDK();
        sdfSDK.init();
        SdfECCKeyPair sdfECCKeyPair;
        if (stdName.equalsIgnoreCase(GMSSLECUtils.SM2p256)) {
            sdfECCKeyPair = sdfSDK.generateKeyPairEcc(AlgId.SGD_SM2);
        } else if (stdName.equalsIgnoreCase(GMSSLECUtils.NISTp256)) {
            sdfECCKeyPair = sdfSDK.generateKeyPairEcc(AlgId.SGD_ECC_NISTP256);
        } else {
            throw new Exception("unSupport " + stdName + " alg");
        }
        sdfSDK.release();
        return sdfECCKeyPair;
    }

    /**
     * 生成 SDF 相关的 EC 私钥
     *
     * @param index    密钥访问索引
     * @param password 密钥访问控制密码
     * @return SDF 相关的 SM2私钥
     */
    public static SdfPrivateKey genSdfPrivateKey(int index, String password) {
        return new SdfPrivateKey(index, password.getBytes());
    }

    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByYunhsm(int index, String stdName) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.YUNHSM, stdName);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByPcie(int index, String stdName) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.PCIE, stdName);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByYunhsm(int index, String stdName) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.YUNHSM, stdName);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByPcie(int index,String stdName) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.PCIE,stdName);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  PCIE卡
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromPCIE(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromSdf(SdfCryptoType sdfCryptoType, int index, String password) {
        try {
            SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
            sdfSDK.init();
            boolean result = sdfSDK.checkPrivateKeyAccessRight(index, password.getBytes());
            sdfSDK.release();
            return result;
        } catch (SdfSDKException e) {
            return false;
        }
    }

}
