package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfSM2SignerUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.params.ParametersWithID;
import org.bouncycastle.jce.provider.BouncyCastleProvider;


import java.io.IOException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.Signature;
import java.security.interfaces.ECPublicKey;

/**
 * @ClassName GMSSLECSignUtils
 * @Description TODO
 * @Date 2020/1/13 16:10
 * @Author FengZhen
 */
public class GMSSLECSignUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }


    /**
     * 使用 BC 进行签名
     *
     * @param privateKey 签名者私钥
     * @param base64Data BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signByBC(PrivateKey privateKey, String base64Data, String signAlgName) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sig = signByBC(privateKey, data, signAlgName);
        return GMSSLByteArrayUtils.base64Encode(sig);
    }

    /**
     * 使用 BC 进行签名
     *
     * @param privateKey 签名者私钥
     * @param data       待签名消息
     * @return 签名数据
     */
    public static byte[] signByBC(PrivateKey privateKey, byte[] data, String signAlgName) throws Exception {
        return GMSSLBCSignUtils.generateSignature(signAlgName, privateKey, data);
    }


    /**
     * 使用 BC 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return BASE64 编码 签名数据
     */
    public static boolean verifyByBC(PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign = GMSSLByteArrayUtils.base64Decode(base64Sign);
        return verifyByBC(publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 BC 进行验签
     *
     * @param publicKey 签名者公钥
     * @param data      签名原文
     * @param sign      签名消息
     * @return 签名数据
     */
    public static boolean verifyByBC(PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        return GMSSLBCSignUtils.verifySignature(signAlgName, publicKey, data, sign);
    }


    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param base64Data         BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data, String signAlgName) throws CryptoException, SdfSDKException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sig = signBySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data, signAlgName);
        return GMSSLByteArrayUtils.base64Encode(sig);
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @param signAlgName        签名算法
     * @return 二进制 签名数据
     */
    public static byte[] signByYunHsm(int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws Exception {
        return signBySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    /**
     * 使用 密码机 进行验证签名
     *
     * @param publicKey   签名者私钥索引
     * @param data        签名原文
     * @param sign        签名数据
     * @param signAlgName 签名算法
     * @return 二进制 签名数据
     */
    public static boolean verifyByYunHsm(PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        return verifyByHSM(null, publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @param signAlgName        签名算法
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws CryptoException, SdfSDKException {
        return signBySdf(sdfCryptoType, null, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdfWithUserId(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] userId, byte[] data, String signAlgName) throws CryptoException, SdfSDKException {
        return signBySdf(sdfCryptoType, userId, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }


    /**
     * 使用 密码机 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyByYunHsm(PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws SdfSDKException {
        return verifyBySdf(SdfCryptoType.YUNHSM, publicKey, base64Data, base64Sign, signAlgName);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyByPcie(PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws SdfSDKException {
        return verifyBySdf(SdfCryptoType.PCIE, publicKey, base64Data, base64Sign, signAlgName);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param base64Data    BASE64 编码 签名原文
     * @param base64Sign    BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws SdfSDKException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign = GMSSLByteArrayUtils.base64Decode(base64Sign);
        return verifyBySdf(sdfCryptoType, publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param data          签名原文
     * @param sign          签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws SdfSDKException {
        return verifyBySdf(sdfCryptoType, null, publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param data          签名原文
     * @param sign          签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdfWithUserId(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] userId, byte[] data, byte[] sign, String signAlgName) throws SdfSDKException {
        return verifyBySdf(sdfCryptoType, userId, publicKey, data, sign, signAlgName);
    }


    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws CryptoException, SdfSDKException {
        SdfECBaseSigner signer = GMSSLSdfSM2SignerUtils.generateSM2Signer(sdfCryptoType, signAlgName);
        return signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    public static byte[] signByYunHsm(byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws CryptoException, SdfSDKException {
        SdfECSigner signer = new SdfECSigner(SdfCryptoType.YUNHSM, signAlgName);
        return signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param signer             签名计算实体类
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfECBaseSigner signer, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws CryptoException, SdfSDKException {
        return GMSSLSM2SignUtils.signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data);
    }

    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws SdfSDKException {
        SdfSM2SignerSM3 signer = new SdfSM2SignerSM3(sdfCryptoType, signAlgName);
        return verifyBySdf(signer, userId, publicKey, data, sign);
    }

    public static boolean verifyByHSM(byte[] userId, PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws SdfSDKException {
        SdfECSigner signer = new SdfECSigner(SdfCryptoType.YUNHSM, signAlgName);
        return verifyBySdf(signer, userId, publicKey, data, sign);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param signer    签名计算实体类
     * @param publicKey 签名者公钥
     * @param data      签名原文
     * @param sign      签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfECBaseSigner signer, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign) throws SdfSDKException {
        return GMSSLSM2SignUtils.verifyBySdf(signer, userId, publicKey, data, sign);
    }
}
