package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLRSAUtils;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;

import java.security.*;
import java.security.interfaces.RSAPublicKey;


public class GMSSLRSASignUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                      签名验签                                    *
     ************************************************************************************/


    /**
     * @param algorithm  算法类型
     * @param privateKey 签名私钥
     * @param data       原数据
     * @return byte[] 数组
     * @throws Exception
     */
    public static String signByBC(String algorithm, PrivateKey privateKey, String data) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] sign = GMSSLRSAUtils.generateSignature(algorithm, privateKey, dataBytes);
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据 base64字符串
     * @param sign      签名数据 base64字符串
     * @return boolean true ? false
     * @throws Exception
     */
    public static boolean verifyByBC(String algorithm, PublicKey publicKey, String data, String sign) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] signBytes = GMSSLByteArrayUtils.base64Decode(sign);
        return GMSSLRSAUtils.verifySignature(algorithm, publicKey, dataBytes, signBytes);
    }


    /**
     * @param algorithm  算法类型
     * @param privateKey 签名私钥
     * @param data       原数据
     * @return byte[] 数组
     * @throws Exception
     */
    public static byte[] signByBC(String algorithm, PrivateKey privateKey, byte[] data) throws Exception {
        return GMSSLRSAUtils.generateSignature(algorithm, privateKey, data);
    }

    /**
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean true ? false
     * @throws Exception
     */
    public static boolean verifyByBC(String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        return GMSSLRSAUtils.verifySignature(algorithm, publicKey, data, sign);
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               byte[]待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signByYunHsm(String algorithm, int privateKeyIndex, String privateKeyPassword, byte[] data) throws SdfSDKException {
        return signBySdf(SdfCryptoType.YUNHSM, algorithm, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data                String Base64待签名消息
     * @return 二进制 签名数据
     */
    public static String signByYunHsm(String algorithm, int privateKeyIndex, String privateKeyPassword, String data) throws SdfSDKException {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] sign = signBySdf(SdfCryptoType.YUNHSM, algorithm, privateKeyIndex, privateKeyPassword, dataBytes);
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * 使用 PCIE 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               byte[]待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signByPcie(String algorithm, int privateKeyIndex, String privateKeyPassword, byte[] data) throws SdfSDKException {
        return signBySdf(SdfCryptoType.PCIE, algorithm, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 PCIE 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               String Base64待签名消息
     * @return 二进制 签名数据
     */
    public static String signByPcie(String algorithm, int privateKeyIndex, String privateKeyPassword, String data) throws SdfSDKException {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] sign = signBySdf(SdfCryptoType.PCIE, algorithm, privateKeyIndex, privateKeyPassword, dataBytes);
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * 使用 密码机 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByYunHsm(String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws SdfSDKException {
        return verifyBySdf(SdfCryptoType.YUNHSM, algorithm, publicKey, data, sign);
    }

    /**
     * 使用 密码机 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByYunHsm(String algorithm, PublicKey publicKey, String data, String sign) throws SdfSDKException {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] signBytes = GMSSLByteArrayUtils.base64Decode(sign);
        return verifyBySdf(SdfCryptoType.YUNHSM, algorithm, publicKey, dataBytes, signBytes);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByPcie(String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws SdfSDKException {
        return verifyBySdf(SdfCryptoType.PCIE, algorithm, publicKey, data, sign);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByPcie(String algorithm, PublicKey publicKey, String data, String sign) throws SdfSDKException {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] signBytes = GMSSLByteArrayUtils.base64Decode(sign);
        return verifyBySdf(SdfCryptoType.PCIE, algorithm, publicKey, dataBytes, signBytes);
    }


    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               byte[]待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, String algorithm, int privateKeyIndex, String privateKeyPassword, byte[] data) throws SdfSDKException {
        AlgorithmIdentifier algorithmIdentifier = new DefaultSignatureAlgorithmIdentifierFinder().find(algorithm);
        SdfRSASigner sdfRSASigner = new SdfRSASigner(sdfCryptoType, algorithmIdentifier);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        SdfRSAKeyParameters rsaKeyParameters = new SdfRSAKeyParameters(sdfPrivateKey);
        sdfRSASigner.init(true, rsaKeyParameters);
        sdfRSASigner.update(data, 0, data.length);
        byte[] sign = sdfRSASigner.generateSignature();
        sdfRSASigner.release();
        return sign;
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param algorithm     算法类型
     * @param publicKey     签名公钥
     * @param data          原数据
     * @param sign          签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws SdfSDKException {
        AlgorithmIdentifier algorithmIdentifier = new DefaultSignatureAlgorithmIdentifierFinder().find(algorithm);
        SdfRSASigner sdfRSASigner = new SdfRSASigner(sdfCryptoType, algorithmIdentifier);
        SdfRSAKeyParameters rsaKeyParameter = new SdfRSAKeyParameters((RSAPublicKey) publicKey);
        sdfRSASigner.init(false, rsaKeyParameter);
        sdfRSASigner.update(data, 0, data.length);
        boolean verify = sdfRSASigner.verifySignature(sign);
        sdfRSASigner.release();
        return verify;
    }
}
