package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfECKey;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.*;

public class GMSSLSM2KeyUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                      密钥对相关                                  *
     ************************************************************************************/
    /**
     * 生成 SM2 密钥对
     *
     * @return SM2 密钥对
     * @throws NoSuchProviderException            找不到相应的 Provider
     * @throws NoSuchAlgorithmException           找不到相应的算法
     * @throws InvalidAlgorithmParameterException KeyPairGenerator 初始化异常
     */
    public static KeyPair generateSM2KeyPairByBC() throws NoSuchProviderException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        return GMSSLX509Utils.generateSM2KeyPair();
    }

    /**
     * 生成 SDF 相关的 SM2 私钥
     *
     * @param index    密钥访问索引
     * @param password 密钥访问控制密码
     * @return SDF 相关的 SM2私钥
     */
    public static SdfPrivateKey genSdfPrivateKey(int index, String password) {
        return new SdfPrivateKey(index, password.getBytes());
    }

    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByYunhsm(int index) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByPcie(int index) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByYunhsm(int index) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByPcie(int index) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  PCIE卡
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromPCIE(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromSdf(SdfCryptoType sdfCryptoType, int index, String password) {
        try {
            SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
            sdfSDK.init();
            boolean result = sdfSDK.checkPrivateKeyAccessRight(index, password.getBytes());
            sdfSDK.release();
            return result;
        } catch (SdfSDKException e) {
            return false;
        }
    }

}
