package com.xdja.pki.gmssl.operator.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentVerifierProvider;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;

import java.security.PublicKey;

public class GMSSLContentVerifierProviderUtils {
    /**
     * 通过BC 生成 content verifier
     *
     * @param publicKey 公钥信息
     * @return ContentVerifierProvider
     * @throws OperatorCreationException
     */
    public static ContentVerifierProvider generateContentVerifierByBC(PublicKey publicKey) throws OperatorCreationException {
        return generateContentVerifierByBC(SubjectPublicKeyInfo.getInstance(publicKey.getEncoded()));
    }

    /**
     * 生成 content verifier with bc
     *
     * @param subjectPublicKeyInfo 公钥信息
     * @return ContentVerifierProvider
     * @throws OperatorCreationException
     */
    public static ContentVerifierProvider generateContentVerifierByBC(SubjectPublicKeyInfo subjectPublicKeyInfo) throws OperatorCreationException {
        return new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(subjectPublicKeyInfo);
    }

    /**
     * 通过BC 生成 content verifier
     *
     * @param publicKey 公钥信息
     * @return ContentVerifierProvider
     */
    public static ContentVerifierProvider generateContentVerifierByYunHsm(PublicKey publicKey) {
        return generateContentVerifierBySdf(SdfCryptoType.YUNHSM, publicKey);
    }

    /**
     * 通过BC 生成 content verifier
     *
     * @param publicKey 公钥信息
     * @return ContentVerifierProvider
     */
    public static ContentVerifierProvider generateContentVerifierByPcie(PublicKey publicKey) {
        return generateContentVerifierBySdf(SdfCryptoType.PCIE, publicKey);
    }

    /**
     * 通过BC 生成 content verifier
     *
     * @param sdfCryptoType sdf 密码算法类型
     * @param publicKey     公钥信息
     * @return ContentVerifierProvider
     */
    public static ContentVerifierProvider generateContentVerifierBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey) {
        return new SdfContentVerifierProvider(sdfCryptoType, publicKey);
    }

    public static ContentVerifierProvider generateContentVerifierBySdfHSM(SdfCryptoType sdfCryptoType, PublicKey publicKey) {
        return new SdfContentVerifierProvider(sdfCryptoType, publicKey, false);
    }


}
