package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfSM4Engine;
import com.xdja.pki.gmssl.crypto.sdf.SdfSymmetricKeyParameters;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.engines.SM4Engine;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;


import java.security.Security;

public class GMSSLSM4BlockCipherUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 使用 BC 进行加密
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBC(String base64Key, String base64Data) {
        return GMSSLByteArrayUtils.base64Encode(encryptByBC(GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 BC 进行加密
     *
     * @param key  会话密钥 数组后长度为16
     * @param data 待加密消息 数组后长度为16
     * @return byte[]  编码 密文消息
     */
    public static byte[] encryptByBC(byte[] key, byte[] data) {
        SM4Engine sM4Engine = new SM4Engine();
        byte[] out = new byte[data.length];
        CipherParameters param = new KeyParameter(key);
        sM4Engine.init(true, param);
        sM4Engine.processBlock(data, 0, out, 0);
        return out;
    }

    /**
     * 使用 BC 进行解密
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBC(String base64Key, String base64Cipher) {
        return GMSSLByteArrayUtils.base64Encode(decryptByBC(GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 BC 进行解密
     *
     * @param key    会话密钥 长度为16
     * @param cipher 加密消息 长度为16
     * @return byte[] 密文消息
     */
    public static byte[] decryptByBC(byte[] key, byte[] cipher) {
        SM4Engine sM4Engine = new SM4Engine();
        byte[] out = new byte[cipher.length];
        CipherParameters param = new KeyParameter(key);
        sM4Engine.init(false, param);
        sM4Engine.processBlock(cipher, 0, out, 0);
        return out;
    }


    /**
     * 使用 加密机Yunhsm 进行加密 传入明文为16位
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYunhsm(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4EngineBySdf(true, SdfCryptoType.YUNHSM,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 加密机Yunhsm 进行解密
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYunhsm(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4EngineBySdf(false, SdfCryptoType.YUNHSM,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 PCIE 进行加密 传入明文为16位
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcie(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4EngineBySdf(true, SdfCryptoType.PCIE,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 PCIE 进行解密
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcie(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4EngineBySdf(false, SdfCryptoType.PCIE,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }


    /**
     * sm4Block 加解密公用类
     *
     * @param forEncryption 是否是加密
     * @param key           byte[] 会话密钥 长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息 长度为16
     * @return BASE64 编码  将明文信息加密后的消息或者将密文信息解密后的消息
     */
    public static byte[] sm4EngineByYunHsm(boolean forEncryption, byte[] key, byte[] data) throws Exception {
        return sm4EngineBySdf(forEncryption, SdfCryptoType.YUNHSM, key, data);
    }

    /**
     * sm4Block 加解密公用类 PCIE卡
     *
     * @param forEncryption 是否是加密
     * @param key           byte[] 会话密钥 长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息 长度为16
     * @return BASE64 编码  将明文信息加密后的消息或者将密文信息解密后的消息
     */
    public static byte[] sm4EngineByPCIE(boolean forEncryption, byte[] key,
                                         byte[] data) throws Exception {
        return sm4EngineBySdf(forEncryption, SdfCryptoType.PCIE, key, data);
    }

    /**
     * sm4Block 加解密公用类
     *
     * @param forEncryption 是否是加密
     * @param sdfCryptoType 加密使用类型 包括加密机YumHsm和PCIE卡
     * @param key           byte[] 会话密钥 长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息 长度为16
     * @return BASE64 编码  将明文信息加密后的消息或者将密文信息解密后的消息
     */
    public static byte[] sm4EngineBySdf(boolean forEncryption, SdfCryptoType sdfCryptoType, byte[] key,
                                        byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            if (forEncryption) {
                return GMSSLSM4ECBEncryptUtils.encrypt(key, data, SdfSymmetricKeyParameters.PaddingType.NoPadding);
            }else {
                return GMSSLSM4ECBEncryptUtils.decrypt(key, data, SdfSymmetricKeyParameters.PaddingType.NoPadding);
            }
        }
        SdfSM4Engine sdfSM4Engine = new SdfSM4Engine(sdfCryptoType);
        byte[] out = new byte[data.length];
        CipherParameters param = new KeyParameter(key);
        sdfSM4Engine.init(forEncryption, param);
        sdfSM4Engine.processBlock(data, 0, out, 0);
        sdfSM4Engine.release();
        return out;
    }

}
