package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.GBAlgorithmID_SGD;
import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.utils.MiniPcieSymmetricEncryptUtils;
import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfSymmetricKeyParameters;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;
import org.bouncycastle.jce.provider.BouncyCastleProvider;


import java.security.Security;


public class GMSSLSM4CBCEncryptUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * SM4 ECB模式 加密数据
     *
     * @param key         对称密钥
     * @param data        需要加密的数据
     * @param iv          初始化向量
     * @param paddingType 填充模式
     * @return
     * @throws Exception
     */
    public static byte[] encrypt(byte[] key, byte[] data, byte[] iv, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE, paddingType, key, iv, data);
            case XDJA_HSM:
                return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM, paddingType, key, iv, data);
            case MINI_PCI_E:
                switch (paddingType) {
                    case NoPadding:
                        return MiniPcieXKFUtils.sm4(key, data, JNIAPI.CBC_ENCRYPT, iv);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return MiniPcieSymmetricEncryptUtils.sm4CbcWithPkcs5Padding(key, data, iv, true);
                    case SSL3Padding:
                    default:
                        return MiniPcieSymmetricEncryptUtils.sm4CbcWithSslv3Padding(key, data, iv, true);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                switch (paddingType) {
                    case NoPadding:
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM4_CBC, key, iv, data);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        data = GMSSLX509Utils.paddingData(data, true);
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM4_CBC, key, iv, data);
                    case SSL3Padding:
                    default:
                        data = GMSSLX509Utils.paddingData(data, false);
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM4_CBC, key, iv, data);
                }
            case BC:
                switch (paddingType) {
                    case NoPadding:
                        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                                key, data, iv);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                                key, data, iv);
                    case SSL3Padding:
                    default:
                        throw new IllegalArgumentException("un support the sm4 ecb sslv3Padding encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
                }
            default:
                throw new IllegalArgumentException("un support the sm4 cbc encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * SM4 ECB解密
     *
     * @param key         对称秘钥
     * @param data        密文数据
     * @param iv          初始化向量
     * @param paddingType 填充模式
     * @return
     * @throws Exception
     */
    public static byte[] decrypt(byte[] key, byte[] data, byte[] iv, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE, paddingType, key, iv, data);
            case XDJA_HSM:
                return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM, paddingType, key, iv, data);
            case MINI_PCI_E:
                switch (paddingType) {
                    case NoPadding:
                        return MiniPcieXKFUtils.sm4(key, data, JNIAPI.CBC_DECRYPT, iv);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return MiniPcieSymmetricEncryptUtils.sm4CbcWithPkcs5Padding(key, data, iv, false);
                    case SSL3Padding:
                    default:
                        return MiniPcieSymmetricEncryptUtils.sm4CbcWithSslv3Padding(key, data, iv, false);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                switch (paddingType) {
                    case NoPadding:
                        return cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM4_CBC, key, iv, data);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        byte[] decrypt = cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM4_CBC, key, iv, data);
                        return GMSSLX509Utils.removePadding(data, decrypt, true);
                    case SSL3Padding:
                    default:
                        byte[] decryptSsl3 = cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM4_CBC, key, iv, data);
                        return GMSSLX509Utils.removePadding(data, decryptSsl3, false);
                }
            case BC:
                switch (paddingType) {
                    case NoPadding:
                        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                                key, data, iv);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                                key, data, iv);
                    case SSL3Padding:
                    default:
                        throw new IllegalArgumentException("un support the sm4 ecb sslv3Padding decrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
                }
            default:
                throw new IllegalArgumentException("un support the sm4 cbc decrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }


    /**
     * 使用 BC 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIV BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBCWithNoPadding(String base64Key, String base64Data, String base64InitIV) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data),
                GMSSLByteArrayUtils.base64Decode(base64InitIV)));
    }
    public static byte[] encryptByBCWithNoPadding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_NoPadding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIV BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBCWithPKCS5Padding(String base64Key, String base64Data, String base64InitIV) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data),
                GMSSLByteArrayUtils.base64Decode(base64InitIV)));
    }
    public static byte[] encryptByBCWithPKCS5Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS5Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIV BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBCWithPKCS7Padding(String base64Key, String base64Data, String base64InitIV) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data),
                GMSSLByteArrayUtils.base64Decode(base64InitIV)));
    }
    public static byte[] encryptByBCWithPKCS7Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS7Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 BC 进行解密 NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @param base64InitIV BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBCWithNoPadding(String base64Key, String base64Data, String base64InitIV) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data),
                GMSSLByteArrayUtils.base64Decode(base64InitIV)));
    }
    public static byte[] decryptByBCWithNoPadding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_NoPadding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIV BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBCWithPKCS5Padding(String base64Key, String base64Data, String base64InitIV) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data),
                GMSSLByteArrayUtils.base64Decode(base64InitIV)));
    }
    public static byte[] decryptByBCWithPKCS5Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS5Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIV BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBCWithPKCS7Padding(String base64Key, String base64Data, String base64InitIV) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data),
                GMSSLByteArrayUtils.base64Decode(base64InitIV)));
    }
    public static byte[] decryptByBCWithPKCS7Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_CBC_PKCS7Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithNoPadding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByYumhsmWithNoPadding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行加密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithPKCS5Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByYumhsmWithPKCS5Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行加密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithPKCS7Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByYumhsmWithPKCS7Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行加密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithSSL3Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByYumhsmWithSSL3Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithNoPadding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByYumhsmWithNoPadding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithPKCS5Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByYumhsmWithPKCS5Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithPKCS7Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByYumhsmWithPKCS7Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithSSL3Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByYumhsmWithSSL3Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithNoPadding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByPcieWithNoPadding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行加密 PKCS5Padding
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithPKCS5Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByPcieWithPKCS5Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行加密 PKCS7Padding
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithPKCS7Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByPcieWithPKCS7Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行加密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithSSL3Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }
    public static byte[] encryptByPcieWithSSL3Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行解密  NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithNoPadding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByPcieWithNoPadding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行解密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithPKCS5Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByPcieWithPKCS5Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行解密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithPKCS7Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByPcieWithPKCS7Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 PCIE 进行解密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithSSL3Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }
    public static byte[] decryptByPcieWithSSL3Padding(byte[] key, byte[] data, byte[] initIV) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data,
                initIV);
    }

    /**
     * 使用 padding模式填充的加解密公用类  PCIE
     *
     * @param forEncryption 是否为加密
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           byte[] 会话密钥 长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息
     * @param iv            byte[] 数组 初始化向量 长度为16
     * @return byte[] 明文消息
     */
    public static byte[] sm4SymmetricWithPaddingByPCIE(
            boolean forEncryption, SdfSymmetricKeyParameters.PaddingType paddingType,
            byte[] key, byte[] iv, byte[] data
    ) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptBySdf(forEncryption, SdfCryptoType.PCIE,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM4_CBC, iv, data);

    }


    /**
     * 使用 padding模式填充的加解密公用类  YunHsm密码机
     *
     * @param forEncryption 是否为加密
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           byte[] 会话密钥 长度为16
     * @param iv            byte[] 数组 初始化向量 长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息
     * @return byte[] 明文消息
     */
    public static byte[] sm4SymmetricWithPaddingByYunHsm(
            boolean forEncryption, SdfSymmetricKeyParameters.PaddingType paddingType,
            byte[] key, byte[] iv, byte[] data
    ) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptBySdf(forEncryption, SdfCryptoType.YUNHSM,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM4_CBC, iv, data);

    }

    /**
     * 使用 padding模式填充的加解密公用类
     *
     * @param forEncryption 是否为加密
     * @param sdfCryptoType 加密使用类型 包括加密机YumHsm和PCIE卡
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           byte[] 会话密钥 长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息
     * @param iv            byte[] 数组 初始化向量 长度为16
     * @return byte[] 明文消息
     */
    public static byte[] sm4SymmetricSdfWithPadding(
            boolean forEncryption, SdfCryptoType sdfCryptoType,
            SdfSymmetricKeyParameters.PaddingType paddingType,
            byte[] key, byte[] iv, byte[] data
    ) throws Exception {

        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptBySdf(forEncryption, sdfCryptoType,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM4_CBC, iv, data);
    }

}
