package com.xdja.pki.gmssl.sdf.yunhsm.utils;

import com.sansec.devicev4.api.CryptoException;
import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.struct.DeviceInfo;
import com.xdja.pki.gmssl.core.utils.GMSSLFileUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.x509.utils.bean.SancHsmInfoEntry;
import com.xdja.pki.gmssl.x509.utils.bean.YunHsmExceptionEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;

/**
 * @ClassName GMSSLSancHsmUtils
 * @Description 三未信安密码机初始化相关类
 * @Date 2020/5/25 11:11
 * @Author FengZhen
 */
public class GMSSLSancHsmUtils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLSancHsmUtils.class);
    public static final String confName = "swsds.ini";

    /**
     * 使用当前配置进行测试连接  测试密码机状态使用
     */
    public static YunHsmExceptionEnum testConnect() {
        try {
            ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
            DeviceInfo deviceInfo = cryptConnection.getDeviceInfo();
            logger.info(deviceInfo.toString());
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("获取设备信息失败");
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }

    /**
     * 测试连接  默认系统已经进行初始化 测试完连接后会使用原来配置进行连接三未信安密码机
     *
     * @param ip        密码机ip
     * @param port      端口
     * @param loginPass 登录密码
     * @return 密码机服务状态信息
     */
    public static YunHsmExceptionEnum testConnect(String ip, int port, String loginPass) {
        return testConnect(ip, port, loginPass, true);
    }

    /**
     * 测试连接
     *
     * @param ip        密码机ip
     * @param port      端口
     * @param loginPass 登录密码
     * @param isInit    系统是否进行初始化
     *                  该判断目的
     *                  1、 系统已经完成初始化 测试连接后需要使用释放连接并重新使用原来的配置进行连接三未信安密码机
     *                  2、 系统未进行初始化，测试连接后释放连接即可。不需要使用原来配置进行连接三未信安密码机。
     *                  不进行判断也可以通过，默认使用密码机配置文件路径连接，但是没有文件的话后台日志会报错
     * @return 密码机服务状态信息
     */
    public static YunHsmExceptionEnum testConnect(String ip, int port, String loginPass, boolean isInit) {
        try {
            //先释放连接
            GMSSLSancConnectionUtils.releaseConnection();
            //获取配置文件地址
            String confPath = getConfPath(true);
            //写配置文件
            GMSSLIniFileUtils.createProfile(confPath, ip, port, loginPass, null, null, null, null);
            //使用测试路径配置进行初始化
            GMSSLSancConnectionUtils.init(confPath + confName);
            //获取设备信息
            DeviceInfo deviceInfo = GMSSLSancConnectionUtils.getCryptConnection().getDeviceInfo();
            //释放连接
            GMSSLSancConnectionUtils.releaseConnection();
            logger.info(deviceInfo.toString());
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("测试三未信安密码机连通性失败", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        } finally {
            //需要使用原来的密码机配置重新连上
            try {
                if (isInit) {
                    String confPath = getConfPath(false) + File.separator + "swsds.ini";
                    File file = new File(confPath);
                    if (file.exists()) {
                        GMSSLSancConnectionUtils.init();
                        logger.info("重新使用原来密码机配置连接三未信安密码机");
                    } else {
                        logger.info("未找到配置文件路径");
                    }
                } else {
                    logger.info("系统暂未进行初始化，不需要重新连接密码机");
                }
            } catch (Exception e) {
                logger.info("三未信安密码机使用原配置文件初始化失败");
            }
        }
    }

    /**
     * 测试连接并保存
     *
     * @param ip        密码机ip
     * @param port      端口
     * @param loginPass 登录密码
     * @return 密码机服务状态信息
     */
    public static YunHsmExceptionEnum initConfigAndTestConnect(String ip, int port, String loginPass) {
        return initConfigAndTestConnect(ip, port, loginPass, true);
    }

    public static YunHsmExceptionEnum initConfigAndTestConnect(String ip, int port, String loginPass, boolean isInit) {
        try {
            YunHsmExceptionEnum yunHsmExceptionEnum = testConnect(ip, port, loginPass, isInit);
            if (yunHsmExceptionEnum != YunHsmExceptionEnum.NORMAL) {
                return yunHsmExceptionEnum;
            }
            //先释放连接
            GMSSLSancConnectionUtils.releaseConnection();
            //获取配置文件地址

            String goalConfPath = getConfPath(false);
            GMSSLIniFileUtils.createProfile(goalConfPath, ip, port, loginPass, null, null, null, null);
            GMSSLPkiCryptoInit.getSancHsmInstance();
            DeviceInfo deviceInfo = GMSSLSancConnectionUtils.getCryptConnection().getDeviceInfo();
            logger.info(deviceInfo.toString());
            return YunHsmExceptionEnum.NORMAL;
        } catch (Exception e) {
            logger.error("测试三未信安密码机连通性失败,", e);
            return YunHsmExceptionEnum.OPEN_DEVICE_IS_FAILURE;
        }
    }

    public static SancHsmInfoEntry getSancDeviceInfo() {
        String confPath = getConfPath(false) + File.separator + "swsds.ini";
        String ip = GMSSLIniFileUtils.getProfileValue(confPath, "HSM1", "ip");
        String port = GMSSLIniFileUtils.getProfileValue(confPath, "HSM1", "port");
        String loginPasswd = GMSSLIniFileUtils.getProfileValue(confPath, "HSM1", "passwd");
        SancHsmInfoEntry sancHsmInfoEntry = new SancHsmInfoEntry();
        sancHsmInfoEntry.setIp(ip);
        sancHsmInfoEntry.setPort(Integer.valueOf(port));
        sancHsmInfoEntry.setLoginPasswd(loginPasswd);
        return sancHsmInfoEntry;
    }

    /**
     * 获取三未信安密码机配置文件路径
     *
     * @param isTest 是否为test路径
     * @return
     */
    public static String getConfPath(boolean isTest) {
        String path;
        String tomcatPath = System.getProperty("catalina.home");
        if (null == tomcatPath || tomcatPath.contains("Temp")) {
            tomcatPath = "/home/xdja";
        }
        if (isTest) {
            path = tomcatPath + "/conf/sancConfTest/";
        } else {
            path = tomcatPath + "/conf/sancConf/";
        }
        return path;
    }
}
