package com.xdja.pki.gmssl.utils.bc;

import com.xdja.pki.gmssl.asn1.x509.SubjectInformationAccess;
import com.xdja.pki.gmssl.x509.utils.bean.CRLEntry;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.*;
import org.bouncycastle.cert.*;
import org.bouncycastle.cert.jcajce.JcaX509CRLConverter;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.PKCS10CertificationRequestBuilder;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;

import java.io.IOException;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.*;
import java.security.spec.ECGenParameterSpec;
import java.util.Date;
import java.util.List;

public class X509CertUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public enum ContentSignatureAlgorithm {
        SM3_WITH_SM2("SM3withSM2");

        private String id;

        public String getId() {
            return id;
        }

        ContentSignatureAlgorithm(String id) {
            this.id = id;
        }

        ContentSignatureAlgorithm(AlgorithmIdentifier algorithmIdentifier) {

        }
    }

    public enum ContentCryptoType {
        BC("bc"),
        PCIE("pcie"),
        YUNHSM("yunhsm");

        private String id;

        public String getId() {
            return id;
        }

        ContentCryptoType(String id) {
            this.id = id;
        }
    }

    /************************************************************************************
     *                                      密钥对生成                                  *
     ************************************************************************************/
    /**
     * 生成 SM2 密钥对
     *
     * @return SM2 密钥对
     * @throws NoSuchProviderException            找不到相应的 Provider
     * @throws NoSuchAlgorithmException           找不到相应的算法
     * @throws InvalidAlgorithmParameterException KeyPairGenerator 初始化异常
     */
    public static KeyPair generateSM2KeyPair()
            throws NoSuchProviderException, NoSuchAlgorithmException, InvalidAlgorithmParameterException {
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance("EC", BouncyCastleProvider.PROVIDER_NAME);

        kpGen.initialize(new ECGenParameterSpec("sm2p256v1"), new SecureRandom());

        return kpGen.generateKeyPair();
    }

    /************************************************************************************
     *                                  证书生成                                        *
     ************************************************************************************/
    /**
     * 生成证书 默认 sm3_with_sm2 签名，现默认 BC 软实现签名，以后默认使用密码机签名
     *
     * @param issuer     签发者 string DN
     * @param subject    证书主体 string DN
     * @param serial     证书序列号
     * @param notBefore  证书有效期开始时间
     * @param notAfter   证书有效期结束时间
     * @param subjectKey 证书主体 公钥信息
     * @param issuerKey  证书签发者 私钥信息
     * @param extensions 证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCert(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PublicKey subjectKey, PrivateKey issuerKey,
            List<Extension> extensions
    ) throws CertIOException, OperatorCreationException, CertificateException {
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, issuerKey,
                extensions,
                ContentSignatureAlgorithm.SM3_WITH_SM2, ContentCryptoType.BC
        );
    }

    /**
     * 生成证书 string DN
     *
     * @param issuer             签发者 string DN
     * @param subject            证书主体 string DN
     * @param serial             证书序列号
     * @param notBefore          证书有效期开始时间
     * @param notAfter           证书有效期结束时间
     * @param subjectKey         证书主体 公钥信息
     * @param issuerKey          证书签发者 私钥信息
     * @param extensions         证书扩展项列表
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param contentCryptoType  算法类型 ContentCryptoType 枚举 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    public static X509Certificate generateCert(
            String issuer, String subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PublicKey subjectKey, PrivateKey issuerKey,
            List<Extension> extensions,
            ContentSignatureAlgorithm signatureAlgorithm, ContentCryptoType contentCryptoType
    ) throws CertIOException, OperatorCreationException, CertificateException {
        return generateCert(
                new X500Name(RFC4519Style.INSTANCE, issuer), new X500Name(RFC4519Style.INSTANCE, subject),
                serial, notBefore, notAfter,
                subjectKey, issuerKey,
                extensions,
                signatureAlgorithm, contentCryptoType
        );
    }

    /**
     * 生成证书 X500Name DN
     *
     * @param issuer             签发者 X500Name DN
     * @param subject            证书主体 X500Name DN
     * @param serial             证书序列号
     * @param notBefore          证书有效期开始时间
     * @param notAfter           证书有效期结束时间
     * @param subjectKey         证书主体 公钥信息
     * @param issuerKey          证书签发者 私钥信息
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param contentCryptoType  算法类型 ContentCryptoType 枚举 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     * @param extensions         证书扩展项列表
     * @return X509证书
     * @throws CertIOException           添加扩展项异常
     * @throws OperatorCreationException 证书签名异常
     * @throws CertificateException      证书生成异常
     */
    private static X509Certificate generateCert(
            X500Name issuer, X500Name subject,
            BigInteger serial, Date notBefore, Date notAfter,
            PublicKey subjectKey, PrivateKey issuerKey,
            List<Extension> extensions,
            ContentSignatureAlgorithm signatureAlgorithm, ContentCryptoType contentCryptoType
    ) throws CertIOException, OperatorCreationException, CertificateException {

        //subject public key info
        SubjectPublicKeyInfo subjectPublicKeyInfo = SubjectPublicKeyInfo.getInstance(subjectKey.getEncoded());

        //builder
        X509v3CertificateBuilder builder = new X509v3CertificateBuilder(issuer, serial, notBefore, notAfter, subject, subjectPublicKeyInfo);

        //extensions
        for (Extension extension : extensions) {
            //add extension error throw CertIOException
            builder.addExtension(extension);
        }

        //signer
        //gen content signer error throw OperatorCreationException
        ContentSigner contentSigner = genContentSigner(contentCryptoType, signatureAlgorithm, issuerKey);
        X509CertificateHolder holder = builder.build(contentSigner);

        //gen certificate getCertificate throw CertificateException
        return new JcaX509CertificateConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCertificate(holder);
    }


    /************************************************************************************
     *                                 CRL 生成                                         *
     ************************************************************************************/

    /**
     * 生成 CRL 默认 sm3_with_sm2 签名，现默认 BC 软实现签名，以后默认使用密码机签名
     *
     * @param issueCert  签发者证书
     * @param issuerKey  签发者私钥信息
     * @param crlNumber  CRL 序列号
     * @param nextUpdate 下次更新时间
     * @param extensions 扩展项列表
     * @param revoked    CRL实体 列表
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRL(
            X509Certificate issueCert, PrivateKey issuerKey,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked
    ) throws IOException, OperatorCreationException, CRLException {
        return generateCRL(issueCert, issuerKey, crlNumber, thisUpdate, nextUpdate, extensions, revoked, ContentCryptoType.BC, ContentSignatureAlgorithm.SM3_WITH_SM2);
    }

    /**
     * 生成 CRL
     *
     * @param issueCert          签发者证书
     * @param issuerKey          签发者私钥信息
     * @param crlNumber          CRL 序列号
     * @param nextUpdate         下次更新时间
     * @param extensions         扩展项列表
     * @param revoked            CRL实体 列表
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param contentCryptoType  算法类型 ContentCryptoType 枚举 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     * @return X509CRL
     * @throws IOException               添加扩展项异常
     * @throws OperatorCreationException CRL签名异常
     * @throws CRLException              CRL生成异常
     */
    public static X509CRL generateCRL(
            X509Certificate issueCert, PrivateKey issuerKey,
            BigInteger crlNumber, Date thisUpdate, Date nextUpdate,
            List<Extension> extensions,
            List<CRLEntry> revoked,
            ContentCryptoType contentCryptoType, ContentSignatureAlgorithm signatureAlgorithm
    ) throws IOException, OperatorCreationException, CRLException {
        X509v2CRLBuilder builder = new X509v2CRLBuilder(new X500Name(issueCert.getSubjectDN().getName()), thisUpdate);

        builder.addExtension(Extension.cRLNumber, false, new CRLNumber(crlNumber));
        builder.setNextUpdate(nextUpdate);

        //extensions
        for (Extension extension : extensions) {
            builder.addExtension(extension);
        }

        if (revoked != null) {
            for (CRLEntry entry : revoked) {
                builder.addCRLEntry(new BigInteger(entry.getUserCertificateSerial(), 16), entry.getRevocationDate(), entry.getReason());
            }
        }

        ContentSigner contentSigner = genContentSigner(contentCryptoType, signatureAlgorithm, issuerKey);
        X509CRLHolder crlHolder = builder.build(contentSigner);
        return new JcaX509CRLConverter().setProvider(BouncyCastleProvider.PROVIDER_NAME).getCRL(crlHolder);
    }

    /************************************************************************************
     *                                 P10 生成 解析                                    *
     ************************************************************************************/
    /**
     * 生成 P10 证书申请  默认 sm3_with_sm2 签名，现默认 BC 软实现签名，以后默认使用密码机签名
     *
     * @param subject    主体 string DN
     * @param publicKey  主体公钥信息
     * @param privateKey 主体私钥信息
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10(String subject, PublicKey publicKey, PrivateKey privateKey) throws OperatorCreationException {
        return generateP10(subject, ContentCryptoType.BC, ContentSignatureAlgorithm.SM3_WITH_SM2, publicKey, privateKey);
    }

    public static PKCS10CertificationRequest generateP10(String subject, ContentCryptoType contentCryptoType, ContentSignatureAlgorithm signatureAlgorithm, PublicKey publicKey, PrivateKey privateKey) throws OperatorCreationException {
        PKCS10CertificationRequestBuilder requestBuilder = new JcaPKCS10CertificationRequestBuilder(new X500Name(subject), publicKey);

        ContentSigner contentSigner = genContentSigner(contentCryptoType, signatureAlgorithm, privateKey);

        return requestBuilder.build(contentSigner);
    }

    /**
     * P10 证书申请 编码成二进制
     *
     * @param pkcs10CertificationRequest org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @return 二进制编码
     * @throws IOException 编码异常
     */
    public static byte[] encodeP10(PKCS10CertificationRequest pkcs10CertificationRequest) throws IOException {
        return pkcs10CertificationRequest.getEncoded();
    }

    /**
     * P10 证书申请解码
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10(byte[] p10) throws IOException {
        return new PKCS10CertificationRequest(p10);
    }

    /************************************************************************************
     *                             ContentSigner 生成                                   *
     ************************************************************************************/
    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param contentCryptoType  算法类型 ContentCryptoType 枚举 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner genContentSigner(ContentCryptoType contentCryptoType, ContentSignatureAlgorithm signatureAlgorithm, PrivateKey privateKey) throws OperatorCreationException {
        //bc
//        AlgorithmIdentifier sigAlgId = new DefaultSignatureAlgorithmIdentifierFinder().find("SM3withSM2");
//        AlgorithmIdentifier digAlgId = new DefaultDigestAlgorithmIdentifierFinder().find(sigAlgId);
//        BcContentSignerBuilder contentSignerBuilder = new BcECContentSignerBuilder(sigAlgId, digAlgId);
//        AsymmetricKeyParameter privateKeyParameter = GMSSLX509Utils.convertECPublicKeyParameters(issuerKey);
//        ContentSigner bcCcontentSigner = contentSignerBuilder.build(privateKeyParameter);
        switch (contentCryptoType) {
            case BC:
                JcaContentSignerBuilder contentSignerBuilder = new JcaContentSignerBuilder(signatureAlgorithm.getId());
                return contentSignerBuilder.build(privateKey);
            case PCIE:
                throw new OperatorCreationException("un support crypto type " + contentCryptoType);
            case YUNHSM:
                throw new OperatorCreationException("un support crypto type " + contentCryptoType);
            default:
                throw new OperatorCreationException("un support crypto type " + contentCryptoType);
        }
    }

    /**
     * 算法类型转换 BC to xdja
     *
     * @param algorithmIdentifier org.bouncycastle.asn1.x509.AlgorithmIdentifier 算法类型
     * @return com.xdja.pki.gmssl.utils.bc.X509CertUtils.GMSSLSdfContentSignatureAlgorithm 算法类型
     */
    public static ContentSignatureAlgorithm convertContentSignatureAlgorithm(AlgorithmIdentifier algorithmIdentifier) {
        if (algorithmIdentifier.getAlgorithm() == GMObjectIdentifiers.sm2sign_with_sm3) {
            return ContentSignatureAlgorithm.SM3_WITH_SM2;
        }
        return null;
    }

    /**
     * 算法类型转换 xdja to BC
     *
     * @param signatureAlgorithm com.xdja.pki.gmssl.utils.bc.X509CertUtils.GMSSLSdfContentSignatureAlgorithm 算法类型
     * @return org.bouncycastle.asn1.x509.AlgorithmIdentifier 算法类型
     */
    public static AlgorithmIdentifier convertAlgorithmIdentifier(ContentSignatureAlgorithm signatureAlgorithm) {
        return new DefaultSignatureAlgorithmIdentifierFinder().find(signatureAlgorithm.getId());
    }

    /**
     * 生成 ContentVerifier 内容验签实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param contentCryptoType  算法类型 ContentCryptoType 枚举 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     * @param signatureAlgorithm 签名算法 GMSSLSdfContentSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param publicKey          验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier genContentVerifier(ContentCryptoType contentCryptoType, ContentSignatureAlgorithm signatureAlgorithm, PublicKey publicKey) throws OperatorCreationException {
        return genContentVerifier(contentCryptoType, convertAlgorithmIdentifier(signatureAlgorithm), publicKey);
    }

    /**
     * 生成 ContentVerifier 内容验签实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param contentCryptoType   算法类型 ContentCryptoType 枚举 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     * @param algorithmIdentifier 签名算法 org.bouncycastle.asn1.x509.AlgorithmIdentifier 签名算法
     * @param publicKey           验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier genContentVerifier(ContentCryptoType contentCryptoType, AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey) throws OperatorCreationException {
        switch (contentCryptoType) {
            case BC:
                ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
                return verifierProvider.get(algorithmIdentifier);
            case PCIE:
                throw new OperatorCreationException("un support crypto type " + contentCryptoType);
            case YUNHSM:
                throw new OperatorCreationException("un support crypto type " + contentCryptoType);
            default:
                throw new OperatorCreationException("un support crypto type " + contentCryptoType);
        }
    }

    /************************************************************************************
     *               证书 CRL 都需要的 Extension 扩展项生成                             *
     ************************************************************************************/


    /**
     * 获取 授权信息访问 扩展
     *
     * @param ldapUrl 签发者证书 ldap 地址
     * @return 授权信息访问扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genAuthorityInfoAccessExtension(String ldapUrl) throws IOException {
        ASN1OctetString instance = new DEROctetString(ldapUrl.getBytes());
        AuthorityInformationAccess issueInfo = new AuthorityInformationAccess(new AccessDescription(
                AccessDescription.id_ad_caIssuers,
                new GeneralName(GeneralName.uniformResourceIdentifier, instance)));
        return new Extension(Extension.authorityInfoAccess, true, issueInfo.getEncoded());
    }

    /************************************************************************************
     *                          证书 Extension 扩展项生成                               *
     ************************************************************************************/


    /**
     * 获取 主体信息访问 扩展
     *
     * @param ldapUrl 主体证书 ldap 地址
     * @return 主体信息访问扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genSubjectInfoAccessExtension(String ldapUrl) throws IOException {
        ASN1OctetString instance = new DEROctetString(ldapUrl.getBytes());
        SubjectInformationAccess subjectInfo = new SubjectInformationAccess(new AccessDescription(
                SubjectInformationAccess.id_ad_caRepository,
                new GeneralName(GeneralName.uniformResourceIdentifier, instance)));
        return new Extension(Extension.subjectInfoAccess, true, subjectInfo.getEncoded());
    }

    /**
     * 获取 一般 自签名根证书 密钥用途 扩展
     *
     * @return 一般自签名根证书 密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genRootCertKeyUsageExtension() throws IOException {
        KeyUsage keyUsage = new KeyUsage(KeyUsage.cRLSign | KeyUsage.keyCertSign);
        return new Extension(Extension.keyUsage, true, keyUsage.getEncoded());
    }

    /**
     * 获取 一般签名证书 密钥用途 扩展
     *
     * @return 一般签名证书 密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genSignatureCertKeyUsageExtension() throws IOException {
        KeyUsage keyUsage = new KeyUsage(KeyUsage.digitalSignature | KeyUsage.nonRepudiation);
        return new Extension(Extension.keyUsage, true,  keyUsage.getEncoded());
    }

    /**
     * 获取 OCSP 签名证书 扩展密钥用途 扩展
     *
     * @return OCSP 签名证书 扩展密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genOCSPCertExtendedKeyUsageExtension() throws IOException {
        return new Extension(Extension.extendedKeyUsage, true, new ExtendedKeyUsage(KeyPurposeId.id_kp_OCSPSigning).getEncoded());
    }

    /**
     * 获取 一般加密证书 密钥用途 扩展
     *
     * @return 一般加密证书 密钥用途 扩展
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genEncryptCertKeyUsageExtension() throws IOException {
        KeyUsage keyUsage = new KeyUsage(KeyUsage.keyEncipherment | KeyUsage.dataEncipherment | KeyUsage.keyAgreement);
        return new Extension(Extension.keyUsage, true, keyUsage.getEncoded());
    }

    /**
     * 获取 签发者 公钥信息 扩展
     *
     * @param rootCert 签发者证书
     * @return 签发者 公钥信息 扩展
     * @throws IOException                  获取二进制编码异常
     * @throws CertificateEncodingException 生成 签发者公钥信息失败
     * @throws NoSuchAlgorithmException     org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils 扩展工具初始化失败
     */
    public static Extension genAuthorityKeyIdentifierExtension(X509Certificate rootCert) throws NoSuchAlgorithmException, CertificateEncodingException, IOException {
        JcaX509ExtensionUtils extUtils = new JcaX509ExtensionUtils();
        AuthorityKeyIdentifier authorityKeyIdentifier = extUtils.createAuthorityKeyIdentifier(rootCert);
        return new Extension(Extension.authorityKeyIdentifier, false, authorityKeyIdentifier.getEncoded());
    }

    /**
     * 获取 主体 公钥信息 扩展
     *
     * @param publicKey 主体公钥信息
     * @return 主体公钥信息扩展
     * @throws IOException              获取二进制编码异常
     * @throws NoSuchAlgorithmException org.bouncycastle.cert.jcajce.JcaX509ExtensionUtils 扩展工具初始化失败
     */
    public static Extension genSubjectKeyIdentifierExtension(PublicKey publicKey) throws IOException, NoSuchAlgorithmException {
        JcaX509ExtensionUtils extUtils = new JcaX509ExtensionUtils();
        SubjectKeyIdentifier subjectKeyIdentifier = extUtils.createSubjectKeyIdentifier(publicKey);
        return new Extension(Extension.subjectKeyIdentifier, false, subjectKeyIdentifier.getEncoded());
    }

    /**
     * 获取 CA证书 标识扩展项
     *
     * @param pathLenConstraint 签发路径长度
     * @return CA证书 标识扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genBasicConstraintsExtension(int pathLenConstraint) throws IOException {
        return new Extension(Extension.basicConstraints, true, new BasicConstraints(pathLenConstraint).getEncoded());
    }

    /**
     * 获取 全量 CRL 发布点扩展项
     *
     * @param crlLdapUrl 全量 CRL ldap 发布点
     * @return CRL 发布点扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genCRLDistributionPointsExtension(String crlLdapUrl) throws IOException {
        DistributionPointName dpn = genDistributionPointName(crlLdapUrl);
        CRLDistPoint crlDistPoint = new CRLDistPoint(new DistributionPoint[]{new DistributionPoint(dpn, null, null)});
        return new Extension(Extension.cRLDistributionPoints, true, crlDistPoint.getEncoded());
    }

    /**
     * 获取 最新（增量） CRL 发布点扩展项
     *
     * @param crlLdapUrl 最新（增量） CRL ldap 发布点
     * @return CRL 发布点扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genFreshestCRL(String crlLdapUrl) throws IOException {
        DistributionPointName dpn = genDistributionPointName(crlLdapUrl);
        CRLDistPoint crlDistPoint = new CRLDistPoint(new DistributionPoint[]{new DistributionPoint(dpn, null, null)});
        return new Extension(Extension.freshestCRL, true, crlDistPoint.getEncoded());
    }

    /************************************************************************************
     *                              CRL Extension 扩展项生成                            *
     ************************************************************************************/
    /**
     * 如果 CRL 为 ARL ，则需要添加此扩展项标识 该 CRL 为 ARL
     *
     * @param arlLdapUrl ARL 对应的 LDAP 存储地址 使用在分块CRL中，必须与证书中发布点扩展值一致
     * @return 标识 ARL 扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genARLExtension(String arlLdapUrl) throws IOException {
        DistributionPointName dpn = genDistributionPointName(arlLdapUrl);
        ReasonFlags reasonFlags = new ReasonFlags(ReasonFlags.cACompromise);
        IssuingDistributionPoint point = new IssuingDistributionPoint(dpn, true, true, reasonFlags, true, true);
        return new Extension(Extension.issuingDistributionPoint, true, point.getEncoded());
    }

    /**
     * 如果 CRL 为 DRL ，则需要添加此扩展项标识 该 CRL 为 DRL
     *
     * @param drlNumber DRL 对应的序列号
     * @return 标识 DRL 扩展项
     * @throws IOException 获取二进制编码异常
     */
    public static Extension genDRLExtension(int drlNumber) throws IOException {
        CRLNumber basecrlnum = new CRLNumber(BigInteger.valueOf(drlNumber));
        return new Extension(Extension.deltaCRLIndicator, true, basecrlnum.getEncoded());
    }


    public static DistributionPointName genDistributionPointName(String ldapUrl) {
        DEROctetString instance = new DEROctetString(ldapUrl.getBytes());
        GeneralName gn = new GeneralName(GeneralName.uniformResourceIdentifier, instance);
        GeneralNames gns = new GeneralNames(gn);
        return new DistributionPointName(0, gns);
    }

}
