package com.xdja.pki.gmssl.x509.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.GMSSLSM2KeyUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancContentSigner;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancContentVerifierProvider;
import com.xdja.pki.gmssl.operator.utils.*;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.RFC4519Style;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.jce.interfaces.ECPublicKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.ContentVerifier;
import org.bouncycastle.operator.ContentVerifierProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.PKCS10CertificationRequestBuilder;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.security.*;
import java.security.spec.InvalidKeySpecException;

import static com.xdja.pki.gmssl.core.utils.GMSSLX509Utils.readCertificateFromPEM;
import static com.xdja.pki.gmssl.core.utils.GMSSLX509Utils.writePEM;

public class GMSSLP10Utils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLP10Utils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * 生成P10
     *
     * @param subject     主体DN
     * @param publicKey   公钥
     * @param privateKey  签名私钥
     * @param algorithm   签名算法
     * @param isWithParam 是否需要签名扩展参数 默认填写false，仅支持卡斯柯项目SM2WithSM3签名填写true
     * @return
     * @throws OperatorCreationException
     */
    public static PKCS10CertificationRequest generateP10(String subject, PublicKey publicKey, PrivateKey privateKey, String algorithm, boolean isWithParam) throws OperatorCreationException {
        return generateP10(new X500Name(RFC4519Style.INSTANCE, subject), publicKey, privateKey, algorithm, isWithParam);
    }

    public static PKCS10CertificationRequest generateP10(X500Name subject, PublicKey publicKey, PrivateKey privateKey, String algorithm, boolean isWithParam) throws OperatorCreationException {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivate = (SdfPrivateKey) privateKey;
                return generateP10SignByPcie(subject, publicKey, pciePrivate.getIndex(), pciePrivate.getStringPassword(), algorithm);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivate = (SdfPrivateKey) privateKey;
                return generateP10SignByYunhsm(subject, publicKey, hsmPrivate.getIndex(), hsmPrivate.getStringPassword(), algorithm);
            case MINI_PCI_E:
                return generateP10ByMiniPcie(subject, publicKey, privateKey, algorithm, isWithParam);
            case SANC_HSM:
                return generateP10BySanc(subject, publicKey, privateKey, algorithm, isWithParam);
            case BC:
            default:
                return generateP10SignByBC(subject, publicKey, privateKey, algorithm);
        }
    }

    /**
     * 验签P10
     *
     * @param pkcs10CertificationRequest P10证书请求
     * @return
     * @throws Exception
     */
    public static boolean verifyP10(PKCS10CertificationRequest pkcs10CertificationRequest) throws Exception {
        SubjectPublicKeyInfo subjectPublicKeyInfo = pkcs10CertificationRequest.getSubjectPublicKeyInfo();
        PublicKey publicKey = GMSSLX509Utils.convertSM2PublicKey(subjectPublicKeyInfo);
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                ContentVerifierProvider pcieContentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByPcie(publicKey);
                return pkcs10CertificationRequest.isSignatureValid(pcieContentVerifierProvider);
            case XDJA_HSM:
                ContentVerifierProvider hsmContentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByYunHsm(publicKey);
                return pkcs10CertificationRequest.isSignatureValid(hsmContentVerifierProvider);
            case MINI_PCI_E:
                return verifyP10ByMiniPcie(pkcs10CertificationRequest, publicKey);
            case SANC_HSM:
                return verifyP10BySanc(pkcs10CertificationRequest, publicKey);
            case BC:
            default:
                ContentVerifierProvider bcContentVerifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierByBC(publicKey);
                return pkcs10CertificationRequest.isSignatureValid(bcContentVerifierProvider);
        }
    }

    private static PKCS10CertificationRequest generateP10BySanc(String subject, PublicKey publicKey, PrivateKey privateKey, String algorithm, boolean isWithParam) throws OperatorCreationException {
        GMSSLSancContentSigner contentSigner = new GMSSLSancContentSigner(algorithm, privateKey, isWithParam);
        return generateP10(subject, contentSigner, publicKey);
    }

    private static PKCS10CertificationRequest generateP10BySanc(X500Name subject, PublicKey publicKey, PrivateKey privateKey, String algorithm, boolean isWithParam) throws OperatorCreationException {
        GMSSLSancContentSigner contentSigner = new GMSSLSancContentSigner(algorithm, privateKey, isWithParam);
        return generateP10(subject, contentSigner, publicKey);
    }

    private static boolean verifyP10BySanc(PKCS10CertificationRequest pkcs10CertificationRequest, PublicKey publicKey) throws Exception {
        GMSSLSancContentVerifierProvider contentVerifierProvider = new GMSSLSancContentVerifierProvider(publicKey);
        return pkcs10CertificationRequest.isSignatureValid(contentVerifierProvider);
    }

    private static PKCS10CertificationRequest generateP10ByMiniPcie(String subject, PublicKey publicKey, PrivateKey privateKey, String algorithm, boolean isWithParam) throws OperatorCreationException {
        GMSSLXkfContentSigner contentSigner = new GMSSLXkfContentSigner(algorithm, privateKey, isWithParam);
        return generateP10(subject, contentSigner, publicKey);
    }

    private static PKCS10CertificationRequest generateP10ByMiniPcie(X500Name subject, PublicKey publicKey, PrivateKey privateKey, String algorithm, boolean isWithParam) throws OperatorCreationException {
        GMSSLXkfContentSigner contentSigner = new GMSSLXkfContentSigner(algorithm, privateKey, isWithParam);
        return generateP10(subject, contentSigner, publicKey);
    }

    private static boolean verifyP10ByMiniPcie(PKCS10CertificationRequest pkcs10CertificationRequest, PublicKey publicKey) throws Exception {
        GMSSLXkfContentVerifierProvider contentVerifierProvider = new GMSSLXkfContentVerifierProvider(publicKey);
        return pkcs10CertificationRequest.isSignatureValid(contentVerifierProvider);
    }

    /************************************************************************************
     *                                 P10 生成 解析                                    *
     ************************************************************************************/
    /**
     * 生成 P10 证书申请 使用 BC 签名
     *
     * @param subject    主体 string DN
     * @param publicKey  主体公钥信息
     * @param privateKey 主体私钥信息
     * @param algorithm  签名算法 GMSSLSignatureAlgorithm 枚举
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10SignByBC(String subject, PublicKey publicKey, PrivateKey privateKey, String algorithm) throws OperatorCreationException {
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, privateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    public static PKCS10CertificationRequest generateP10SignByBC(X500Name subject, PublicKey publicKey, PrivateKey privateKey, String algorithm) throws OperatorCreationException {
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByBC(algorithm, privateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    /**
     * 生成 P10 证书申请 使用 密码机 签名
     *
     * @param subject            主体 string DN
     * @param publicKey          主体公钥信息
     * @param privateKeyIndex    主体私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param privateKeyPassword 主体私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param algorithm          签名算法 GMSSLSignatureAlgorithm 枚举
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10SignByYunhsm(
            String subject, PublicKey publicKey,
            int privateKeyIndex, String privateKeyPassword, String algorithm
    ) throws OperatorCreationException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateP10(subject, publicKey, sdfPrivateKey, algorithm, false);
        }
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, sdfPrivateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    public static PKCS10CertificationRequest generateP10SignByYunhsm(
            X500Name subject, PublicKey publicKey,
            int privateKeyIndex, String privateKeyPassword, String algorithm
    ) throws OperatorCreationException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateP10(subject, publicKey, sdfPrivateKey, algorithm, false);
        }
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByYunhsm(algorithm, sdfPrivateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    /**
     * 生成 P10 证书申请  使用 PCIE卡 签名
     *
     * @param subject            主体 string DN
     * @param publicKey          主体公钥信息
     * @param privateKeyIndex    主体私钥信息 使用 PCIE YUNHSM 时 密钥访问索引
     * @param privateKeyPassword 主体私钥信息 使用 PCIE YUNHSM 时 密钥访问控制密码
     * @param algorithm          签名算法 GMSSLSignatureAlgorithm 枚举
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws OperatorCreationException P10 签名异常
     */
    public static PKCS10CertificationRequest generateP10SignByPcie(
            String subject, PublicKey publicKey,
            int privateKeyIndex, String privateKeyPassword, String algorithm
    ) throws OperatorCreationException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByPcie(algorithm, sdfPrivateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    public static PKCS10CertificationRequest generateP10SignByPcie(
            X500Name subject, PublicKey publicKey,
            int privateKeyIndex, String privateKeyPassword, String algorithm
    ) throws OperatorCreationException {
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        ContentSigner contentSigner = GMSSLContentSignerUtils.generateContentSignerByPcie(algorithm, sdfPrivateKey);
        return generateP10(subject, contentSigner, publicKey);
    }

    /**
     * 生成 P10 证书申请
     *
     * @param subject       主体
     * @param contentSigner 签名者
     * @param publicKey     公钥
     * @return P10
     * @throws OperatorCreationException
     */
    public static PKCS10CertificationRequest generateP10(
            String subject, ContentSigner contentSigner, PublicKey publicKey
    ) throws OperatorCreationException {
        PKCS10CertificationRequestBuilder requestBuilder = new JcaPKCS10CertificationRequestBuilder(new X500Name(subject), publicKey);

        PKCS10CertificationRequest pkcs10 = requestBuilder.build(contentSigner);
//        GMSSLByteArrayUtils.printHexBinary(logger, "generate p10 signature", pkcs10.getSignature());

        return pkcs10;
    }

    public static PKCS10CertificationRequest generateP10(
            X500Name subject, ContentSigner contentSigner, PublicKey publicKey
    ) throws OperatorCreationException {
        PKCS10CertificationRequestBuilder requestBuilder = new JcaPKCS10CertificationRequestBuilder(subject, publicKey);

        PKCS10CertificationRequest pkcs10 = requestBuilder.build(contentSigner);
//        GMSSLByteArrayUtils.printHexBinary(logger, "generate p10 signature", pkcs10.getSignature());

        return pkcs10;
    }

    /**
     * P10 证书申请 编码成二进制
     *
     * @param pkcs10CertificationRequest org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @return 二进制编码
     * @throws IOException 编码异常
     */
    public static byte[] encodeP10(PKCS10CertificationRequest pkcs10CertificationRequest) throws IOException {
        return pkcs10CertificationRequest.getEncoded();
    }


    /**
     * P10 证书申请解码
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10(String base64P10) throws IOException {
        return new PKCS10CertificationRequest(GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10(byte[] p10) throws IOException {
        return new PKCS10CertificationRequest(p10);
    }

    /**
     * P10 证书申请解码 使用 BC 验签
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByBC(String base64P10) throws IOException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(GMSSLByteArrayUtils.base64Decode(base64P10));
        decodeP10VerifyByBC(pkcs10CertificationRequest);
        return new PKCS10CertificationRequest(GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码 使用 BC 验签
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByBC(byte[] p10) throws IOException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(p10);
        decodeP10VerifyByBC(pkcs10CertificationRequest);
        return new PKCS10CertificationRequest(p10);
    }

    private static void decodeP10VerifyByBC(PKCS10CertificationRequest pkcs10CertificationRequest) throws IOException {
        SubjectPublicKeyInfo subjectPublicKeyInfo = pkcs10CertificationRequest.getSubjectPublicKeyInfo();
        boolean verify;
        try {
            ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(subjectPublicKeyInfo);
            verify = pkcs10CertificationRequest.isSignatureValid(verifierProvider);
        } catch (PKCSException | OperatorCreationException e) {
            throw new IOException("verify error", e);
        }
        if (!verify) {
            logger.info("pkcs10 verify={}", verify);
            throw new IOException("can not verify");
        }
    }

    /**
     * P10 证书申请解码 使用 密码机 验签
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByYunhsm(String base64P10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.YUNHSM, GMSSLByteArrayUtils.base64Decode(base64P10));
    }

    /**
     * P10 证书申请解码 使用 密码机 验签
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByYunhsm(byte[] p10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.YUNHSM, p10);
    }


    /**
     * P10 证书申请解码 使用 PCIE 验签
     *
     * @param base64P10 Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByPCIE(String base64P10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.PCIE, GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码 使用 PCIE 验签
     *
     * @param p10 二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyByPCIE(byte[] p10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        return decodeP10VerifyBySDF(SdfCryptoType.PCIE, p10);
    }


    /**
     * P10 证书申请解码 使用 SDF 验签
     *
     * @param sdfCryptoType SDF 密码算法类型 密码机 PCIE
     * @param base64P10     Base64编码的  二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyBySDF(SdfCryptoType sdfCryptoType, String base64P10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(base64P10);
        decodeP10VerifyBySDF(pkcs10CertificationRequest, sdfCryptoType);
        return new PKCS10CertificationRequest(GMSSLByteArrayUtils.base64Decode(base64P10));
    }


    /**
     * P10 证书申请解码 使用 SDF 验签
     *
     * @param sdfCryptoType SDF 密码算法类型 密码机 PCIE
     * @param p10           二进制编码 P10 证书申请
     * @return org.bouncycastle.pkcs.PKCS10CertificationRequest P10 证书申请
     * @throws IOException 解码异常
     */
    public static PKCS10CertificationRequest decodeP10VerifyBySDF(SdfCryptoType sdfCryptoType, byte[] p10) throws IOException, SdfSDKException, InvalidKeySpecException, NoSuchAlgorithmException {
        PKCS10CertificationRequest pkcs10CertificationRequest = decodeP10(p10);
        decodeP10VerifyBySDF(pkcs10CertificationRequest, sdfCryptoType);
        return new PKCS10CertificationRequest(p10);
    }

    private static void decodeP10VerifyBySDF(PKCS10CertificationRequest pkcs10CertificationRequest, SdfCryptoType sdfCryptoType) throws InvalidKeySpecException, NoSuchAlgorithmException, IOException, SdfSDKException {
        SubjectPublicKeyInfo subjectPublicKeyInfo = pkcs10CertificationRequest.getSubjectPublicKeyInfo();
        PublicKey publicKey = GMSSLX509Utils.convertSM2PublicKey(subjectPublicKeyInfo);
        ContentVerifierProvider provider = new ContentVerifierProvider() {
            @Override
            public boolean hasAssociatedCertificate() {
                return false;
            }

            @Override
            public X509CertificateHolder getAssociatedCertificate() {
                return null;
            }

            @Override
            public ContentVerifier get(AlgorithmIdentifier algorithm) throws OperatorCreationException {
                return GMSSLContentVerifierUtils.generateContentVerifierBySdf(sdfCryptoType, algorithm, publicKey);
            }
        };
        boolean verify;
        try {
            verify = pkcs10CertificationRequest.isSignatureValid(provider);
        } catch (PKCSException e) {
            throw new SdfSDKException("verify error", e);
        }
        if (!verify) {
            throw new IOException("can not verify");
        }
    }


    public static void writeP10ToFile(String path, String name, Object object) throws Exception {
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        String filename = path + File.separator + name + ".p10";
        FileWriter writer = new FileWriter(filename);
        writePEM(object, writer);
        writer.close();
    }
}
