package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.struct.key.IRSArefPublicKey;
import com.sansec.devicev4.gb.struct.key.rsa.RSArefKeyPair;
import com.xdja.pcie.base.RSAPrivateKey;
import com.xdja.pcie.base.RSAPublicKey;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfRSAKey;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConverUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancKeyTypeEnum;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPrivateKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRSAPublicKey;
import com.xdja.pki.gmssl.sdf.bean.SdfRsaKeyPair;
import com.xdja.pki.gmssl.sdf.pcie.PcieSdfSDK;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDK;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.jcajce.provider.asymmetric.rsa.BCRSAPrivateCrtKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.BigIntegers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPrivateCrtKeySpec;
import java.security.spec.RSAPrivateKeySpec;
import java.security.spec.RSAPublicKeySpec;

public class GMSSLRSAKeyUtils {

    public static final String RSA_ALGORITHM_NAME = "RSA";
    public static final int RSA_KEY_SIZE_1024 = 1024;
    public static final int RSA_KEY_SIZE_2048 = 2048;
    public static final int RSA_KEY_SIZE_3072 = 3072;
    public static final int RSA_KEY_SIZE_4096 = 4096;

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    private static Logger logger = LoggerFactory.getLogger(GMSSLRSAKeyUtils.class);
    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/
    /**
     * @param keySize 秘钥长度   参见定义 RSA_KEY_SIZE_1024  RSA_KEY_SIZE_2048
     * @return 密钥对
     * @throws Exception
     */
    public static KeyPair generateRSAKeyPair(int keySize) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return generateKeyPairBySdf(SdfCryptoType.YUNHSM, keySize);
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                RSArefKeyPair rsArefKeyPair = cryptConnection.generateRSAKeyPair(keySize);
                return new KeyPair(GMSSLSancConverUtils.converRSAPublicKey(rsArefKeyPair.getPublicKey()),
                        GMSSLSancConverUtils.converRSAPrivateKey(rsArefKeyPair.getPrivateKey()));
            case MINI_PCI_E:
            case BC:
            default:
                return generateKeyPairByBC(keySize);
        }
    }


    /**
     * 向硬件写入密钥对 只支持PCIE卡
     *
     * @param index    索引
     * @param password 密码
     * @param keyPair  密钥对
     * @return 公钥
     */
    public static boolean writeKeyPairToHardWare(int index, String password, KeyPair keyPair) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                PcieSdfSDK pcieSdfSDK = new PcieSdfSDK();
                BCRSAPrivateCrtKey rsaPrivateCrtKey = (BCRSAPrivateCrtKey) keyPair.getPrivate();
                boolean b = GMSSLRSAKeyUtils.getPrivateKeyAccessRightFromHardware(index, password);
                if (!b) {
                    logger.error("获取私钥访问权限失败，两码校验未通过，index={}，password={}", index, password);
                    return false;
                }
                try {
                    pcieSdfSDK.importRSAPrivateKey(RSAPrivateKey.getInstance(rsaPrivateCrtKey), index, password.getBytes(), false);
                    pcieSdfSDK.importRSAPublicKey(RSAPublicKey.getInstance(rsaPrivateCrtKey), index, false);
                    pcieSdfSDK.importRSAPrivateKey(RSAPrivateKey.getInstance(rsaPrivateCrtKey), index, password.getBytes(), true);
                    pcieSdfSDK.importRSAPublicKey(RSAPublicKey.getInstance(rsaPrivateCrtKey), index, true);
                    return true;
                } catch (Exception e) {
                    logger.error("导入加密密钥对失败", e);
                    return false;
                }
            default:
                throw new Exception("un support write RSA KeyPair ToHardWare  with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }


    /**
     * 生成 SDF 相关的 RSA 私钥
     *
     * @param index    密钥访问索引
     * @param password 密钥访问控制密码
     * @return SDF 相关的 RSA私钥
     */
    public static SdfPrivateKey genSdfPrivateKey(int index, String password) {
        return new SdfPrivateKey(index, password);
    }

    /**
     * 从硬件获取私钥访问权限 仅支持 XDJA_HSM PCI_E
     *
     * @param index    秘钥索引
     * @param password 访问控制码
     * @return
     */
    public static boolean getPrivateKeyAccessRightFromHardware(int index, String password) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
            case PCI_E:
                return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
            default:
                throw new Exception("the method un support the crypto type " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    public static PublicKey getPublicKeyFromHardware(int index, boolean isEncryptKey) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                if (isEncryptKey) {
                    return getEncryptPublicKeyByPcie(index);
                } else {
                    return getSignPublicKeyByPcie(index);
                }
            case XDJA_HSM:
                if (isEncryptKey) {
                    return getEncryptPublicKeyByYunhsm(index);
                } else {
                    return getSignPublicKeyByYunhsm(index);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                IRSArefPublicKey rsaPublicKey;
                if (isEncryptKey) {
                    rsaPublicKey = cryptConnection.getRSAPublicKey(index, GMSSLSancKeyTypeEnum.ENCRYPT.getKeyType());
                } else {
                    rsaPublicKey = cryptConnection.getRSAPublicKey(index, GMSSLSancKeyTypeEnum.SIGN.getKeyType());
                }
                return GMSSLSancConverUtils.converRSAPublicKey(rsaPublicKey);
            case MINI_PCI_E:
            case BC:
            default:
                throw new Exception("un support the method with the crypto type " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }
    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/

    /************************************************************************************
     *                                      密钥对相关                                  *
     ************************************************************************************/
    /**
     * 生成 RSA 密钥对
     *
     * @param keySize 密钥长度
     * @return RSA 密钥对
     * @throws NoSuchProviderException  找不到相应的 Provider
     * @throws NoSuchAlgorithmException 找不到相应的算法
     */
    public static KeyPair generateKeyPairByBC(int keySize) throws NoSuchProviderException, NoSuchAlgorithmException {
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance(RSA_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
        kpGen.initialize(keySize);
        return kpGen.generateKeyPair();
    }

    /**
     * 生成 RSA 密钥对
     *
     * @param keySize 密钥长度
     * @return RSA 密钥对
     * @throws NoSuchProviderException  找不到相应的 Provider
     * @throws NoSuchAlgorithmException 找不到相应的算法
     * @throws InvalidKeySpecException  构建公钥或私钥失败
     * @throws SdfSDKException
     */
    public static KeyPair generateKeyPairByYunHsm(int keySize) throws Exception {
        return generateKeyPairBySdf(SdfCryptoType.YUNHSM, keySize);
    }

    public static KeyPair generateKeyPairBySdf(SdfCryptoType sdfCryptoType, int keySize) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateRSAKeyPair(keySize);
        }
        SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
        sdfSDK.init();
        SdfRsaKeyPair sdfRsaKeyPair = sdfSDK.generateKeyPairRsa(keySize);
        sdfSDK.release();
        SdfRSAPrivateKey rsaPrivateKey = sdfRsaKeyPair.getSdfRSAPrivateKey();
        RSAPrivateCrtKeySpec spec = new RSAPrivateCrtKeySpec(BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getM()),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getE()),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getD()),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPrime()[0]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPrime()[1]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPexp()[0]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getPexp()[1]),
                BigIntegers.fromUnsignedByteArray(rsaPrivateKey.getCoef()));
        KeyFactory keyFactory = KeyFactory.getInstance(RSA_ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
        PrivateKey privateKey = keyFactory.generatePrivate(spec);
        SdfRSAPublicKey rsaPublicKey = sdfRsaKeyPair.getSdfRSAPublicKey();
        RSAPublicKeySpec spec1 = new RSAPublicKeySpec(BigIntegers.fromUnsignedByteArray(rsaPublicKey.getM()),
                BigIntegers.fromUnsignedByteArray(rsaPublicKey.getE()));//存储的就是这两个大整形数
        PublicKey pk = keyFactory.generatePublic(spec1);
        return new KeyPair(pk, privateKey);
    }


    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByYunhsm(int index) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return getPublicKeyFromHardware(index, false);
        }
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfRSAKey.exportSignPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByPcie(int index) throws Exception {
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfRSAKey.exportSignPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByYunhsm(int index) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return getPublicKeyFromHardware(index, true);
        }
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.YUNHSM);
        PublicKey publicKey = sdfRSAKey.exportEncryptPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByPcie(int index) throws Exception {
        SdfRSAKey sdfRSAKey = new SdfRSAKey(SdfCryptoType.PCIE);
        PublicKey publicKey = sdfRSAKey.exportEncryptPublicKey(index);
        sdfRSAKey.release();
        return publicKey;
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.YUNHSM, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  PCIE卡
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromPCIE(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean checkPrivateKeyAccessRightFromSdf(SdfCryptoType sdfCryptoType, int index, String password) {
        try {
            SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
            sdfSDK.init();
            boolean result = sdfSDK.checkPrivateKeyAccessRight(index, password.getBytes());
            sdfSDK.release();
            return result;
        } catch (SdfSDKException e) {
            return false;
        }
    }
}
