package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.struct.key.IRSArefPublicKey;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLRSAUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConverUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancDigestPreProcessUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancKeyTypeEnum;
import com.xdja.pki.gmssl.operator.utils.GMSSLPKCS1Encodeing;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;

import java.security.*;
import java.security.interfaces.RSAPublicKey;


public class GMSSLRSASignUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/
    /**
     * RSA签名
     *
     * @param privateKey    签名私钥
     * @param data          签名原文
     * @param signAlgorithm 签名算法
     * @return 签名数据
     * @throws Exception
     */
    public static byte[] sign(PrivateKey privateKey, byte[] data, String signAlgorithm) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivateKey = (SdfPrivateKey) privateKey;
                return signBySdf(SdfCryptoType.PCIE, signAlgorithm, pciePrivateKey.getIndex(), pciePrivateKey.getStringPassword(), data);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivateKey = (SdfPrivateKey) privateKey;
                return signBySdf(SdfCryptoType.YUNHSM, signAlgorithm, hsmPrivateKey.getIndex(), hsmPrivateKey.getStringPassword(), data);
            case SANC_HSM:
                return signBySancHsm(privateKey, data, signAlgorithm);
            case BC:
            case MINI_PCI_E:
            default:
                return signByBC(signAlgorithm, privateKey, data);
        }
    }

    /**
     * RSA验签
     *
     * @param publicKey     验签公钥
     * @param data          验签原文
     * @param sign          签名数据
     * @param signAlgorithm 签名算法
     * @return
     * @throws Exception
     */
    public static boolean verify(PublicKey publicKey, byte[] data, byte[] sign, String signAlgorithm) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return verifyBySdf(SdfCryptoType.PCIE, signAlgorithm, publicKey, data, sign);
            case XDJA_HSM:
                return verifyBySdf(SdfCryptoType.YUNHSM, signAlgorithm, publicKey, data, sign);
            case SANC_HSM:
                return verifyBySancHsm(publicKey, data, sign, signAlgorithm);
            case BC:
            case MINI_PCI_E:
            default:
                return verifyByBC(signAlgorithm, publicKey, data, sign);
        }
    }


    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/
    public static byte[] signBySancHsm(PrivateKey privateKey, byte[] data, String signAlgorithm) throws Exception {
        SdfPrivateKey sancPrivate = (SdfPrivateKey) privateKey;
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        byte[] digest = sancPreProcess(data, signAlgorithm);
        IRSArefPublicKey rsaPublicKey = cryptConnection.getRSAPublicKey(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.SIGN.getKeyType());
        byte[] padding = GMSSLPKCS1Encodeing.encodePrivateBlock(digest, 0, digest.length, rsaPublicKey.getBits());
        byte[] endData = GMSSLByteArrayUtils.changeByteArrayLength(padding, rsaPublicKey.getBits() / 8);
        return cryptConnection.rsaPrivateKeyOperation(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.SIGN.getKeyType(), endData);
    }

    public static boolean verifyBySancHsm(PublicKey publicKey, byte[] data, byte[] sign, String signAlgorithm) throws Exception {
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        byte[] digest = sancPreProcess(data, signAlgorithm);
        IRSArefPublicKey arefPublicKey = GMSSLSancConverUtils.converRSARefPublicKey(publicKey);
        byte[] operation = cryptConnection.rsaPublicKeyOperation(arefPublicKey, sign);
        byte[] normal = GMSSLPKCS1Encodeing.decodeBlock(operation, arefPublicKey.getBits());
        if (GMSSLByteArrayUtils.isEqual(normal, digest)) {
            return true;
        }
        GMSSLByteArrayUtils.printHexBinary(null, "operation", operation);
        GMSSLByteArrayUtils.printHexBinary(null, "digest", digest);
        GMSSLByteArrayUtils.printHexBinary(null, "normal", normal);
        return false;
    }

    private static byte[] sancPreProcess(byte[] data, String signAlgorithm) throws Exception {
        byte[] digest;
        if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA256_WITH_RSA.getSigAlgName())) {
            digest = GMSSLSancDigestPreProcessUtils.sha256Digest(data);
        } else if (signAlgorithm.equalsIgnoreCase(GMSSLSignatureAlgorithm.SHA1_WITH_RSA.getSigAlgName())) {
            digest = GMSSLSancDigestPreProcessUtils.sha1Digest(data);
        } else {
            throw new Exception("can't support the signAlgorithm " + signAlgorithm);
        }
        return digest;
    }
    /************************************************************************************
     *                                      签名验签                                    *
     ************************************************************************************/


    /**
     * @param algorithm  算法类型
     * @param privateKey 签名私钥
     * @param data       原数据
     * @return byte[] 数组
     * @throws Exception
     */
    public static String signByBC(String algorithm, PrivateKey privateKey, String data) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] sign = GMSSLRSAUtils.generateSignature(algorithm, privateKey, dataBytes);
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据 base64字符串
     * @param sign      签名数据 base64字符串
     * @return boolean true ? false
     * @throws Exception
     */
    public static boolean verifyByBC(String algorithm, PublicKey publicKey, String data, String sign) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] signBytes = GMSSLByteArrayUtils.base64Decode(sign);
        return GMSSLRSAUtils.verifySignature(algorithm, publicKey, dataBytes, signBytes);
    }


    /**
     * @param algorithm  算法类型
     * @param privateKey 签名私钥
     * @param data       原数据
     * @return byte[] 数组
     * @throws Exception
     */
    public static byte[] signByBC(String algorithm, PrivateKey privateKey, byte[] data) throws Exception {
        return GMSSLRSAUtils.generateSignature(algorithm, privateKey, data);
    }

    /**
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean true ? false
     * @throws Exception
     */
    public static boolean verifyByBC(String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        return GMSSLRSAUtils.verifySignature(algorithm, publicKey, data, sign);
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               byte[]待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signByYunHsm(String algorithm, int privateKeyIndex, String privateKeyPassword, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return sign(new SdfPrivateKey(privateKeyIndex), data, algorithm);
        }
        return signBySdf(SdfCryptoType.YUNHSM, algorithm, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               String Base64待签名消息
     * @return 二进制 签名数据
     */
    public static String signByYunHsm(String algorithm, int privateKeyIndex, String privateKeyPassword, String data) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] sign;
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            sign = sign(new SdfPrivateKey(privateKeyIndex), dataBytes, algorithm);
        } else {
            sign = signBySdf(SdfCryptoType.YUNHSM, algorithm, privateKeyIndex, privateKeyPassword, dataBytes);
        }
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * 使用 PCIE 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               byte[]待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signByPcie(String algorithm, int privateKeyIndex, String privateKeyPassword, byte[] data) throws Exception {
        return signBySdf(SdfCryptoType.PCIE, algorithm, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 PCIE 进行签名
     *
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               String Base64待签名消息
     * @return 二进制 签名数据
     */
    public static String signByPcie(String algorithm, int privateKeyIndex, String privateKeyPassword, String data) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] sign = signBySdf(SdfCryptoType.PCIE, algorithm, privateKeyIndex, privateKeyPassword, dataBytes);
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * 使用 密码机 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByYunHsm(String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        return verifyBySdf(SdfCryptoType.YUNHSM, algorithm, publicKey, data, sign);
    }

    /**
     * 使用 密码机 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByYunHsm(String algorithm, PublicKey publicKey, String data, String sign) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] signBytes = GMSSLByteArrayUtils.base64Decode(sign);
        return verifyBySdf(SdfCryptoType.YUNHSM, algorithm, publicKey, dataBytes, signBytes);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByPcie(String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        return verifyBySdf(SdfCryptoType.PCIE, algorithm, publicKey, data, sign);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param algorithm 算法类型
     * @param publicKey 签名公钥
     * @param data      原数据
     * @param sign      签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyByPcie(String algorithm, PublicKey publicKey, String data, String sign) throws Exception {
        byte[] dataBytes = GMSSLByteArrayUtils.base64Decode(data);
        byte[] signBytes = GMSSLByteArrayUtils.base64Decode(sign);
        return verifyBySdf(SdfCryptoType.PCIE, algorithm, publicKey, dataBytes, signBytes);
    }


    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param algorithm          算法类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               byte[]待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, String algorithm, int privateKeyIndex, String privateKeyPassword, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return sign(new SdfPrivateKey(privateKeyIndex), data, algorithm);
        }
        AlgorithmIdentifier algorithmIdentifier = new DefaultSignatureAlgorithmIdentifierFinder().find(algorithm);
        SdfRSASigner sdfRSASigner = new SdfRSASigner(sdfCryptoType, algorithmIdentifier);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        SdfRSAKeyParameters rsaKeyParameters = new SdfRSAKeyParameters(sdfPrivateKey);
        sdfRSASigner.init(true, rsaKeyParameters);
        sdfRSASigner.update(data, 0, data.length);
        byte[] sign = sdfRSASigner.generateSignature();
        sdfRSASigner.release();
        return sign;
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param algorithm     算法类型
     * @param publicKey     签名公钥
     * @param data          原数据
     * @param sign          签名数据
     * @return boolean  是否验证通过
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, String algorithm, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return verify(publicKey, data, sign, algorithm);
        }
        AlgorithmIdentifier algorithmIdentifier = new DefaultSignatureAlgorithmIdentifierFinder().find(algorithm);
        SdfRSASigner sdfRSASigner = new SdfRSASigner(sdfCryptoType, algorithmIdentifier);
        SdfRSAKeyParameters rsaKeyParameter = new SdfRSAKeyParameters((RSAPublicKey) publicKey);
        sdfRSASigner.init(false, rsaKeyParameter);
        sdfRSASigner.update(data, 0, data.length);
        boolean verify = sdfRSASigner.verifySignature(sign);
        sdfRSASigner.release();
        return verify;
    }
}
