package com.xdja.pki.gmssl.operator.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentVerifier;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentVerifierProviderBuilder;

import java.security.PublicKey;
import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class GMSSLContentVerifierUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                             ContentVerifier 生成                                   *
     ************************************************************************************/

    /**
     * BC 生成 ContentVerifier 内容验签实体类
     *
     * @param algorithm 签名算法 GMSSLSignatureAlgorithm 枚举 支持 SM3withSM3
     * @param publicKey 验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByBC(String algorithm, PublicKey publicKey) throws OperatorCreationException {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        return verifierProvider.get(GMSSLSignatureAlgorithm.convertAlgorithmIdentifier(algorithm, false));
    }

    public static ContentVerifierProvider generateContentVerifierProviderByBC(PublicKey publicKey) throws OperatorCreationException {
        return new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
    }

    public static ContentVerifierProvider generateContentVerifierProviderByBC(SubjectPublicKeyInfo subjectPublicKeyInfo) throws OperatorCreationException {
        return new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(subjectPublicKeyInfo);
    }

    /**
     * BC 生成 ContentVerifier 内容验签实体类
     *
     * @param algorithmIdentifier 签名算法 org.bouncycastle.asn1.x509.AlgorithmIdentifier 签名算法
     * @param publicKey           验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByBC(AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey) throws OperatorCreationException {
        ContentVerifierProvider verifierProvider = new JcaContentVerifierProviderBuilder().setProvider(BouncyCastleProvider.PROVIDER_NAME).build(publicKey);
        return verifierProvider.get(algorithmIdentifier);
    }

    /**
     * 密码机 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey 验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByYunhsm(String algorithm, PublicKey publicKey) throws OperatorCreationException {
        return generateContentVerifierBySdf(SdfCryptoType.YUNHSM, algorithm, publicKey);
    }

    /**
     * PCIE 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey 验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierByPcie(String algorithm, PublicKey publicKey) throws OperatorCreationException {
        return generateContentVerifierBySdf(SdfCryptoType.PCIE, algorithm, publicKey);
    }

    /**
     * SDF 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey 验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierBySdf(SdfCryptoType sdfCryptoType, String algorithm, PublicKey publicKey) throws OperatorCreationException {
        AlgorithmIdentifier algorithmIdentifier = GMSSLSignatureAlgorithm.convertAlgorithmIdentifier(algorithm, false);
        return generateContentVerifierBySdf(sdfCryptoType, algorithmIdentifier, publicKey);
    }

    /**
     * SDF 生成 ContentVerifier 内容验签实体类
     *
     * @param publicKey 验签者公钥
     * @return ContentVerifier 内容验签实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentVerifier generateContentVerifierBySdf(SdfCryptoType sdfCryptoType, AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey) throws OperatorCreationException {
        try {
            return new SdfContentVerifier(sdfCryptoType, algorithmIdentifier, publicKey);
        } catch (SdfSDKException e) {
            throw new OperatorCreationException(e.getMessage());
        }
    }

    public static ContentVerifier generateContentVerifierBySdfHSM(SdfCryptoType sdfCryptoType, AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey) throws OperatorCreationException {
        try {
            return new SdfContentVerifier(sdfCryptoType, algorithmIdentifier, publicKey, false);
        } catch (SdfSDKException e) {
            throw new OperatorCreationException(e.getMessage());
        }
    }

}
