package com.xdja.pki.gmssl.x509.utils.bean;

import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x9.X9ObjectIdentifiers;
import org.bouncycastle.operator.DefaultSignatureAlgorithmIdentifierFinder;

public enum GMSSLSignatureAlgorithm {
    // sm2 sm3
    SM3_WITH_SM2(GMObjectIdentifiers.sm2sign_with_sm3, "SM3WITHSM2"),
    // sha1 rsa
    SHA1_WITH_RSA(PKCSObjectIdentifiers.sha1WithRSAEncryption, "SHA1WITHRSA"),

    SHA1_WITH_ECDSA(X9ObjectIdentifiers.ecdsa_with_SHA1, "SHA1WITHECDSA"),
    // sha256 rsa
    SHA256_WITH_RSA(PKCSObjectIdentifiers.sha256WithRSAEncryption, "SHA256WITHRSA"),
    // sha512 rsa
    SHA512_WITH_RSA(PKCSObjectIdentifiers.sha512WithRSAEncryption, "SHA512WITHRSA"),

    SHA256_WITH_ECDSA(X9ObjectIdentifiers.ecdsa_with_SHA256, "SHA256WITHECDSA"),
    ;

    private ASN1ObjectIdentifier aid;
    private String sigAlgName;

    public ASN1ObjectIdentifier getAid() {
        return aid;
    }

    public String getSigAlgName() {
        return sigAlgName;
    }

    GMSSLSignatureAlgorithm(ASN1ObjectIdentifier aid, String sigAlgName) {
        this.aid = aid;
        this.sigAlgName = sigAlgName;
    }

    public AlgorithmIdentifier convertAlgorithmIdentifier() {
        return new DefaultSignatureAlgorithmIdentifierFinder().find(this.sigAlgName);
    }

    public static AlgorithmIdentifier convertAlgorithmIdentifier(String algorithm) {
        return new DefaultSignatureAlgorithmIdentifierFinder().find(algorithm);
    }

    public static AlgorithmIdentifier convertAlgorithmIdentifier(String algorithm, boolean isWithSignParam) {
        GMSSLSignatureAlgorithm gmsslSignatureAlgorithm = convertContentSignatureAlgorithm(algorithm);
        assert gmsslSignatureAlgorithm != null;
        if (isWithSignParam) {
            return new AlgorithmIdentifier(gmsslSignatureAlgorithm.aid, DERNull.INSTANCE);
        } else {
            return new DefaultSignatureAlgorithmIdentifierFinder().find(algorithm);
        }
    }

    public static GMSSLSignatureAlgorithm convertContentSignatureAlgorithm(AlgorithmIdentifier algorithmIdentifier) {
        ASN1ObjectIdentifier algorithm = algorithmIdentifier.getAlgorithm();
        return convertContentSignatureAlgorithm(algorithm);
    }

    public static GMSSLSignatureAlgorithm convertContentSignatureAlgorithm(ASN1ObjectIdentifier aid) {
        if (aid.getId().equalsIgnoreCase(GMObjectIdentifiers.sm2sign_with_sm3.getId())) {
            return SM3_WITH_SM2;
        } else if (aid.getId().equalsIgnoreCase(PKCSObjectIdentifiers.sha1WithRSAEncryption.getId())) {
            return SHA1_WITH_RSA;
        } else if (aid.getId().equalsIgnoreCase(PKCSObjectIdentifiers.sha256WithRSAEncryption.getId())) {
            return SHA256_WITH_RSA;
        } else if (aid.getId().equalsIgnoreCase(PKCSObjectIdentifiers.sha512WithRSAEncryption.getId())) {
            return SHA512_WITH_RSA;
        } else if (aid.getId().equalsIgnoreCase(X9ObjectIdentifiers.ecdsa_with_SHA256.getId())) {
            return SHA256_WITH_ECDSA;
        } else if (aid.getId().equalsIgnoreCase(X9ObjectIdentifiers.ecdsa_with_SHA1.getId())) {
            return SHA1_WITH_ECDSA;
        } else {
            return null;
        }
    }

    public static GMSSLSignatureAlgorithm convertContentSignatureAlgorithm(String sigAlgName) {
        if (sigAlgName.equalsIgnoreCase(SM3_WITH_SM2.getSigAlgName())) {
            return SM3_WITH_SM2;
        } else if (sigAlgName.equalsIgnoreCase(SHA1_WITH_RSA.getSigAlgName())) {
            return SHA1_WITH_RSA;
        } else if (sigAlgName.equalsIgnoreCase(SHA256_WITH_RSA.getSigAlgName())) {
            return SHA256_WITH_RSA;
        } else if (sigAlgName.equalsIgnoreCase(SHA512_WITH_RSA.getSigAlgName())) {
            return SHA512_WITH_RSA;
        } else if (sigAlgName.equalsIgnoreCase(SHA256_WITH_ECDSA.getSigAlgName())) {
            return SHA256_WITH_ECDSA;
        } else if (sigAlgName.equalsIgnoreCase(SHA1_WITH_ECDSA.getSigAlgName())) {
            return SHA1_WITH_ECDSA;
        } else {
            return null;
        }

    }
}
