package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLBCCipherUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfSymmetricKeyParameters;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;

/**
 * @author sxy
 * @description
 * @date 2019年12月25日 18:57
 **/
public class GMSSLAES128ECBEncryptUtils {

    /**
     * SM1 ECB 使用密钥加密密钥进行加密
     *
     * @param kek               密钥加密密钥
     * @param kekIndex          密钥加密密钥对应的密钥索引
     * @param data              需要加密的数据
     * @param paddingType       填充类型
     * @param keyEncryptKeyType 生成密钥加密密钥的加密类型 目前只有SdfAlgIdSymmetric.SGD_SM1_ECB 和 SdfAlgIdSymmetric.SGD_SM4_ECB
     *                          根据生成是调用的方法
     * @return 加密后的数据
     * @throws Exception
     */
    public static byte[] encryptWithKek(byte[] kek, int kekIndex, SdfAlgIdSymmetric keyEncryptKeyType, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptWithKekBySdf(true, SdfCryptoType.YUNHSM, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_AES_ECB, data);
            default:
                throw new IllegalArgumentException("un support kek encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * SM1 ECB 使用密钥加密密钥进行加密
     *
     * @param kek               密钥加密密钥
     * @param kekIndex          密钥加密密钥对应的密钥索引
     * @param data              需要解密的数据
     * @param paddingType       填充类型
     * @param keyEncryptKeyType 生成密钥加密密钥的加密类型 目前只有SdfAlgIdSymmetric.SGD_SM1_ECB 和 SdfAlgIdSymmetric.SGD_SM4_ECB
     *                          根据生成是调用的方法
     * @return
     * @throws Exception
     */
    public static byte[] decryptWithKek(byte[] kek, int kekIndex, SdfAlgIdSymmetric keyEncryptKeyType, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptWithKekBySdf(false, SdfCryptoType.YUNHSM, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_AES_ECB, data);
            default:
                throw new IllegalArgumentException("un support kek encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }


    public static byte[] encrypt(byte[] key, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptBySdf(true, SdfCryptoType.YUNHSM, paddingType, key, SdfAlgIdSymmetric.SGD_AES_ECB, data);
            default:
                switch (paddingType) {
                    case NoPadding:
                        return GMSSLBCCipherUtils.symmetricECBEncrypt(GMSSLBCCipherUtils.AES_ECB_NoPadding, key, data);
                    case PKCS5Padding:
                        return GMSSLBCCipherUtils.symmetricECBEncrypt(GMSSLBCCipherUtils.AES_ECB_PKCS5Padding, key, data);
                    case PKCS7Padding:
                        return GMSSLBCCipherUtils.symmetricECBEncrypt(GMSSLBCCipherUtils.AES_ECB_PKCS7Padding, key, data);
                    default:
                        break;
                }
        }
        throw new Exception("un support the padding type " + paddingType);
    }

    public static byte[] decrypt(byte[] key, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptBySdf(false, SdfCryptoType.YUNHSM, paddingType, key, SdfAlgIdSymmetric.SGD_AES_ECB, data);
            default:
                switch (paddingType) {
                    case NoPadding:
                        return GMSSLBCCipherUtils.symmetricECBDecrypt(GMSSLBCCipherUtils.AES_ECB_NoPadding, key, data);
                    case PKCS5Padding:
                        return GMSSLBCCipherUtils.symmetricECBDecrypt(GMSSLBCCipherUtils.AES_ECB_PKCS5Padding, key, data);
                    case PKCS7Padding:
                        return GMSSLBCCipherUtils.symmetricECBDecrypt(GMSSLBCCipherUtils.AES_ECB_PKCS7Padding, key, data);
                    default:
                        break;
                }
        }
        throw new Exception("un support the padding type " + paddingType);
    }

}
