package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;

import com.sansec.devicev4.gb.struct.key.sm2.SM2refPublicKey;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refSignature;
import com.xdja.SafeKey.Sm2PublicKey;
import com.xdja.SafeKey.bean.MiniPcieIndexEnum;
import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLBCSignUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConverUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancDigestPreProcessUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancKeyTypeEnum;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfSM2SignerUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.*;


/**
 * @ClassName GMSSLECSignUtils
 * @Description TODO
 * @Date 2020/1/13 16:10
 * @Author FengZhen
 */
public class GMSSLECSignUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/
    /**
     * EC签名工具类  SANC预处理为软算法
     *
     * @param privateKey  签名私钥 通过硬件传值SdfPrivate(index , password)
     * @param data        签名原文
     * @param userId      签名的userID 默认不传， 特殊需求传
     * @param signAlgName 签名算法 参考 GMSSLSignatureAlgorithm  MiniPCIE 和 SANC暂时只支持SM2
     * @return
     * @throws Exception
     */
    public static byte[] sign(PrivateKey privateKey, byte[] data, byte[] userId, String signAlgName) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivateKey = (SdfPrivateKey) privateKey;
                return signBySdf(SdfCryptoType.PCIE, userId, pciePrivateKey.getIndex(), pciePrivateKey.getStringPassword(), data, signAlgName);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivateKey = (SdfPrivateKey) privateKey;
                return signBySdf(SdfCryptoType.YUNHSM, userId, hsmPrivateKey.getIndex(), hsmPrivateKey.getStringPassword(), data, signAlgName);
            case MINI_PCI_E:
                // TODO: 2020/5/20 userId 暂不支持
                SdfPrivateKey miniPciePrivateKey = (SdfPrivateKey) privateKey;
                return signByMiniPcie(miniPciePrivateKey.getIndex(), miniPciePrivateKey.getStringPassword(), data, signAlgName);
            case SANC_HSM:
                return signBySancHsm(privateKey, data, userId, signAlgName);
            case BC:
            default:
                if (null != userId && signAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
                    return GMSSLBCSignUtils.sm2SignWithUserId(privateKey, data, userId);
                } else {
                    return GMSSLBCSignUtils.generateSignature(signAlgName, privateKey, data);
                }
        }
    }

    /**
     * EC 验签接口  SANC预处理为软算法
     *
     * @param publicKey   验签公钥
     * @param data        签名原文
     * @param signature   签名值
     * @param userId      签名的userID 默认不传， 特殊需求传
     * @param signAlgName 签名算法 参考 GMSSLSignatureAlgorithm  MiniPCIE 和 SANC暂时只支持SM2
     * @return
     * @throws Exception
     */
    public static boolean verify(PublicKey publicKey, byte[] data, byte[] signature, byte[] userId, String signAlgName) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return verifyBySdf(SdfCryptoType.PCIE, userId, publicKey, data, signature, signAlgName);
            case XDJA_HSM:
                return verifyBySdf(SdfCryptoType.YUNHSM, userId, publicKey, data, signature, signAlgName);
            case MINI_PCI_E:
                // TODO: 2020/5/20 userId 暂不支持
                return verifyByMiniPcie(publicKey, data, signature, signAlgName);
            case SANC_HSM:
                return verifyBySancHsm(publicKey, data, signature, userId, signAlgName);
            case BC:
            default:
                if (null != userId && signAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
                    return GMSSLBCSignUtils.sm2VerifyWithUserId(publicKey, userId, data, signature);
                } else {
                    return GMSSLBCSignUtils.verifySignature(signAlgName, publicKey, data, signature);
                }
        }
    }

    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/

    private static byte[] signBySancHsm(PrivateKey privateKey, byte[] data, byte[] userId, String signAlgName) throws Exception {
        SdfPrivateKey sancPrivate = (SdfPrivateKey) privateKey;
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        if (signAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            //导出公钥
            SM2refPublicKey sm2PublicKey = cryptConnection.getSM2PublicKey(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.SIGN.getKeyType());
            //做预处理
            byte[] sm3Digest = GMSSLSancDigestPreProcessUtils.sm3Digest(GMSSLSancConverUtils.converSM2PublicKey(sm2PublicKey), data, userId);
            //签名
            SM2refSignature sm2refSignature = cryptConnection.sm2Sign(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.SIGN.getKeyType(), sm3Digest);
            return GMSSLSancConverUtils.converSM2Signature(sm2refSignature);
        }
        throw new IllegalArgumentException("sanc hsm interface un support the signAlgName " + signAlgName);
    }

    private static boolean verifyBySancHsm(PublicKey publicKey, byte[] data, byte[] signature, byte[] userId, String signAlgName) throws Exception {
        if (signAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            //做预处理
            byte[] sm3Digest = GMSSLSancDigestPreProcessUtils.sm3Digest(publicKey, data, userId);

            SM2refSignature sm2refSignature = GMSSLSancConverUtils.converSM2RefSignature(signature);
            return GMSSLSancConnectionUtils.getCryptConnection().sm2Verify(GMSSLSancConverUtils.converSM2RefPublicKey(publicKey), sm3Digest, sm2refSignature);
        }
        throw new IllegalArgumentException("sanc hsm interface un support the signAlgName " + signAlgName);
    }

    public static byte[] signByMiniPcie(int index, String password, byte[] data, String signAlgName) throws Exception {
        if (signAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            //calculate sm3 hash
//            byte[] hash = MiniPcieXKFUtils.sm3(data);
//            System.out.println(Hex.toHexString(hash));
            //sm2 sign
            return MiniPcieXKFUtils.sm2SignWithInternalHash(data, MiniPcieIndexEnum.getInfoFromIndex(index).getSignPublicKeyIndex(), MiniPcieIndexEnum.getInfoFromIndex(index).getSignPrivateIndex(), password);
        }
        throw new IllegalArgumentException("mini pcie card interface un support the signAlgName " + signAlgName);
    }

    public static boolean verifyByMiniPcie(PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        if (signAlgName.equalsIgnoreCase(GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName())) {
            //sm2 sign
            Sm2PublicKey sm2PublicKey = new Sm2PublicKey(publicKey);
            return MiniPcieXKFUtils.sm2VerifyWithInternalHash(data, sign, sm2PublicKey);
        }
        throw new IllegalArgumentException("mini pcie card interface un support the signAlgName " + signAlgName);
    }

    /**
     * 使用 BC 进行签名
     *
     * @param privateKey 签名者私钥
     * @param base64Data BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signByBC(PrivateKey privateKey, String base64Data, String signAlgName) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sig = signByBC(privateKey, data, signAlgName);
        return GMSSLByteArrayUtils.base64Encode(sig);
    }

    /**
     * 使用 BC 进行签名
     *
     * @param privateKey 签名者私钥
     * @param data       待签名消息
     * @return 签名数据
     */
    public static byte[] signByBC(PrivateKey privateKey, byte[] data, String signAlgName) throws Exception {
        return GMSSLBCSignUtils.generateSignature(signAlgName, privateKey, data);
    }


    /**
     * 使用 BC 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return BASE64 编码 签名数据
     */
    public static boolean verifyByBC(PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign = GMSSLByteArrayUtils.base64Decode(base64Sign);
        return verifyByBC(publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 BC 进行验签
     *
     * @param publicKey 签名者公钥
     * @param data      签名原文
     * @param sign      签名消息
     * @return 签名数据
     */
    public static boolean verifyByBC(PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        return GMSSLBCSignUtils.verifySignature(signAlgName, publicKey, data, sign);
    }


    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param base64Data         BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data, String signAlgName) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign;
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            sign = sign(new SdfPrivateKey(privateKeyIndex), data, null, signAlgName);
        } else {
            sign = signBySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data, signAlgName);
        }
        return GMSSLByteArrayUtils.base64Encode(sign);
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @param signAlgName        签名算法
     * @return 二进制 签名数据
     */
    public static byte[] signByYunHsm(int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws Exception {
        return signBySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    /**
     * 使用 密码机 进行验证签名
     *
     * @param publicKey   签名者私钥索引
     * @param data        签名原文
     * @param sign        签名数据
     * @param signAlgName 签名算法
     * @return 二进制 签名数据
     */
    public static boolean verifyByYunHsm(PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        return verifyByHSM(null, publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @param signAlgName        签名算法
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws Exception {
        return signBySdf(sdfCryptoType, null, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdfWithUserId(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] userId, byte[] data, String signAlgName) throws Exception {
        return signBySdf(sdfCryptoType, userId, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }


    /**
     * 使用 密码机 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyByYunHsm(PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws Exception {
        return verifyBySdf(SdfCryptoType.YUNHSM, publicKey, base64Data, base64Sign, signAlgName);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyByPcie(PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws Exception {
        return verifyBySdf(SdfCryptoType.PCIE, publicKey, base64Data, base64Sign, signAlgName);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param base64Data    BASE64 编码 签名原文
     * @param base64Sign    BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data, String base64Sign, String signAlgName) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign = GMSSLByteArrayUtils.base64Decode(base64Sign);
        return verifyBySdf(sdfCryptoType, publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param data          签名原文
     * @param sign          签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        return verifyBySdf(sdfCryptoType, null, publicKey, data, sign, signAlgName);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param data          签名原文
     * @param sign          签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdfWithUserId(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] userId, byte[] data, byte[] sign, String signAlgName) throws Exception {
        return verifyBySdf(sdfCryptoType, userId, publicKey, data, sign, signAlgName);
    }


    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return sign(new SdfPrivateKey(privateKeyIndex), data, userId, signAlgName);
        }
        SdfECBaseSigner signer = GMSSLSdfSM2SignerUtils.generateSM2Signer(sdfCryptoType, signAlgName);
        return signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    public static byte[] signByYunHsm(byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return sign(new SdfPrivateKey(privateKeyIndex), data, userId, signAlgName);
        }
        SdfECSigner signer = new SdfECSigner(SdfCryptoType.YUNHSM, signAlgName);
        return signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data, signAlgName);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param signer             签名计算实体类
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfECBaseSigner signer, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data, String signAlgName) throws Exception {
        return GMSSLSM2SignUtils.signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data);
    }

    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return verify(publicKey, data, sign, userId, signAlgName);
        }
        SdfSM2SignerSM3 signer = new SdfSM2SignerSM3(sdfCryptoType, signAlgName);
        return verifyBySdf(signer, userId, publicKey, data, sign);
    }

    public static boolean verifyByHSM(byte[] userId, PublicKey publicKey, byte[] data, byte[] sign, String signAlgName) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return verify(publicKey, data, sign, userId, signAlgName);
        }
        SdfECSigner signer = new SdfECSigner(SdfCryptoType.YUNHSM, signAlgName);
        return verifyBySdf(signer, userId, publicKey, data, sign);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param signer    签名计算实体类
     * @param publicKey 签名者公钥
     * @param data      签名原文
     * @param sign      签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfECBaseSigner signer, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        return GMSSLSM2SignUtils.verifyBySdf(signer, userId, publicKey, data, sign);
    }
}
