package com.xdja.pki.gmssl.crypto.utils;


import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.GBAlgorithmID_SGD;
import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.utils.MiniPcieSymmetricEncryptUtils;
import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfSymmetricKeyParameters;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.jce.provider.BouncyCastleProvider;


import java.security.Security;

import static com.xdja.SafeKey.utils.MiniPcieXKFUtils.sm1;

public class GMSSLSM1CBCEncryptUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * SM1 CBC 使用密钥加密密钥进行加密
     *
     * @param kek               密钥加密密钥
     * @param kekIndex          密钥加密密钥对应的密钥索引
     * @param data              需要加密的数据
     * @param paddingType       填充类型
     * @param iv                初始化向量
     * @param keyEncryptKeyType 生成密钥加密密钥的加密类型 目前只有SdfAlgIdSymmetric.SGD_SM1_ECB 和 目前只有SdfAlgIdSymmetric.SGD_SM4_ECB
     *                          根据生成是调用的方法
     * @return 加密后的数据
     * @throws Exception
     */
    public static byte[] encryptWithKek(byte[] kek, int kekIndex, SdfAlgIdSymmetric keyEncryptKeyType, byte[] data, byte[] iv, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptWithKekBySdf(true, SdfCryptoType.YUNHSM, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);
            default:
                throw new IllegalArgumentException("un support kek encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * SM1 CBC 使用密钥加密密钥进行加密
     *
     * @param kek               密钥加密密钥
     * @param kekIndex          密钥加密密钥对应的密钥索引
     * @param data              需要解密的数据
     * @param paddingType       填充类型
     * @param iv                初始化向量
     * @param keyEncryptKeyType 生成密钥加密密钥的加密类型 目前只有SdfAlgIdSymmetric.SGD_SM1_ECB 和 目前只有SdfAlgIdSymmetric.SGD_SM4_ECB
     *                          根据生成是调用的方法
     * @return
     * @throws Exception
     */
    public static byte[] decryptWithKek(byte[] kek, int kekIndex, SdfAlgIdSymmetric keyEncryptKeyType, byte[] data, byte[] iv, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptWithKekBySdf(false, SdfCryptoType.YUNHSM, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);
            default:
                throw new IllegalArgumentException("un support kek encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }


    /**
     * SM1 CBC 加密接口  该方法不支持软实现
     *
     * @param key         对称加密密钥
     * @param data        原数据
     * @param iv          向量iv
     * @param paddingType padding模式
     * @return 加密数据
     * @throws Exception
     */
    public static byte[] encrypt(byte[] key, byte[] data, byte[] iv, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return sm1SymmetricSdfWithPadding(true, SdfCryptoType.PCIE, paddingType, key, iv, data);
            case XDJA_HSM:
                return sm1SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM, paddingType, key, iv, data);
            case MINI_PCI_E:
                switch (paddingType) {
                    case NoPadding:
                        return sm1(key, data, JNIAPI.CBC_ENCRYPT, iv);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return MiniPcieSymmetricEncryptUtils.sm1CbcWithPkcs5Padding(key, data, iv, true);
                    case SSL3Padding:
                    default:
                        return MiniPcieSymmetricEncryptUtils.sm1CbcWithSslv3Padding(key, data, iv, true);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                switch (paddingType) {
                    case NoPadding:
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM1_CBC, key, iv, data);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        data = GMSSLX509Utils.paddingData(data, true);
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM1_CBC, key, iv, data);
                    case SSL3Padding:
                    default:
                        data = GMSSLX509Utils.paddingData(data, false);
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM1_CBC, key, iv, data);
                }
            case BC:
            default:
                throw new IllegalArgumentException("un support the sm1 cbc encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * SM1 CBC 解密  该方法不支持软实现
     *
     * @param key         对称加密密钥
     * @param data        加密数据
     * @param iv          向量
     * @param paddingType padding填充模式
     * @return
     * @throws Exception
     */
    public static byte[] decrypt(byte[] key, byte[] data, byte[] iv, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return sm1SymmetricSdfWithPadding(false, SdfCryptoType.PCIE, paddingType, key, iv, data);
            case XDJA_HSM:
                return sm1SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM, paddingType, key, iv, data);
            case MINI_PCI_E:
                switch (paddingType) {
                    case NoPadding:
                        return sm1(key, data, JNIAPI.CBC_DECRYPT, iv);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return MiniPcieSymmetricEncryptUtils.sm1CbcWithPkcs5Padding(key, data, iv, false);
                    case SSL3Padding:
                    default:
                        return MiniPcieSymmetricEncryptUtils.sm1CbcWithSslv3Padding(key, data, iv, false);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                switch (paddingType) {
                    case NoPadding:
                        return cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM1_CBC, key, iv, data);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        byte[] decrypt = cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM1_CBC, key, iv, data);
                        return GMSSLX509Utils.removePadding(data, decrypt, true);
                    case SSL3Padding:
                    default:
                        byte[] decryptSsl3 = cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM1_CBC, key, iv, data);
                        return GMSSLX509Utils.removePadding(data, decryptSsl3, false);
                }
            case BC:
            default:
                throw new IllegalArgumentException("un support the sm1 cbc decrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }


    /**
     * 使用 加密机Yunhsm 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithNoPadding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 加密机Yunhsm 进行加密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithPKCS5Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 加密机Yunhsm 进行加密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithPKCS7Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 加密机Yunhsm 进行加密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithSSL3Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 加密机Yunhsm 进行解密 NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithNoPadding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 加密机Yunhsm 进行解密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithPKCS5Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 加密机Yunhsm 进行解密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithPKCS7Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 加密机Yunhsm 进行解密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithSSL3Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 PCIE 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithNoPadding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 PCIE 进行加密 PKCS5Padding
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithPKCS5Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 PCIE 进行加密 PKCS7Padding
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithPKCS7Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 PCIE 进行加密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data   BASE64 编码 待加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithSSL3Padding(String base64Key, String base64Data, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    /**
     * 使用 PCIE 进行解密  NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithNoPadding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 PCIE 进行解密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithPKCS5Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 PCIE 进行解密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithPKCS7Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 PCIE 进行解密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @param base64InitIv BASE64 编码 初始化向量 转换为byte[]数组后长度为16
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithSSL3Padding(String base64Key, String base64Cipher, String base64InitIv) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm1SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64InitIv),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    /**
     * 使用 padding模式填充的加解密公用类 YUNHSM
     *
     * @param forEncryption 是否为加密
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           byte[] 会话密钥 长度为16
     * @param iv            byte[] 初始化向量长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息
     * @return byte[] 编码 明文消息
     */
    public static byte[] sm1SymmetricWithPaddingByYunHsm(boolean forEncryption,
                                                         SdfSymmetricKeyParameters.PaddingType paddingType,
                                                         byte[] key, byte[] iv, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptBySdf(forEncryption, SdfCryptoType.YUNHSM,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);

    }

    /**
     * 使用 padding模式填充的加解密公用类 PCIE
     *
     * @param forEncryption 是否为加密
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           byte[] 会话密钥 长度为16
     * @param iv            byte[] 初始化向量长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息
     * @return byte[] 编码 明文消息
     */
    public static byte[] sm1SymmetricWithPaddingByPCIE(boolean forEncryption,
                                                       SdfSymmetricKeyParameters.PaddingType paddingType,
                                                       byte[] key, byte[] iv, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptBySdf(forEncryption, SdfCryptoType.PCIE,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);

    }

    /**
     * 使用 padding模式填充的加解密公用类
     *
     * @param forEncryption 是否为加密
     * @param sdfCryptoType 加密使用类型 包括加密机YumHsm和PCIE卡
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           byte[] 会话密钥 长度为16
     * @param iv            byte[] 初始化向量长度为16
     * @param data          byte[] 数组 待加密消息或已加密消息
     * @return byte[] 编码 明文消息
     */
    public static byte[] sm1SymmetricSdfWithPadding(boolean forEncryption, SdfCryptoType sdfCryptoType,
                                                    SdfSymmetricKeyParameters.PaddingType paddingType,
                                                    byte[] key, byte[] iv, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            if (forEncryption) {
                return encrypt(key, data, iv, paddingType);
            } else {
                return decrypt(key, data, iv, paddingType);
            }
        }
        return GMSSLSymmetricEncryptUtils.symmetricCBCEncryptBySdf(forEncryption, sdfCryptoType,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM1_CBC, iv, data);

    }
}
