package com.xdja.pki.gmssl.crypto.init;

import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLRSAUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.SecretKey;
import java.security.Key;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.RSAPublicKey;
import java.text.MessageFormat;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * @author: houzhe
 * @date: 2021-11-03
 * @description: HsmServer KeyStore 工具类
 **/
public class GMSSLHsmKeyStoreUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(GMSSLHsmKeyStoreUtils.class);

    /**
     * 解析HSM KeyStore
     * @param keyStore KeyStore
     * @return 结果
     */
    public static Map<String, GMSSLHsmKeyStoreBean> analysisKeyStore(KeyStore keyStore) {
        if (null == keyStore) {
            return null;
        }
        try {
            Map<String, GMSSLHsmKeyStoreBean> map = new HashMap<>();
            Enumeration<String> aliases = keyStore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                if (keyStore.isKeyEntry(alias)) {
                    GMSSLHsmKeyStoreBean bean = new GMSSLHsmKeyStoreBean();
                    Key key = keyStore.getKey(alias, GMSSLHSMConstants.KEYSTORE_PASSWORD.toCharArray());
                    if (key instanceof SecretKey) {
                        bean.setSecretKey((SecretKey) key);
                    } else {
                        PrivateKey privateKey = (PrivateKey) key;
                        String algorithm = privateKey.getAlgorithm();
                        bean.setPrivateKey(privateKey);
                        if ("EC".equalsIgnoreCase(algorithm)) {
                            bean.setPublicKey(GMSSLECUtils.convertPublicKey(privateKey));
                        } else if ("RSA".equalsIgnoreCase(algorithm)) {
                            RSAPrivateCrtKey rsaPrivateKey = (RSAPrivateCrtKey) privateKey;
                            RSAPublicKey rsaPublicKey = GMSSLRSAUtils.convertRSAPublicKey(rsaPrivateKey.getModulus().toByteArray(), rsaPrivateKey.getPublicExponent().toByteArray());
                            bean.setPublicKey(rsaPublicKey);
                        }
                    }
                    bean.setPwd(GMSSLHSMConstants.KEYSTORE_PASSWORD);
                    bean.setIndex(alias);
                    map.put(alias, bean);
                }
            }
            return map;
        } catch (Exception e) {
            LOGGER.error("hsmKeyStore解析失败", e);
            return null;
        }
    }

    /**
     * 获取对称密钥
     * @param index 密钥
     * @return 结果
     * @throws Exception 异常
     */
    public static GMSSLHsmKeyStoreBean getSymKey(Integer index) throws Exception{
        String key = MessageFormat.format(GMSSLHSMConstants.SYMMETRIC_ALIAS, index);
        return getHsmKeyStoreBean(key);
    }

    /**
     * 获取SM2密钥
     * @param index 密钥
     * @return 结果
     * @throws Exception 异常
     */
    public static GMSSLHsmKeyStoreBean getAsymKey(Integer index, boolean isEnc) throws Exception{
        String key = isEnc ? MessageFormat.format(GMSSLHSMConstants.ASYMMETRIC_ENC_ALIAS, index)
                : MessageFormat.format(GMSSLHSMConstants.ASYMMETRIC_SIGN_ALIAS, index);
        return getHsmKeyStoreBean(key);
    }

    /**
     * 根据索引获取
     * @param index 索引
     * @return 结果
     * @throws Exception 索引不存在
     */
    public static GMSSLHsmKeyStoreBean getHsmKeyStoreBean(String index) throws Exception {
        GMSSLHsmKeyStoreBean bean = GMSSLPkiCryptoInit.getHsmKeyStoreMap().get(index);
        if (null == bean) {
            throw new Exception("索引对应密钥不存在");
        }
        return bean;
    }

}
