package com.xdja.pki.gmssl.crypto.utils;


import com.xdja.pki.gmssl.core.utils.GMSSLBCCipherUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class GMSSLECIESEncryptUtils {

    private static Logger logger = LoggerFactory.getLogger(GMSSLECIESEncryptUtils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/
    public static byte[] encrypt(PublicKey publicKey, byte[] data) throws Exception {
        return encrypt(publicKey, data, 256);
    }

    /**
     * ECISE加密
     *
     * @param publicKey 外部加密公钥
     * @param data      加密数据
     * @param bits      密钥长度
     * @return 加密后的数据
     * @throws Exception
     */
    public static byte[] encrypt(PublicKey publicKey, byte[] data, int bits) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return GMSSLBCCipherUtils.encryptData(GMSSLBCCipherUtils.ECIES, publicKey, data);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return encrypt(SdfCryptoType.YUNHSM, publicKey, data, bits);
            case PCI_E:
                return encrypt(SdfCryptoType.PCIE, publicKey, data);
            case MINI_PCI_E:
            case SANC_HSM:
            case BC:
            default:
                return GMSSLBCCipherUtils.encryptData(GMSSLBCCipherUtils.ECIES, publicKey, data);
        }
    }


    public static byte[] decrypt(PrivateKey privateKey, byte[] cipher) throws Exception {
        return decrypt(privateKey, cipher, 256);
    }

    /**
     * ECISE解密
     *
     * @param privateKey 解密私钥
     * @param cipher     密文
     * @param bits       密钥长度
     * @return 解密后的数据
     * @throws Exception
     */
    public static byte[] decrypt(PrivateKey privateKey, byte[] cipher, int bits) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            SdfPrivateKey sdfPrivateKey = (SdfPrivateKey) privateKey;
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), true);
            return GMSSLBCCipherUtils.decryptData(GMSSLBCCipherUtils.ECIES, bean.getPrivateKey(), cipher);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return decrypt(SdfCryptoType.YUNHSM, privateKey, cipher, bits);
            case PCI_E:
                return decrypt(SdfCryptoType.PCIE, privateKey, cipher);
            case MINI_PCI_E:
            case SANC_HSM:
            case BC:
            default:
                return GMSSLBCCipherUtils.decryptData(GMSSLBCCipherUtils.ECIES, privateKey, cipher);
        }
    }

    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/

    @Deprecated
    public static byte[] encrypt(GMSSLCryptoType cryptoType, PublicKey publicKey, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return GMSSLBCCipherUtils.encryptData(GMSSLBCCipherUtils.ECIES, publicKey, data);
        }

        switch (cryptoType) {
            case BC:
                return GMSSLBCCipherUtils.encryptData(GMSSLBCCipherUtils.ECIES, publicKey, data);
            case XDJA_HSM:
                return encrypt(SdfCryptoType.YUNHSM, publicKey, data);
            case PCI_E:
                return encrypt(SdfCryptoType.PCIE, publicKey, data);
            default:
                throw new Exception("un support this crypto" + cryptoType);
        }

    }

    @Deprecated
    public static String encrypt(GMSSLCryptoType cryptoType, PublicKey publicKey, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(encrypt(cryptoType, publicKey, GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    @Deprecated
    public static byte[] decrypt(GMSSLCryptoType cryptoType, PrivateKey privateKey, byte[] cipher) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            SdfPrivateKey sdfPrivateKey = (SdfPrivateKey) privateKey;
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), true);
            return GMSSLBCCipherUtils.decryptData(GMSSLBCCipherUtils.ECIES, bean.getPrivateKey(), cipher);
        }
        if (privateKey instanceof SdfPrivateKey && cryptoType == GMSSLCryptoType.BC) {
            throw new Exception("privateKey is sdfPrivateKey, but crypto type is bc");
        }
        switch (cryptoType) {
            case XDJA_HSM:
                return decrypt(SdfCryptoType.YUNHSM, privateKey, cipher);
            case BC:
                return GMSSLBCCipherUtils.decryptData(GMSSLBCCipherUtils.ECIES, privateKey, cipher);
            case PCI_E:
                return decrypt(SdfCryptoType.PCIE, privateKey, cipher);
            default:
                throw new Exception("un support this crypto" + cryptoType);
        }
    }

    @Deprecated
    public static String decrypt(GMSSLCryptoType cryptoType, PrivateKey privateKey, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(decrypt(cryptoType, privateKey, GMSSLByteArrayUtils.base64Decode(base64Data)));

    }

    private static byte[] encrypt(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] data) throws Exception {
        return encrypt(sdfCryptoType, publicKey, data, 256);
    }

    private static byte[] encrypt(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] data, int bits) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return GMSSLBCCipherUtils.encryptData(GMSSLBCCipherUtils.ECIES, publicKey, data);
        }
        SdfECEngine sdfECEngine = new SdfECEngine(sdfCryptoType.getSdfSDK(), GMSSLECUtils.NISTp256, bits);
        sdfECEngine.init(true, new SdfECKeyParameters((ECPublicKey) publicKey));
        byte[] sdfCipher = sdfECEngine.encrypt(data);
        sdfECEngine.release();
        return sdfCipher;
    }

    private static byte[] decrypt(SdfCryptoType sdfCryptoType, PrivateKey privateKey, byte[] cipher) throws Exception {
        return decrypt(sdfCryptoType, privateKey, cipher, 256);
    }

    private static byte[] decrypt(SdfCryptoType sdfCryptoType, PrivateKey privateKey, byte[] cipher, int bits) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            SdfPrivateKey sdfPrivateKey = (SdfPrivateKey) privateKey;
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), true);
            return GMSSLBCCipherUtils.decryptData(GMSSLBCCipherUtils.ECIES, bean.getPrivateKey(), cipher);
        }
        SdfECEngine sdfECEngine = new SdfECEngine(sdfCryptoType.getSdfSDK(), GMSSLECUtils.NISTp256, bits);
        sdfECEngine.init(false, new SdfECKeyParameters((SdfPrivateKey) privateKey));
        byte[] data = sdfECEngine.decrypt(cipher);
        sdfECEngine.release();
        return data;
    }

}
