package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.crypto_hsm.sds.CMDCrypto;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refKeyPair;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refPublicKey;
import com.xdja.SafeKey.Sm2PrivateKey;
import com.xdja.SafeKey.Sm2PublicKey;
import com.xdja.SafeKey.bean.MiniPcieIndexEnum;
import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLECUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfECKey;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConverUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancKeyTypeEnum;
import com.xdja.pki.gmssl.sdf.SdfSDK;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.sdf.pcie.PcieSdfSDK;
import com.xdja.pki.gmssl.sdf.yunhsm.YunhsmSdfSDKUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.asn1.gm.GMNamedCurves;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.sec.SECObjectIdentifiers;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.KeyPair;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;

/**
 * @ClassName GMSSLECKeyUtils
 * @Description NIST密钥对
 * @Date 2020/1/13 14:05
 * @Author FengZhen
 */
public class GMSSLECKeyUtils {
    static {

        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    private static Logger logger = LoggerFactory.getLogger(GMSSLECKeyUtils.class);

    public static final String SM2p256 = GMNamedCurves.getName(GMObjectIdentifiers.sm2p256v1);
    public static final String NISTp256 = NISTNamedCurves.getName(SECObjectIdentifiers.secp256r1);
    public static final String NISTp384 = NISTNamedCurves.getName(SECObjectIdentifiers.secp384r1);
    public static final String NISTp521 = NISTNamedCurves.getName(SECObjectIdentifiers.secp521r1);
    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/

    /**
     * 生成 密钥对
     *
     * @param stdName 曲线名称可选 GMSSLECKeyUtils.SM2p256  NISTp256
     * @return 对应曲线密钥对
     */
    public static KeyPair generateECKeyPair(String stdName) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return GMSSLECUtils.generateKeyPair(stdName);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return generateECKeyPairBySdf(SdfCryptoType.PCIE, stdName);
            case XDJA_HSM:
                return generateECKeyPairBySdf(SdfCryptoType.YUNHSM, stdName);
            case DONGJIN_HSM:
                return generateECKeyPairBySdf(SdfCryptoType.DONGJIN, stdName);
            case MINI_PCI_E:
                //todo 暂时只支持SM2
                return MiniPcieXKFUtils.generateSm2KeyPair();
            case SANC_HSM:
                return generateECKeyPairBySancHsm(stdName);
            case BC:
            default:
                return GMSSLECUtils.generateKeyPair(stdName);
        }
    }

    /**
     * 生成 SDF 相关的 EC 私钥
     *
     * @param index    密钥访问索引
     * @param password 密钥访问控制密码
     * @return SDF 相关的 SM2私钥
     */
    public static SdfPrivateKey genSdfPrivateKey(int index, String password) {
        return new SdfPrivateKey(index, password);
    }

    public static SdfPrivateKey genSdfPrivateKey(int index) {
        return new SdfPrivateKey(index);
    }

    /**
     * 从硬件获取私钥访问权限
     *
     * @param index    索引
     * @param password 访问控制码
     * @return 是否可以获取
     */
    public static boolean getPrivateKeyAccessRightFromHardware(int index, String password) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return getPrivateKeyAccessRightFromYunHsm(index, password);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return YunhsmSdfSDKUtils.checkPrivateKeyAccess(index, password);
            case DONGJIN_HSM:
                return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.DONGJIN, index, password);
            case PCI_E:
                return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
            case MINI_PCI_E:
                //todo  ROlE  暂未进行区分
                return MiniPcieXKFUtils.verifyPin(password.getBytes());
            default:
                throw new Exception("un support the method with the crypto type " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * 根据密钥索引从硬件获取公钥
     *
     * @param index        索引
     * @param stdName      曲线名称
     * @param isEncryptKey 是否为加密密钥
     * @return 公钥
     */
    public static PublicKey getPublicKeyFromHardware(int index, String stdName, boolean isEncryptKey) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(index, isEncryptKey);
            return bean.getPublicKey();
        }

        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                if (isEncryptKey) {
                    return getEncryptPublicKeyByPcie(index, stdName);
                } else {
                    return getSignPublicKeyByPcie(index, stdName);
                }
            case XDJA_HSM:
                if (isEncryptKey) {
                    return getEncryptPublicKeyByYunhsm(index, stdName);
                } else {
                    return getSignPublicKeyByYunhsm(index, stdName);
                }
            case DONGJIN_HSM:
                if (isEncryptKey) {
                    return getEncPublicKeyByDongJinHsm(index, stdName);
                } else {
                    return getSignPublicKeyByDongJinHsm(index, stdName);
                }
            case MINI_PCI_E:
                //todo  暂时只支持SM2   NIST对应的接口 JNI未 提供 bean 类
                if (isEncryptKey) {
                    return MiniPcieXKFUtils.readSm2PublicKey(MiniPcieIndexEnum.getInfoFromIndex(index).getEncPublicKeyIndex()).getPublicKey(stdName);
                } else {
                    return MiniPcieXKFUtils.readSm2PublicKey(MiniPcieIndexEnum.getInfoFromIndex(index).getSignPublicKeyIndex()).getPublicKey(stdName);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                SM2refPublicKey sm2PublicKey;
                if (isEncryptKey) {
                    sm2PublicKey = cryptConnection.getSM2PublicKey(index, GMSSLSancKeyTypeEnum.ENCRYPT.getKeyType());
                } else {
                    sm2PublicKey = cryptConnection.getSM2PublicKey(index, GMSSLSancKeyTypeEnum.SIGN.getKeyType());
                }
                return GMSSLSancConverUtils.converSM2PublicKey(sm2PublicKey);
            case BC:
            default:
                throw new Exception("un support the method with the crypto type " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * 向硬件写入密钥对 只支持miniPCIE卡和PCIE卡
     *
     * @param index        索引
     * @param password     密码
     * @param keyPair      密钥对
     * @param isEncryptKey 是否为加密密钥
     * @return 公钥
     */
    public static boolean writeKeyPairToHardWare(int index, String password, KeyPair keyPair, boolean isEncryptKey) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case MINI_PCI_E:
                if (isEncryptKey) {
                    boolean b = MiniPcieXKFUtils.writeSm2PublicKey(MiniPcieIndexEnum.getInfoFromIndex(index).getEncPublicKeyIndex(), password, new Sm2PublicKey(keyPair.getPublic()));
                    if (!b) {
                        return false;
                    }
                    return MiniPcieXKFUtils.writeSm2PrivateKey(MiniPcieIndexEnum.getInfoFromIndex(index).getEncPrivateIndex(), password, new Sm2PrivateKey(keyPair.getPrivate()));

                } else {
                    boolean b = MiniPcieXKFUtils.writeSm2PublicKey(MiniPcieIndexEnum.getInfoFromIndex(index).getSignPublicKeyIndex(), password, new Sm2PublicKey(keyPair.getPublic()));
                    if (!b) {
                        return false;
                    }
                    return MiniPcieXKFUtils.writeSm2PrivateKey(MiniPcieIndexEnum.getInfoFromIndex(index).getSignPrivateIndex(), password, new Sm2PrivateKey(keyPair.getPrivate()));
                }
            case PCI_E:
                PcieSdfSDK pcieSdfSDK = new PcieSdfSDK();
                if (isEncryptKey) {
                    try {
                        pcieSdfSDK.importSM2PublicKey(keyPair.getPublic(), index, false);
                        pcieSdfSDK.importSM2PrivateKey(keyPair.getPrivate(), index, password.getBytes(), false);
                        return true;
                    } catch (Exception e) {
                        logger.error("导入SM2加密密钥对失败", e);
                        return false;
                    }
                } else {
                    try {
                        pcieSdfSDK.importSM2PublicKey(keyPair.getPublic(), index, true);
                        pcieSdfSDK.importSM2PrivateKey(keyPair.getPrivate(), index, password.getBytes(), true);
                        return true;
                    } catch (Exception e) {
                        logger.error("导入SM2签名密钥对失败", e);
                        return false;
                    }
                }
            default:
                throw new Exception("un support writeCertToHardWare  with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/


    /************************************************************************************
     *                                      密钥对相关                                  *
     ************************************************************************************/
    public static KeyPair generateECKeyPairBySancHsm(String stdName) throws Exception {
        CMDCrypto cmdCrypto = new CMDCrypto();
        PublicKey publicKey;
        PrivateKey privateKey;
        if (stdName.equalsIgnoreCase(GMSSLECKeyUtils.SM2p256)) {
            SM2refKeyPair sm2refKeyPair = cmdCrypto.generateSM2KeyPair(256);
            publicKey = GMSSLX509Utils.convertSM2PublicKey(sm2refKeyPair.getPublicKey().getX(), sm2refKeyPair.getPublicKey().getY());
            privateKey = GMSSLX509Utils.convertSM2PrivateKey(sm2refKeyPair.getPrivateKey().getD());
        } else {
            return null;
        }
        return new KeyPair(publicKey, privateKey);
    }

    /**
     * 生成 EC 密钥对
     *
     * @return EC 密钥对
     */
    @Deprecated
    public static KeyPair generateECKeyPairByBC(String stdName) throws Exception {
        return GMSSLECUtils.generateKeyPair(stdName);
    }


    public static KeyPair generateECKeyPairBySdf(SdfCryptoType sdfCryptoType, String stdName) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return generateECKeyPair(stdName);
        }
        SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
        sdfSDK.init();
        SdfECKey sdfECKey = new SdfECKey(sdfCryptoType, stdName);
        return sdfECKey.generateKeyPair();
    }

//    /**
//     * 生成 EC 密钥对
//     *
//     * @return EC 密钥对
//     */
//    public static SdfECCKeyPair generateECKeyPairByYunHsm(String stdName) throws Exception {
//
//    }

    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByYunhsm(int index, String stdName) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()
                || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM
                || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            return getPublicKeyFromHardware(index, stdName, false);
        }
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.YUNHSM, stdName);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }


    /**
     * 根据密钥索引 从 密码机 获取签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByDongJinHsm(int index, String stdName) throws Exception {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.DONGJIN, stdName);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncPublicKeyByDongJinHsm(int index, String stdName) throws Exception {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.DONGJIN, stdName);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }


    /**
     * 根据密钥索引 从 PCIE 获取 签名公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getSignPublicKeyByPcie(int index, String stdName) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.PCIE, stdName);
        PublicKey publicKey = sdfECKey.exportSignPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }


    /**
     * 根据密钥索引 从 密码机 获取加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByYunhsm(int index, String stdName) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()
                || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM
                || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            return getPublicKeyFromHardware(index, stdName, true);
        }
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.YUNHSM, stdName);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 根据密钥索引 从 PCIE 获取 加密公钥
     *
     * @param index 密钥访问索引
     * @return 密钥索引对应的公钥信息
     */
    public static PublicKey getEncryptPublicKeyByPcie(int index, String stdName) throws SdfSDKException {
        SdfECKey sdfECKey = new SdfECKey(SdfCryptoType.PCIE, stdName);
        PublicKey publicKey = sdfECKey.exportEncryptPublicKey(index);
        sdfECKey.release();
        return publicKey;
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    public static boolean getPrivateKeyAccessRightFromYunHsm(int index, String password) {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            GMSSLHsmKeyStoreBean bean;
            try {
                bean = GMSSLHsmKeyStoreUtils.getAsymKey(index, false);
            } catch (Exception e) {
                try {
                    bean = GMSSLHsmKeyStoreUtils.getAsymKey(index, true);
                } catch (Exception exception) {
                    return false;
                }
            }
            return bean.getPwd().equals(password);
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.DONGJIN, index, password);
        }
        return YunhsmSdfSDKUtils.checkPrivateKeyAccess(index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  密码机
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    @Deprecated
    public static boolean checkPrivateKeyAccessRightFromYunHsm(int index, String password) {
        return getPrivateKeyAccessRightFromYunHsm(index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码  PCIE卡
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */
    @Deprecated
    public static boolean checkPrivateKeyAccessRightFromPCIE(int index, String password) {
        return checkPrivateKeyAccessRightFromSdf(SdfCryptoType.PCIE, index, password);
    }

    /**
     * 外部校验密钥索引和私钥访问控制码
     *
     * @param index    密钥访问索引
     * @param password 私钥访问控制码
     * @return boolean
     */

    public static boolean checkPrivateKeyAccessRightFromSdf(SdfCryptoType sdfCryptoType, int index, String password) {
        try {
            if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
                sdfCryptoType = SdfCryptoType.DONGJIN;
            }
            SdfSDK sdfSDK = sdfCryptoType.getSdfSDK();
            sdfSDK.init();
            boolean result = sdfSDK.checkPrivateKeyAccessRight(index, password.getBytes());
            sdfSDK.release();
            return result;
        } catch (SdfSDKException e) {
            return false;
        }
    }

}
