package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.struct.key.IRSArefPublicKey;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLRSAUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConverUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancKeyTypeEnum;
import com.xdja.pki.gmssl.operator.utils.GMSSLPKCS1Encodeing;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.jcajce.provider.asymmetric.rsa.BCRSAPublicKey;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;

/**
 * @author sxy
 * @description
 * @date 2019年09月18日 15:57
 **/
public class GMSSLRSAEncryptUtils {


    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/
    /**
     * RSA公钥加密  仅支持BC
     *
     * @param publicKey 公钥加密
     * @param data      加密数据
     * @return
     * @throws Exception
     */
    public static byte[] encrypt(PublicKey publicKey, byte[] data) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
            case PCI_E:
            case MINI_PCI_E:
            case SANC_HSM:
            case BC:
            case DONGJIN_HSM:
            default:
                return GMSSLRSAUtils.encryptData(publicKey, data);
        }
    }

    /**
     * RSA私钥解密 仅支持BC
     *
     * @param privateKey RSA私钥解密
     * @param cipher     密文
     * @return
     * @throws Exception
     */
    public static byte[] decrypt(PrivateKey privateKey, byte[] cipher) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
            case PCI_E:
            case MINI_PCI_E:
            case SANC_HSM:
            case BC:
            default:
                return GMSSLRSAUtils.decryptData(privateKey, cipher);
        }
    }

    /**
     * PKCS1填充模式加密 仅支持BC SANC_HSM
     *
     * @param publicKey 加密公钥
     * @param data      需要加密数据  1024不得超过 128 -11 = 117 2048不得超过 256-11 = 245
     * @return 密文
     * @throws Exception
     */
    public static byte[] encryptWithPKCS1(PublicKey publicKey, byte[] data) throws Exception {
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case SANC_HSM:
                int bitLength = ((BCRSAPublicKey) publicKey).getModulus().bitLength();
                byte[] encodeData = GMSSLPKCS1Encodeing.encodeBlock(data, 0, data.length, bitLength);
                IRSArefPublicKey arefPublicKey = GMSSLSancConverUtils.converRSARefPublicKey(publicKey);
                return GMSSLSancConnectionUtils.getCryptConnection().rsaPublicKeyOperation(arefPublicKey, encodeData);
            case BC:
            case XDJA_HSM:
            case PCI_E:
            case MINI_PCI_E:
            default:
                return GMSSLRSAUtils.encryptDataPKCS1Padding(publicKey, data);
        }
    }

    /**
     * PKCS1填充模式解密密 仅支持BC SANC_HSM
     *
     * @param privateKey 解密私钥
     * @param cipher     密文
     * @return 解密原文
     * @throws Exception
     */
    public static byte[] decryptWithPKCS1(PrivateKey privateKey, byte[] cipher) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            SdfPrivateKey pciePrivateKey = (SdfPrivateKey) privateKey;
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(pciePrivateKey.getIndex(), true);
            return GMSSLRSAUtils.decryptDataPKCS1Padding(bean.getPrivateKey(), cipher);
        }

        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                SdfPrivateKey sancPrivate = (SdfPrivateKey) privateKey;
                byte[] decrypt = cryptConnection.rsaPrivateKeyOperation(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.ENCRYPT.getKeyType(), cipher);
                IRSArefPublicKey rsaPublicKey = cryptConnection.getRSAPublicKey(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.ENCRYPT.getKeyType());
                return GMSSLPKCS1Encodeing.decodeBlock(decrypt, rsaPublicKey.getBits());
            case BC:
            case XDJA_HSM:
            case PCI_E:
            case MINI_PCI_E:
            default:
                return GMSSLRSAUtils.decryptDataPKCS1Padding(privateKey, cipher);
        }
    }


    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/

    /************************************************************************************
     *                                      NoPadding                                  *
     ************************************************************************************/
    /**
     * 使用BC软算法对数据进行加密
     * NOPadding
     *
     * @param publicKey 公钥
     * @param data      代加密数据 base64编码
     * @return 加密数据 base64编码
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static String encryptDataByBC(
            PublicKey publicKey, String data
    ) throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        byte[] plain = GMSSLByteArrayUtils.base64Decode(data);
        byte[] cipher = GMSSLRSAUtils.encryptData(publicKey, plain);
        return GMSSLByteArrayUtils.base64Encode(cipher);
    }

    /**
     * 使用BC软算法对数据进行解密
     * NoPadding
     *
     * @param privateKey 私钥
     * @param data       带解密数据 base64编码
     * @return 解密后的数据 base64编码
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static String decryptDataByBC(
            PrivateKey privateKey, String data
    ) throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        byte[] cipher = GMSSLByteArrayUtils.base64Decode(data);
        byte[] plain = GMSSLRSAUtils.decryptData(privateKey, cipher);
        return GMSSLByteArrayUtils.base64Encode(plain);
    }

    /**
     * 使用BC软算法对数据进行加密
     * NoPadding
     *
     * @param publicKey 公钥
     * @param data      代加密数据
     * @return 加密数据 byte数组
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static byte[] encryptDataByBC(
            PublicKey publicKey, byte[] data
    ) throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return GMSSLRSAUtils.encryptData(publicKey, data);
    }

    /**
     * 使用BC软算法对数据进行解密
     * NoPadding
     *
     * @param privateKey 私钥
     * @param data       带解密数据
     * @return 解密后的数据 byte数组
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static byte[] decryptDataByBC(
            PrivateKey privateKey, byte[] data
    ) throws NoSuchPaddingException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, BadPaddingException, IllegalBlockSizeException {
        return GMSSLRSAUtils.decryptData(privateKey, data);
    }

    /************************************************************************************
     *                                      PKCS1Padding                                  *
     ************************************************************************************/

    /**
     * 使用BC软算法对数据进行加密
     * PKCS1Padding
     *
     * @param publicKey 公钥
     * @param data      代加密数据 base64编码
     * @return 加密数据 base64编码
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static String encryptDataPKCS1ByBC(
            PublicKey publicKey, String data
    ) throws InvalidCipherTextException {
        byte[] plain = GMSSLByteArrayUtils.base64Decode(data);
        byte[] cipher = GMSSLRSAUtils.encryptDataPKCS1Padding(publicKey, plain);
        return GMSSLByteArrayUtils.base64Encode(cipher);
    }

    /**
     * 使用BC软算法对数据进行解密
     * PKCS1Padding
     *
     * @param privateKey 私钥
     * @param data       带解密数据 base64编码
     * @return 解密后的数据 base64编码
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static String decryptDataPKCS1ByBC(
            PrivateKey privateKey, String data
    ) throws InvalidCipherTextException {
        byte[] cipher = GMSSLByteArrayUtils.base64Decode(data);
        byte[] plain = GMSSLRSAUtils.decryptDataPKCS1Padding(privateKey, cipher);
        return GMSSLByteArrayUtils.base64Encode(plain);
    }

    /**
     * 使用BC软算法对数据进行加密
     * PKCS1Padding
     *
     * @param publicKey 公钥
     * @param data      代加密数据
     * @return 加密数据 byte数组
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static byte[] encryptDataPKCS1ByBC(
            PublicKey publicKey, byte[] data
    ) throws InvalidCipherTextException {
        return GMSSLRSAUtils.encryptDataPKCS1Padding(publicKey, data);
    }


    /**
     * 使用BC软算法对数据进行解密
     * PKCS1Padding
     *
     * @param privateKey 私钥
     * @param data       带解密数据
     * @return 解密后的数据 byte数组
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     */
    public static byte[] decryptDataPKCS1ByBC(
            PrivateKey privateKey, byte[] data
    ) throws InvalidCipherTextException {
        return GMSSLRSAUtils.decryptDataPKCS1Padding(privateKey, data);
    }
}
