package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfRandom;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;


import java.security.Security;

public class GMSSLRandomUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                     只维护以下方法                                 *
     ************************************************************************************/

    /**
     * 产生随机数
     *
     * @param length 随机数长度
     * @return
     * @throws Exception
     */
    public static byte[] generateRandom(int length) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return com.xdja.pki.gmssl.core.utils.GMSSLRandomUtils.generateRandom(length);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfRandom pcieRandom = new SdfRandom(SdfCryptoType.PCIE);
                return pcieRandom.generateRandom(length);
            case XDJA_HSM:
                SdfRandom hsmRandom = new SdfRandom(SdfCryptoType.YUNHSM);
                return hsmRandom.generateRandom(length);
            case DONGJIN_HSM:
                SdfRandom dJRandom = new SdfRandom(SdfCryptoType.DONGJIN);
                return dJRandom.generateRandom(length);
            case MINI_PCI_E:
                return MiniPcieXKFUtils.generateRandom(length);
            case SANC_HSM:
                return GMSSLSancConnectionUtils.getCryptConnection().generateRandom(length);
            case BC:
            default:
                return com.xdja.pki.gmssl.core.utils.GMSSLRandomUtils.generateRandom(length);
        }
    }

    /************************************************************************************
     *                                      只维护以上方法                                *
     ************************************************************************************/


    /************************************************************************************
     *                                      随机数                                      *
     ************************************************************************************/
    /**
     * 使用 密码机 生成随机数
     *
     * @param len 随机数长度
     * @return BASE 64 编码的随机数
     * @throws SdfSDKException Sdf SDK 异常
     */
    public static String generateRandomByYunhsm(int len) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() ||
                GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            return generateRandomBySdf(SdfCryptoType.DONGJIN, len);
        }
        return generateRandomBySdf(SdfCryptoType.YUNHSM, len);
    }

    /**
     * 使用 PCIE 生成随机数
     *
     * @param len 随机数长度
     * @return BASE 64 编码的随机数
     * @throws SdfSDKException Sdf SDK 异常
     */
    public static String generateRandomByPcie(int len) throws Exception {
        return generateRandomBySdf(SdfCryptoType.PCIE, len);
    }

    /**
     * 使用 密码机 生成随机数
     *
     * @param sdfCryptoType SDF 算法类型
     * @param len           随机数长度
     * @return BASE 64 编码的随机数
     * @throws SdfSDKException Sdf SDK 异常
     */
    public static String generateRandomBySdf(SdfCryptoType sdfCryptoType, int len) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() ||
                GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return Base64.toBase64String(generateRandom(len));
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        SdfRandom sdfRandom = new SdfRandom(sdfCryptoType);
        byte[] random = sdfRandom.generateRandom(len);
        return GMSSLByteArrayUtils.base64Encode(random);
    }
}
