package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refCipher;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refPublicKey;
import com.xdja.SafeKey.utils.MiniPcieSm2EncryptUtils;
import com.xdja.pki.gmssl.asn1.crypto.ASN1SM2Cipher;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfECEngine;
import com.xdja.pki.gmssl.crypto.sdf.SdfECKeyParameters;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConverUtils;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancKeyTypeEnum;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ECKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.interfaces.ECPublicKey;

public class GMSSLSM2EncryptUtils {

    private static final Logger logger = LoggerFactory.getLogger(GMSSLSM2EncryptUtils.class.getName());

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * SM2加密
     *
     * @param publicKey 加密公钥
     * @param data      加密数据
     * @return
     * @throws Exception
     */
    public static byte[] encrypt(PublicKey publicKey, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return encryptByBC(publicKey, data);
        }

        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return encryptBySdf(SdfCryptoType.PCIE, publicKey, data, 0, data.length);
            case MINI_PCI_E:
                return MiniPcieSm2EncryptUtils.encrypt(data, publicKey, false);
            case XDJA_HSM:
                return encryptBySdf(SdfCryptoType.YUNHSM, publicKey, data, 0, data.length);
            case DONGJIN_HSM:
                return encryptBySdf(SdfCryptoType.DONGJIN, publicKey, data, 0, data.length);
            case SANC_HSM:
                return encryptBySanc(publicKey, data);
            case BC:
            default:
                return encryptByBC(publicKey, data);
        }
    }

    /**
     * SM2解密
     *
     * @param privateKey 解密私钥
     * @param data       加密数据
     * @return 源数据
     * @throws Exception
     */
    public static byte[] decrypt(PrivateKey privateKey, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            SdfPrivateKey pciePrivate = (SdfPrivateKey) privateKey;
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(pciePrivate.getIndex(), true);
            AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(bean.getPrivateKey());
            return decryptByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        }

        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivate = (SdfPrivateKey) privateKey;
                return decryptBySdf(SdfCryptoType.PCIE, pciePrivate.getIndex(), pciePrivate.getStringPassword(), data, 0, data.length);
            case MINI_PCI_E:
                SdfPrivateKey miniPciePrivate = (SdfPrivateKey) privateKey;
                return MiniPcieSm2EncryptUtils.decrypt(data, miniPciePrivate.getIndex(), miniPciePrivate.getStringPassword(), false);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivate = (SdfPrivateKey) privateKey;
                return decryptBySdf(SdfCryptoType.YUNHSM, hsmPrivate.getIndex(), hsmPrivate.getStringPassword(), data, 0, data.length);
            case DONGJIN_HSM:
                SdfPrivateKey dJPrivate = (SdfPrivateKey) privateKey;
                return decryptBySdf(SdfCryptoType.DONGJIN, dJPrivate.getIndex(), dJPrivate.getStringPassword(), data, 0, data.length);
            case SANC_HSM:
                return decryptBySanc(privateKey, data);
            case BC:
            default:
                AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
                return decryptByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        }
    }

    /**
     * ASN1格式 SM2 加密
     *
     * @param publicKey SM2公钥
     * @param data      需要加密的数据
     * @return
     * @throws Exception
     */
    public static byte[] encryptAsn1Data(PublicKey publicKey, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return encryptASN1ByBC(publicKey, data);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return encryptASN1BySdf(SdfCryptoType.PCIE, publicKey, data, 0, data.length);
            case MINI_PCI_E:
                return MiniPcieSm2EncryptUtils.encrypt(data, publicKey, true);
            case XDJA_HSM:
                return encryptASN1BySdf(SdfCryptoType.YUNHSM, publicKey, data, 0, data.length);
            case DONGJIN_HSM:
                return encryptASN1BySdf(SdfCryptoType.DONGJIN, publicKey, data, 0, data.length);
            case SANC_HSM:
                return encryptAsn1BySanc(publicKey, data);
            case BC:
            default:
                return encryptASN1ByBC(publicKey, data);
        }
    }

    /**
     * ASN1格式 SM2 解密
     *
     * @param privateKey SM2私钥
     * @param data       加密的数据
     * @return
     * @throws Exception
     */
    public static byte[] decryptAsn1Cipher(PrivateKey privateKey, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            SdfPrivateKey pciePrivate = (SdfPrivateKey) privateKey;
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getAsymKey(pciePrivate.getIndex(), true);
            AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(bean.getPrivateKey());
            return decryptASN1ByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                SdfPrivateKey pciePrivate = (SdfPrivateKey) privateKey;
                return decryptASN1BySdf(SdfCryptoType.PCIE, pciePrivate.getIndex(), pciePrivate.getStringPassword(), data, 0, data.length);
            case MINI_PCI_E:
                SdfPrivateKey miniPciePrivate = (SdfPrivateKey) privateKey;
                return MiniPcieSm2EncryptUtils.decrypt(data, miniPciePrivate.getIndex(), miniPciePrivate.getStringPassword(), true);
            case XDJA_HSM:
                SdfPrivateKey hsmPrivate = (SdfPrivateKey) privateKey;
                return decryptASN1BySdf(SdfCryptoType.YUNHSM, hsmPrivate.getIndex(), hsmPrivate.getStringPassword(), data, 0, data.length);
            case DONGJIN_HSM:
                SdfPrivateKey dJPrivate = (SdfPrivateKey) privateKey;
                return decryptASN1BySdf(SdfCryptoType.DONGJIN, dJPrivate.getIndex(), dJPrivate.getStringPassword(), data, 0, data.length);
            case SANC_HSM:
                return decryptAsn1BySanc(privateKey, data);
            case BC:
            default:
                AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
                return decryptASN1ByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        }
    }

    /************************************************************************************
     *                                      加密解密                                    *
     *                   GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3             *
     ************************************************************************************/
    private static byte[] encryptBySanc(PublicKey publicKey, byte[] data) throws Exception {
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        SM2refPublicKey sm2refPublicKey = GMSSLSancConverUtils.converSM2RefPublicKey(publicKey);
        SM2refCipher sm2refCipher = cryptConnection.sm2Encrypt(sm2refPublicKey, data);
        byte[] x = sm2refCipher.getX();
        byte[] y = sm2refCipher.getY();
        byte[] c = sm2refCipher.getC();
        c = GMSSLByteArrayUtils.filterByteArrayZeroInFoot(c, data.length);
        byte[] m = sm2refCipher.getM();
        byte[] PO = new byte[x.length + y.length + 1];
        PO[0] = 0x04;
        System.arraycopy(x, 0, PO, 1, x.length);
        System.arraycopy(y, 0, PO, x.length + 1, y.length);
        byte[] cipher = new byte[PO.length + c.length + m.length];
        System.arraycopy(PO, 0, cipher, 0, PO.length);
        System.arraycopy(c, 0, cipher, PO.length, c.length);
        System.arraycopy(m, 0, cipher, PO.length + c.length, m.length);
        return cipher;
    }

    private static byte[] decryptBySanc(PrivateKey privateKey, byte[] data) throws Exception {
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        SdfPrivateKey sancPrivate = (SdfPrivateKey) privateKey;
        byte[] x = new byte[32];
        byte[] y = new byte[32];
        byte[] m = new byte[32];
        byte[] c = new byte[data.length - 1 - x.length - y.length - m.length];
        System.arraycopy(data, 1, x, 0, x.length);
        System.arraycopy(data, 1 + x.length, y, 0, y.length);
        System.arraycopy(data, 1 + x.length + y.length, c, 0, c.length);
        System.arraycopy(data, 1 + x.length + y.length + c.length, m, 0, m.length);
        SM2refCipher sm2refCipher = new SM2refCipher(x, y, c, m);
        return cryptConnection.sm2Decrypt(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.ENCRYPT.getKeyType(), sm2refCipher);
    }


    private static byte[] encryptAsn1BySanc(PublicKey publicKey, byte[] data) throws Exception {
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        SM2refPublicKey sm2refPublicKey = GMSSLSancConverUtils.converSM2RefPublicKey(publicKey);
        SM2refCipher sm2refCipher = cryptConnection.sm2Encrypt(sm2refPublicKey, data);
        ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(
                sm2refCipher.getX(),
                sm2refCipher.getY(),
                sm2refCipher.getM(),
                GMSSLByteArrayUtils.filterByteArrayZeroInFoot(sm2refCipher.getC(), data.length));
        System.out.println(sm2refCipher.toString());
        return asn1SM2Cipher.getEncoded();
    }

    private static byte[] decryptAsn1BySanc(PrivateKey privateKey, byte[] data) throws Exception {
        ASN1SM2Cipher asn1SM2Cipher = ASN1SM2Cipher.getInstance(data);
        ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
        SdfPrivateKey sancPrivate = (SdfPrivateKey) privateKey;
        assert asn1SM2Cipher != null;
        //   public byte[] C = new byte[136];
        System.out.println(data.length - 1 - 32 - 32 - 32);
        SM2refCipher sm2refCipher = new SM2refCipher(
                GMSSLByteArrayUtils.changeByteArrayLength(asn1SM2Cipher.getxCoordinate().toByteArray(), 32),
                GMSSLByteArrayUtils.changeByteArrayLength(asn1SM2Cipher.getyCoordinate().toByteArray(), 32),
                asn1SM2Cipher.getCipherText(),
                GMSSLByteArrayUtils.changeByteArrayLength(asn1SM2Cipher.getHash(), 32));
        System.out.println(sm2refCipher.toString());
        return cryptConnection.sm2Decrypt(sancPrivate.getIndex(), GMSSLSancKeyTypeEnum.ENCRYPT.getKeyType(), sm2refCipher);
    }

    /**
     * 使用 BC 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptByBC(PublicKey publicKey, String base64Data) throws CryptoException, IOException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPublicKeyParameters(publicKey);
        byte[] cipher = encryptByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(cipher);
    }

    public static byte[] encryptByBC(PublicKey publicKey, byte[] data) throws CryptoException, IOException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPublicKeyParameters(publicKey);
        return encryptByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
    }

    /**
     * 使用 密码机 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptByYunhsm(PublicKey publicKey, String base64Data) throws Exception {
        return encryptBySdf(SdfCryptoType.YUNHSM, publicKey, base64Data);
    }

    /**
     * 使用 PCIE 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptByPcie(PublicKey publicKey, String base64Data) throws Exception {
        return encryptBySdf(SdfCryptoType.PCIE, publicKey, base64Data);
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static String encryptBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sdfCipher = encryptBySdf(sdfCryptoType, publicKey, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(sdfCipher);
    }

    /**
     * 使用 BC 进行解密
     *
     * @param privateKey 加密者私钥
     * @param base64Data BASE64 编码 加密消息 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 加密原文
     */
    public static String decryptByBC(PrivateKey privateKey, String base64Data) throws IOException, InvalidCipherTextException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw InvalidCipherTextException
        byte[] out = decryptByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(out);
    }

    /**
     * 使用 密码机 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 明文
     */
    public static String decryptByYunhsm(int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        return decryptBySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 PCIE 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 明文
     */
    public static String decryptByPcie(int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        return decryptBySdf(SdfCryptoType.PCIE, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 SDF 进行解密
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @return BASE64 编码 明文
     */
    public static String decryptBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] out = decryptBySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(out);
    }


    /************************************************************************************
     *                                      加密解密                                    *
     *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
     ************************************************************************************/

    /**
     * 使用 BC 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     */
    public static String encryptASN1ByBC(PublicKey publicKey, String base64Data) throws CryptoException, IOException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw IOException
        byte[] cipher = encryptASN1ByBC(publicKey, data);
        return GMSSLByteArrayUtils.base64Encode(cipher);
    }

    /**
     * 使用 BC 进行加密
     *
     * @param publicKey 加密者公钥
     * @param data      BASE64 编码 待加密消息
     * @return 二进制 编码 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     */
    public static byte[] encryptASN1ByBC(PublicKey publicKey, byte[] data) throws CryptoException, IOException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPublicKeyParameters(publicKey);
        byte[] cipher = encryptASN1ByBC((ECKeyParameters) keyParameter, new SecureRandom(), data, 0, data.length);
        return cipher;
    }

    /**
     * 使用 密码机 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文
     */
    public static String encryptASN1ByYunhsm(PublicKey publicKey, String base64Data) throws Exception {
        return encryptASN1BySdf(SdfCryptoType.YUNHSM, publicKey, base64Data);
    }

    /**
     * 使用 PCIE 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文
     */
    public static String encryptASN1ByPcie(PublicKey publicKey, String base64Data) throws Exception {
        return encryptASN1BySdf(SdfCryptoType.PCIE, publicKey, base64Data);
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param publicKey  加密者公钥
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文
     */
    public static String encryptASN1BySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sdfCipher = encryptASN1BySdf(sdfCryptoType, publicKey, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(sdfCipher);
    }

    /**
     * 使用 BC 进行解密
     *
     * @param privateKey 加密者私钥
     * @param base64Data BASE64 编码 加密消息
     * @return BASE64 编码 加密原文
     */
    public static String decryptASN1ByBC(PrivateKey privateKey, String base64Data) throws IOException, InvalidCipherTextException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        //will throw InvalidCipherTextException
        byte[] out = decryptASN1ByBC(privateKey, data);
        return GMSSLByteArrayUtils.base64Encode(out);
    }

    /**
     * 使用 BC 进行解密
     *
     * @param privateKey 加密者私钥
     * @return 加密原文
     */
    public static byte[] decryptASN1ByBC(PrivateKey privateKey, byte[] data) throws IOException, InvalidCipherTextException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
        //will throw InvalidCipherTextException
        byte[] out = decryptASN1ByBC((ECKeyParameters) keyParameter, data, 0, data.length);
        return out;
    }

    /**
     * 使用 密码机 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文
     * @return BASE64 编码 明文
     */
    public static String decryptASN1ByYunhsm(int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        return decryptASN1BySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 PCIE 进行解密
     *
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文
     * @return BASE64 编码 明文
     */
    public static String decryptASN1ByPcie(int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        return decryptASN1BySdf(SdfCryptoType.PCIE, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 SDF 进行解密
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    解密私钥索引
     * @param privateKeyPassword 解密私钥访问控制码
     * @param base64Data         BASE64 编码 密文
     * @return BASE64 编码 明文
     */
    public static String decryptASN1BySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] out = decryptASN1BySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data, 0, data.length);
        return GMSSLByteArrayUtils.base64Encode(out);
    }

    /**
     * SM2 加密 输出 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     *
     * @param keyParameter 密钥配置
     * @param secureRandom 随机数
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 输出 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     * @throws InvalidCipherTextException
     */
    public static byte[] encryptByBC(ECKeyParameters keyParameter, SecureRandom secureRandom, byte[] input, int inOff, int length) throws InvalidCipherTextException {
        SM2Engine sm2Engine = new SM2Engine();
        sm2Engine.init(true, new ParametersWithRandom(keyParameter, secureRandom));
        return sm2Engine.processBlock(input, inOff, length);
    }

    /**
     * SM2 解密 输入 GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 明文
     * @throws InvalidCipherTextException
     */
    public static byte[] decryptByBC(ECKeyParameters keyParameter, byte[] input, int inOff, int length) throws InvalidCipherTextException {
        SM2Engine sm2Engine = new SM2Engine();
        sm2Engine.init(false, keyParameter);
        return sm2Engine.processBlock(input, inOff, length);
    }

    /**
     * SM2 加密 输出 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 输出 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     * @throws InvalidCipherTextException
     */
    public static byte[] encryptASN1ByBC(ECKeyParameters keyParameter, SecureRandom secureRandom, byte[] input, int inOff, int length) throws InvalidCipherTextException, IOException {
        SM2Engine sm2Engine = new SM2Engine();
        //这里用来生成密文，与rsa不同的是不需要用pkcs#1结构体，传入参数为：公钥+随机数
        sm2Engine.init(true, new ParametersWithRandom(keyParameter, secureRandom));

        byte[] ciphertext = sm2Engine.processBlock(input, inOff, length);

        ECCurve curve = keyParameter.getParameters().getCurve();
        int curveLength = (curve.getFieldSize() + 7) / 8;

        byte[] c1 = new byte[curveLength * 2 + 1];
        System.arraycopy(ciphertext, inOff, c1, 0, c1.length);

        ECPoint c1P = curve.decodePoint(c1);
        byte[] xb = c1P.getXCoord().getEncoded();
        byte[] yb = c1P.getYCoord().getEncoded();

        SM3Digest digest = new SM3Digest();

        byte[] c2 = new byte[length];
        System.arraycopy(ciphertext, ciphertext.length - length - digest.getDigestSize(), c2, 0, length);

        byte[] c3 = new byte[digest.getDigestSize()];
        System.arraycopy(ciphertext, ciphertext.length - digest.getDigestSize(), c3, 0, digest.getDigestSize());

        ASN1SM2Cipher asn1SM2Cipher = new ASN1SM2Cipher(xb, yb, c3, c2);
        byte[] result = asn1SM2Cipher.toASN1Primitive().getEncoded();

        return result;
    }

    /**
     * SM2 解密 输入 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @return 明文
     * @throws InvalidCipherTextException
     */
    public static byte[] decryptASN1ByBC(ECKeyParameters keyParameter, byte[] input) throws InvalidCipherTextException {
        return decryptASN1ByBC(keyParameter, input, 0, input.length);
    }

    /**
     * SM2 解密 输入 GM/T 0009-2012 7.2 加密数据结构 ASN.1 结构体
     *
     * @param keyParameter 密钥配置
     * @param input        输入数据
     * @param inOff        输入起始位置
     * @param length       长度
     * @return 明文
     * @throws InvalidCipherTextException
     */
    public static byte[] decryptASN1ByBC(ECKeyParameters keyParameter, byte[] input, int inOff, int length) throws InvalidCipherTextException {
        byte[] sm2CipherASN1 = new byte[length];
        System.arraycopy(input, inOff, sm2CipherASN1, 0, length);

        ASN1SM2Cipher asn1 = ASN1SM2Cipher.getInstance(sm2CipherASN1);

        //generate c1
        ECCurve curve = keyParameter.getParameters().getCurve();
        BigInteger x = asn1.getxCoordinate();
        BigInteger y = asn1.getyCoordinate();
        ECPoint point = curve.createPoint(x, y);
        byte[] c1 = point.getEncoded(false);
//        GMSSLByteArrayUtils.printHexBinary(logger, "c1", c1);
        //get c2
        byte[] c2 = asn1.getCipherText();
//        GMSSLByteArrayUtils.printHexBinary(logger, "c2", c2);

        //get c3
        byte[] c3 = asn1.getHash();
//        GMSSLByteArrayUtils.printHexBinary(logger, "c3", c3);

        byte[] ciphertext = Arrays.concatenate(c1, c2, c3);

        SM2Engine sm2Engine = new SM2Engine();
        //这里如果要是 decrypt 解密的话 需要传入 private key
        sm2Engine.init(false, keyParameter);

        byte[] M = sm2Engine.processBlock(ciphertext, 0, ciphertext.length);

        return M;
    }

    /************************************************************************************
     *                                      加密解密                                    *
     *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
     ************************************************************************************/


    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType SDF 密码类型 密码机、PCIE
     * @param publicKey     加密者公钥
     * @param input         待加密消息
     * @param inOff         输入数据 起始位置
     * @param length        输入数据 长度
     * @return GM/T 0003.1-2012 6.1 A8 输出密文 C = C1 || C2 || C3
     */
    public static byte[] encryptBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] input, int inOff, int length) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() ||
                GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return encrypt(publicKey, input);
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        //will throw SdfSDKException
        SdfECEngine sdfECEngine = new SdfECEngine(sdfCryptoType);
        sdfECEngine.init(true, new SdfECKeyParameters((ECPublicKey) publicKey));
        //will throw InvalidCipherTextException
        byte[] sdfCipher = sdfECEngine.processBlock(input, inOff, length);
        //will throw SdfSDKException
        sdfECEngine.release();
        return sdfCipher;
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType      SDF 密码类型 密码机、PCIE
     * @param privateKeyIndex    SDF私钥索引
     * @param privateKeyPassword SDF私钥访问控制码
     * @param input              密文 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     * @param inOff              输入数据 起始位置
     * @param length             输入数据 长度
     * @return 明文
     */
    public static byte[] decryptBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] input, int inOff, int length) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() ||
                GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return decrypt(new SdfPrivateKey(privateKeyIndex, privateKeyPassword), input);
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        //will throw SdfSDKException
        SdfECEngine sdfECEngineDecrypt = new SdfECEngine(sdfCryptoType);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        sdfECEngineDecrypt.init(false, new SdfECKeyParameters(sdfPrivateKey));
        //will throw InvalidCipherTextException
        byte[] out = sdfECEngineDecrypt.processBlock(input, inOff, length);
        //will throw SdfSDKException
        sdfECEngineDecrypt.release();
        return out;
    }


    /************************************************************************************
     *                                      加密解密                                    *
     *                      GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文               *
     ************************************************************************************/

    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType SDF 密码类型 密码机、PCIE
     * @param publicKey     加密者公钥
     * @param input         待加密消息
     * @param inOff         输入数据 起始位置
     * @param length        输入数据 长度
     * @return GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     */
    public static byte[] encryptASN1BySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] input, int inOff, int length) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() ||
                GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return encryptAsn1Data(publicKey, input);
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        //will throw SdfSDKException
        SdfECEngine sdfECEngine = new SdfECEngine(sdfCryptoType);
        sdfECEngine.init(true, new SdfECKeyParameters((ECPublicKey) publicKey));
        //will throw InvalidCipherTextException
        byte[] sdfCipher = sdfECEngine.processBlockASN1(input, inOff, length);
        //will throw SdfSDKException
        sdfECEngine.release();
        return sdfCipher;
    }

    /**
     * 使用 SDF 进行加密
     *
     * @param sdfCryptoType      SDF 密码类型 密码机、PCIE
     * @param privateKeyIndex    SDF私钥索引
     * @param privateKeyPassword SDF私钥访问控制码
     * @param input              密文 GM/T 0009-2012 7.2 加密数据结构 ASN1 结构密文
     * @param inOff              输入数据 起始位置
     * @param length             输入数据 长度
     * @return 明文
     */
    public static byte[] decryptASN1BySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] input, int inOff, int length) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() ||
                GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return decryptAsn1Cipher(new SdfPrivateKey(privateKeyIndex, privateKeyPassword), input);
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        //will throw SdfSDKException
        SdfECEngine sdfECEngineDecrypt = new SdfECEngine(sdfCryptoType);
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        sdfECEngineDecrypt.init(false, new SdfECKeyParameters(sdfPrivateKey));
        //will throw InvalidCipherTextException
        byte[] out = sdfECEngineDecrypt.processBlockASN1(input, inOff, length);
        //will throw SdfSDKException
        sdfECEngineDecrypt.release();
        return out;
    }

}
