package com.xdja.pki.gmssl.crypto.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.*;
import com.xdja.pki.gmssl.operator.utils.GMSSLContentVerifierProviderUtils;
import com.xdja.pki.gmssl.operator.utils.GMSSLSdfSM2SignerUtils;
import com.xdja.pki.gmssl.x509.utils.GMSSLCertUtils;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.Certificate;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.CryptoException;
import org.bouncycastle.crypto.params.AsymmetricKeyParameter;
import org.bouncycastle.crypto.params.ParametersWithID;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.crypto.signers.SM2Signer;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentVerifier;
import org.bouncycastle.operator.ContentVerifierProvider;


import java.io.IOException;
import java.io.OutputStream;
import java.security.*;
import java.security.cert.X509Certificate;
import java.security.interfaces.ECPublicKey;

public class GMSSLSM2SignUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                                      签名验签                                    *
     ************************************************************************************/

    /**
     * 使用 BC 进行签名
     *
     * @param privateKey 签名者私钥
     * @param base64Data BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signByBC(PrivateKey privateKey, String base64Data) throws CryptoException, IOException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sig = signByBC(privateKey, data);
        return GMSSLByteArrayUtils.base64Encode(sig);
    }

    /**
     * 使用 BC 进行签名
     *
     * @param privateKey 签名者私钥
     * @param data       待签名消息
     * @return 签名数据
     */
    public static byte[] signByBC(PrivateKey privateKey, byte[] data) throws CryptoException, IOException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPrivateKeyKeyParameters(privateKey);
        SM2Signer signer = new SM2Signer();
        signer.init(true, new ParametersWithRandom(keyParameter, new SecureRandom()));
        signer.update(data, 0, data.length);
        //will throw CryptoException
        return signer.generateSignature();
    }

    /**
     * 使用 密码机 进行签名
     *
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param base64Data         BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signByYunhsm(int privateKeyIndex, String privateKeyPassword, String base64Data) throws CryptoException, Exception {
        return signBySdf(SdfCryptoType.YUNHSM, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 PCIE 进行签名
     *
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param base64Data         BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signByPcie(int privateKeyIndex, String privateKeyPassword, String base64Data) throws CryptoException, Exception {
        return signBySdf(SdfCryptoType.PCIE, privateKeyIndex, privateKeyPassword, base64Data);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param base64Data         BASE64 编码 待签名消息
     * @return BASE64 编码 签名数据
     */
    public static String signBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, String base64Data) throws CryptoException, Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sig = signBySdf(sdfCryptoType, privateKeyIndex, privateKeyPassword, data);
        return GMSSLByteArrayUtils.base64Encode(sig);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] data) throws CryptoException, Exception {
        return signBySdf(sdfCryptoType, null, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param sdfCryptoType      SDF 密码类型
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdfWithUserId(SdfCryptoType sdfCryptoType, int privateKeyIndex, String privateKeyPassword, byte[] userId, byte[] data) throws CryptoException, Exception {
        return signBySdf(sdfCryptoType, userId, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 BC 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return BASE64 编码 签名数据
     */
    public static boolean verifyByBC(PublicKey publicKey, String base64Data, String base64Sign) throws IOException {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign = GMSSLByteArrayUtils.base64Decode(base64Sign);
        return verifyByBC(publicKey, data, sign);
    }

    /**
     * 使用 BC 进行验签
     *
     * @param publicKey 签名者公钥
     * @param data      签名原文
     * @param sign      签名消息
     * @return 签名数据
     */
    public static boolean verifyByBC(PublicKey publicKey, byte[] data, byte[] sign) throws IOException {
        //will throw IOException
        AsymmetricKeyParameter keyParameter = GMSSLX509Utils.convertECPublicKeyParameters(publicKey);
        SM2Signer signer = new SM2Signer();
        signer.init(false, keyParameter);
        signer.update(data, 0, data.length);
        return signer.verifySignature(sign);
    }

    /**
     * 使用 密码机 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyByYunhsm(PublicKey publicKey, String base64Data, String base64Sign) throws Exception {
        return verifyBySdf(SdfCryptoType.YUNHSM, publicKey, base64Data, base64Sign);
    }

    /**
     * 使用 PCIE 进行验签
     *
     * @param publicKey  签名者公钥
     * @param base64Data BASE64 编码 签名原文
     * @param base64Sign BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyByPcie(PublicKey publicKey, String base64Data, String base64Sign) throws Exception {
        return verifyBySdf(SdfCryptoType.PCIE, publicKey, base64Data, base64Sign);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param base64Data    BASE64 编码 签名原文
     * @param base64Sign    BASE64 编码 签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, String base64Data, String base64Sign) throws Exception {
        byte[] data = GMSSLByteArrayUtils.base64Decode(base64Data);
        byte[] sign = GMSSLByteArrayUtils.base64Decode(base64Sign);
        return verifyBySdf(sdfCryptoType, publicKey, data, sign);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param data          签名原文
     * @param sign          签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        return verifyBySdf(sdfCryptoType, null, publicKey, data, sign);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param sdfCryptoType SDF 密码类型
     * @param publicKey     签名者公钥
     * @param data          签名原文
     * @param sign          签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdfWithUserId(SdfCryptoType sdfCryptoType, PublicKey publicKey, byte[] userId, byte[] data, byte[] sign) throws Exception {
        return verifyBySdf(sdfCryptoType, userId, publicKey, data, sign);
    }

    public static boolean verifyCertByYunHsm(X509Certificate cert, PublicKey publicKey) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()
                || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM
                || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            return GMSSLCertUtils.verifyCert(publicKey, cert);
        }
        try {
            Certificate certificate = GMSSLX509Utils.convertCertificate(cert);
            return verify(certificate.getSignatureAlgorithm(), publicKey,
                    certificate.getTBSCertificate().getEncoded(),
                    certificate.getSignature().getOctets());
        } catch (Exception e) {
            throw new Exception("exception processing signature: " + e, e);
        }
    }

    public static boolean verify(AlgorithmIdentifier algorithmIdentifier, PublicKey publicKey, byte[] encoded, byte[] expected) throws Exception {
        ContentVerifierProvider verifierProvider = GMSSLContentVerifierProviderUtils.generateContentVerifierBySdf(SdfCryptoType.YUNHSM, publicKey);
        ContentVerifier verifier = verifierProvider.get(algorithmIdentifier);
        OutputStream out = verifier.getOutputStream();
        out.write(encoded);
        out.close();
        return verifier.verify(expected);
    }

    public static byte[] signBySdf(SdfCryptoType sdfCryptoType, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data) throws CryptoException, Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM ) {
            return GMSSLECSignUtils.sign(new SdfPrivateKey(privateKeyIndex), data, userId, GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName());
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        SdfECBaseSigner signer = GMSSLSdfSM2SignerUtils.generateSM2Signer(sdfCryptoType);
        return signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data);
    }

    public static byte[] signBySdfHSM(SdfCryptoType sdfCryptoType, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data) throws CryptoException, Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM ) {
            return GMSSLECSignUtils.sign(new SdfPrivateKey(privateKeyIndex), data, userId, GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName());
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        SdfECSigner signer = new SdfECSigner(sdfCryptoType);
        return signBySdf(signer, userId, privateKeyIndex, privateKeyPassword, data);
    }

    /**
     * 使用 SDF 进行签名
     *
     * @param signer             签名计算实体类
     * @param privateKeyIndex    签名者私钥索引
     * @param privateKeyPassword 签名者私钥访问密码
     * @param data               BASE64 编码 待签名消息
     * @return 二进制 签名数据
     */
    public static byte[] signBySdf(SdfECBaseSigner signer, byte[] userId, int privateKeyIndex, String privateKeyPassword, byte[] data) throws CryptoException, Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM) {
            return GMSSLECSignUtils.sign(new SdfPrivateKey(privateKeyIndex), data, userId, GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName());
        }
        //will throw Exception
        SdfPrivateKey sdfPrivateKey = GMSSLSM2KeyUtils.genSdfPrivateKey(privateKeyIndex, privateKeyPassword);
        CipherParameters parameters;
        SdfECKeyParameters sdfECKeyParameters = new SdfECKeyParameters(sdfPrivateKey);
        if (userId != null) {
            parameters = new ParametersWithID(sdfECKeyParameters, userId);
        } else {
            parameters = sdfECKeyParameters;
        }
        signer.init(true, parameters);
        signer.update(data, 0, data.length);
        byte[] signature = signer.generateSignature();
        //will throw Exception
        signer.release();
        return signature;
    }

    public static boolean verifyBySdf(SdfCryptoType sdfCryptoType, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer() || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM ) {
            return GMSSLECSignUtils.verify(publicKey, data, sign, userId, GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName());
        }
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        SdfEcSignerWithSoftDigest signer = new SdfEcSignerWithSoftDigest(sdfCryptoType);
        return verifyBySdf(signer, userId, publicKey, data, sign);
    }

    public static boolean verifyByHSM(SdfCryptoType sdfCryptoType, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        SdfECSigner signer = new SdfECSigner(sdfCryptoType);
        return verifyBySdf(signer, userId, publicKey, data, sign);
    }

    /**
     * 使用 SDF 进行验签
     *
     * @param signer    签名计算实体类
     * @param publicKey 签名者公钥
     * @param data      签名原文
     * @param sign      签名消息
     * @return 验签结果
     */
    public static boolean verifyBySdf(SdfECBaseSigner signer, byte[] userId, PublicKey publicKey, byte[] data, byte[] sign) throws Exception {
        //will throw Exception
        if (GMSSLPkiCryptoInit.isHsmServer() || GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.SANC_HSM ) {
            return GMSSLECSignUtils.verify(publicKey, data, sign, userId, GMSSLSignatureAlgorithm.SM3_WITH_SM2.getSigAlgName());
        }
        CipherParameters parameters;
        SdfECKeyParameters sdfECKeyParameters = new SdfECKeyParameters((ECPublicKey) publicKey);
        if (userId != null) {
            parameters = new ParametersWithID(sdfECKeyParameters, userId);
        } else {
            parameters = sdfECKeyParameters;
        }
        signer.init(false, parameters);
        signer.update(data, 0, data.length);
        boolean isVerify = signer.verifySignature(sign);
        //will throw Exception
        signer.release();
        return isVerify;
    }

}
