package com.xdja.pki.gmssl.crypto.utils;

import com.sansec.devicev4.api.ISDSCrypto;
import com.sansec.devicev4.gb.GBAlgorithmID_SGD;
import com.xdja.SafeKey.JNIAPI;
import com.xdja.SafeKey.utils.MiniPcieSymmetricEncryptUtils;
import com.xdja.SafeKey.utils.MiniPcieXKFUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfSymmetricKeyParameters;
import com.xdja.pki.gmssl.crypto.utils.sanc.GMSSLSancConnectionUtils;
import com.xdja.pki.gmssl.sdf.bean.SdfAlgIdSymmetric;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLCryptoType;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.security.Security;


public class GMSSLSM4ECBEncryptUtils {
    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /**
     * SM4 ECB 使用密钥加密密钥进行加密
     *
     * @param kek               密钥加密密钥
     * @param kekIndex          密钥加密密钥对应的密钥索引
     * @param data              需要加密的数据
     * @param paddingType       填充类型
     * @param keyEncryptKeyType 生成密钥加密密钥的加密类型 目前只有SdfAlgIdSymmetric.SGD_SM1_ECB 和 目前只有SdfAlgIdSymmetric.SGD_SM4_ECB
     *                          根据生成是调用的方法
     * @return 加密后的数据
     * @throws Exception
     */
    public static byte[] encryptWithKek(byte[] kek, int kekIndex, SdfAlgIdSymmetric keyEncryptKeyType, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getSymKey(kekIndex);
            byte[] bytes = decryptByBCWithNoPadding(bean.getSecretKey().getEncoded(), kek);
            return encryptByBc(bytes, data, paddingType);
        }

        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptWithKekBySdf(true, SdfCryptoType.YUNHSM, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_SM4_ECB, data);
            case DONGJIN_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptWithKekBySdf(true, SdfCryptoType.DONGJIN, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_SM4_ECB, data);
            default:
                throw new IllegalArgumentException("un support kek encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * SM4 ECB 使用密钥加密密钥进行加密
     *
     * @param kek               密钥加密密钥
     * @param kekIndex          密钥加密密钥对应的密钥索引
     * @param data              需要解密的数据
     * @param paddingType       填充类型
     * @param keyEncryptKeyType 生成密钥加密密钥的加密类型 目前只有SdfAlgIdSymmetric.SGD_SM1_ECB 和 目前只有SdfAlgIdSymmetric.SGD_SM4_ECB
     *                          根据生成是调用的方法
     * @return
     * @throws Exception
     */
    public static byte[] decryptWithKek(byte[] kek, int kekIndex, SdfAlgIdSymmetric keyEncryptKeyType, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            GMSSLHsmKeyStoreBean bean = GMSSLHsmKeyStoreUtils.getSymKey(kekIndex);
            byte[] bytes = decryptByBCWithNoPadding(bean.getSecretKey().getEncoded(), kek);
            return decryptByBc(bytes, data, paddingType);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case XDJA_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptWithKekBySdf(false, SdfCryptoType.YUNHSM, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_SM4_ECB, data);
            case DONGJIN_HSM:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptWithKekBySdf(false, SdfCryptoType.DONGJIN, paddingType, keyEncryptKeyType, kekIndex, kek, SdfAlgIdSymmetric.SGD_SM4_ECB, data);
            default:
                throw new IllegalArgumentException("un support kek encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }


    /**
     * SM4 ECB加密
     *
     * @param key         对称秘钥
     * @param data        加密数据原文
     * @param paddingType 填充模式
     * @return
     * @throws Exception
     */
    public static byte[] encrypt(byte[] key, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return encryptByBc(key, data, paddingType);
        }

        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE, paddingType, key, data);
            case XDJA_HSM:
                return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM, paddingType, key, data);
            case DONGJIN_HSM:
                return sm4SymmetricSdfWithPadding(true, SdfCryptoType.DONGJIN, paddingType, key, data);
            case MINI_PCI_E:
                switch (paddingType) {
                    case NoPadding:
                        return MiniPcieXKFUtils.sm4(key, data, JNIAPI.ECB_ENCRYPT, null);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return MiniPcieSymmetricEncryptUtils.sm4EcbWithPkcs5Padding(key, data, true);
                    case SSL3Padding:
                    default:
                        return MiniPcieSymmetricEncryptUtils.sm4EcbWithSslv3Padding(key, data, true);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                switch (paddingType) {
                    case NoPadding:
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM4_ECB, key, null, data);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        data = GMSSLX509Utils.paddingData(data, true);
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM4_ECB, key, null, data);
                    case SSL3Padding:
                    default:
                        data = GMSSLX509Utils.paddingData(data, false);
                        return cryptConnection.encrypt(GBAlgorithmID_SGD.SGD_SM4_ECB, key, null, data);
                }
            case BC:
                return encryptByBc(key, data, paddingType);
            default:
                throw new IllegalArgumentException("un support the sm4 ebc encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * 解密模式
     *
     * @param key         解密的对称秘钥
     * @param data        加密数据
     * @param paddingType 填充模式
     * @return
     * @throws Exception
     */
    public static byte[] decrypt(byte[] key, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        if (GMSSLPkiCryptoInit.isHsmServer()) {
            return decryptByBc(key, data, paddingType);
        }
        switch (GMSSLPkiCryptoInit.getCryptoType()) {
            case PCI_E:
                return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE, paddingType, key, data);
            case XDJA_HSM:
                return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM, paddingType, key, data);
            case DONGJIN_HSM:
                return sm4SymmetricSdfWithPadding(false, SdfCryptoType.DONGJIN, paddingType, key, data);
            case MINI_PCI_E:
                switch (paddingType) {
                    case NoPadding:
                        return MiniPcieXKFUtils.sm4(key, data, JNIAPI.ECB_DECRYPT, null);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        return MiniPcieSymmetricEncryptUtils.sm4EcbWithPkcs5Padding(key, data, false);
                    case SSL3Padding:
                    default:
                        return MiniPcieSymmetricEncryptUtils.sm4EcbWithSslv3Padding(key, data, false);
                }
            case SANC_HSM:
                ISDSCrypto cryptConnection = GMSSLSancConnectionUtils.getCryptConnection();
                switch (paddingType) {
                    case NoPadding:
                        return cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM4_ECB, key, null, data);
                    case PKCS7Padding:
                    case PKCS5Padding:
                        byte[] decrypt = cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM4_ECB, key, null, data);
                        return GMSSLX509Utils.removePadding(data, decrypt, true);
                    case SSL3Padding:
                    default:
                        byte[] decryptSsl3 = cryptConnection.decrypt(GBAlgorithmID_SGD.SGD_SM4_ECB, key, null, data);
                        return GMSSLX509Utils.removePadding(data, decryptSsl3, false);
                }
            case BC:
                return decryptByBc(key, data, paddingType);
            default:
                throw new IllegalArgumentException("un support the sm4 ecb decrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * 使用 BC 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBCWithNoPadding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByBCWithNoPadding(byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                key,
                data);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBCWithPKCS5Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByBCWithPKCS5Padding(byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                key,
                data);
    }

    /**
     * 使用 BC 进行加密 PKCS7Padding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByBCWithPKCS7Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByBCWithPKCS7Padding(byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS7Padding,
                key,
                data);
    }

    /**
     * 使用 BC 进行解密 NoPadding填充
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBCWithNoPadding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] decryptByBCWithNoPadding(byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                key,
                data);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 加密消息
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBCWithPKCS5Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] decryptByBCWithPKCS5Padding(byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                key,
                data);
    }

    /**
     * 使用 BC 进行加密 PKCS5Padding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 加密消息
     * @return BASE64 编码 密文消息
     */
    public static String decryptByBCWithPKCS7Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] decryptByBCWithPKCS7Padding(byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS7Padding,
                key,
                data);
    }


    /**
     * 使用 加密机Yunhsm 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithNoPadding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByYumhsmWithNoPadding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data);
    }


    /**
     * 使用 加密机Yunhsm 进行加密 PKCS5Padding填充
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithPKCS5Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByYumhsmWithPKCS5Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data);
    }

    /**
     * 使用 加密机Yunhsm 进行加密 PKCS7Padding填充
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithPKCS7Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByYumhsmWithPKCS7Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data);
    }

    /**
     * 使用 加密机Yunhsm 进行加密 SSL3Padding填充
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByYumhsmWithSSL3Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByYumhsmWithSSL3Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithNoPadding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByYumhsmWithNoPadding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithPKCS5Padding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByYumhsmWithPKCS5Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithPKCS7Padding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByYumhsmWithPKCS7Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data);
    }

    /**
     * 使用 加密机Yunhsm 进行解密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByYumhsmWithSSL3Padding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByYumhsmWithSSL3Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.YUNHSM,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data);
    }

    /**
     * 使用 PCIE 进行加密 NoPadding填充  传入明文为16整数倍
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithNoPadding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByPcieWithNoPadding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                data);
    }

    /**
     * 使用 PCIE 进行加密 PKCS5Padding
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithPKCS5Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByPcieWithPKCS5Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                data);
    }

    /**
     * 使用 PCIE 进行加密 PKCS7Padding
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithPKCS7Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByPcieWithPKCS7Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                data);
    }

    /**
     * 使用 PCIE 进行加密 SSL3Padding填充
     *
     * @param base64Key  BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Data BASE64 编码 待加密消息
     * @return BASE64 编码 密文消息
     */
    public static String encryptByPcieWithSSL3Padding(String base64Key, String base64Data) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Data)));
    }

    public static byte[] encryptByPcieWithSSL3Padding(byte[] key, byte[] data) throws Exception {
        return sm4SymmetricSdfWithPadding(true, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                data);
    }

    /**
     * 使用 PCIE 进行解密  NoPadding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息 转换为byte[]数组后长度为16整数倍
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithNoPadding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByPcieWithNoPadding(byte[] key, byte[] cipher) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.NoPadding,
                key,
                cipher);
    }

    /**
     * 使用 PCIE 进行解密 PKCS5Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithPKCS5Padding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByPcieWithPKCS5Padding(byte[] key, byte[] cipher) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS5Padding,
                key,
                cipher);
    }

    /**
     * 使用 PCIE 进行解密 PKCS7Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithPKCS7Padding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByPcieWithPKCS7Padding(byte[] key, byte[] cipher) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.PKCS7Padding,
                key,
                cipher);
    }

    /**
     * 使用 PCIE 进行解密 SSL3Padding填充
     *
     * @param base64Key    BASE64 编码 会话密钥 转换为byte[]数组后长度为16
     * @param base64Cipher BASE64 编码 加密消息
     * @return BASE64 编码 明文消息
     */
    public static String decryptByPcieWithSSL3Padding(String base64Key, String base64Cipher) throws Exception {
        return GMSSLByteArrayUtils.base64Encode(sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                GMSSLByteArrayUtils.base64Decode(base64Key),
                GMSSLByteArrayUtils.base64Decode(base64Cipher)));
    }

    public static byte[] decryptByPcieWithSSL3Padding(byte[] key, byte[] cipher) throws Exception {
        return sm4SymmetricSdfWithPadding(false, SdfCryptoType.PCIE,
                SdfSymmetricKeyParameters.PaddingType.SSL3Padding,
                key,
                cipher);
    }

    /**
     * 使用 padding模式填充的加解密公用类
     *
     * @param forEncryption 是否为加密
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           16位byte[] 会话密钥
     * @param data          二进制byte[]数据 待加密消息或已加密消息
     * @return BASE64 编码 明文消息
     */
    public static byte[] sm4SymmetricWithPaddingByPCIE(boolean forEncryption, SdfSymmetricKeyParameters.PaddingType paddingType,
                                                       byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptBySdf(forEncryption, SdfCryptoType.PCIE,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM4_ECB, data);

    }

    /**
     * 使用 padding模式填充的加解密公用类 加密机YumHsm
     *
     * @param forEncryption 是否为加密
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           16位byte[] 会话密钥
     * @param data          二进制byte[]数据 待加密消息或已加密消息
     * @return BASE64 编码 明文消息
     */
    public static byte[] sm4SymmetricWithPaddingByYunHsm(boolean forEncryption, SdfSymmetricKeyParameters.PaddingType paddingType,
                                                         byte[] key, byte[] data) throws Exception {
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptBySdf(forEncryption, SdfCryptoType.YUNHSM,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM4_ECB, data);

    }

    /**
     * 使用 padding模式填充的加解密公用类
     *
     * @param forEncryption 是否为加密
     * @param sdfCryptoType 加密使用类型 包括加密机YumHsm和PCIE卡
     * @param paddingType   填充类型 NoPadding PKCS5Padding PKCS7Padding SSL3Padding
     * @param key           16位byte[] 会话密钥
     * @param data          二进制byte[]数据 待加密消息或已加密消息
     * @return BASE64 编码 明文消息
     */
    public static byte[] sm4SymmetricSdfWithPadding(boolean forEncryption, SdfCryptoType sdfCryptoType,
                                                    SdfSymmetricKeyParameters.PaddingType paddingType,
                                                    byte[] key, byte[] data) throws Exception {
        if (GMSSLPkiCryptoInit.getCryptoType() == GMSSLCryptoType.DONGJIN_HSM) {
            sdfCryptoType = SdfCryptoType.DONGJIN;
        }
        return GMSSLSymmetricEncryptUtils.symmetricECBEncryptBySdf(forEncryption, sdfCryptoType,
                paddingType, key, SdfAlgIdSymmetric.SGD_SM4_ECB, data);

    }

    /**
     * BC SM4 ECB加密
     *
     * @param key         对称秘钥
     * @param data        加密数据原文
     * @param paddingType 填充模式
     * @return 结果
     * @throws Exception 异常
     */
    public static byte[] encryptByBc(byte[] key, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (paddingType) {
            case NoPadding:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                        GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                        key, data);
            case PKCS7Padding:
            case PKCS5Padding:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(true,
                        GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                        key, data);
            case SSL3Padding:
            default:
                throw new IllegalArgumentException("un support the sm4 ecb sslv3Padding encrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

    /**
     * BC SM4 ECB 解密
     *
     * @param key         对称秘钥
     * @param data        加密数据原文
     * @param paddingType 填充模式
     * @return 结果
     * @throws Exception 异常
     */
    public static byte[] decryptByBc(byte[] key, byte[] data, SdfSymmetricKeyParameters.PaddingType paddingType) throws Exception {
        switch (paddingType) {
            case NoPadding:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                        GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_NoPadding,
                        key, data);
            case PKCS7Padding:
            case PKCS5Padding:
                return GMSSLSymmetricEncryptUtils.symmetricECBEncryptByBC(false,
                        GMSSLSymmetricEncryptUtils.EncryptTypeByBC.SM4_ECB_PKCS5Padding,
                        key, data);
            case SSL3Padding:
            default:
                throw new IllegalArgumentException("un support the sm4 ecb sslv3Padding decrypt method with the crypto " + GMSSLPkiCryptoInit.getCryptoType());
        }
    }

}
