package com.xdja.pki.gmssl.crypto.utils.sanc;

import com.sansec.devicev4.gb.struct.key.IRSArefPrivateKey;
import com.sansec.devicev4.gb.struct.key.IRSArefPublicKey;
import com.sansec.devicev4.gb.struct.key.rsa.RSArefPublicKeyEx;
import com.sansec.devicev4.gb.struct.key.rsa.RSArefPublicKeyLite;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refPrivateKey;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refPublicKey;
import com.sansec.devicev4.gb.struct.key.sm2.SM2refSignature;
import com.xdja.pki.gmssl.core.utils.GMSSLByteArrayUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLRSAUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.utils.GMSSLRSAKeyUtils;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jcajce.provider.asymmetric.rsa.BCRSAPrivateCrtKey;
import org.bouncycastle.jcajce.provider.asymmetric.rsa.BCRSAPublicKey;
import org.bouncycastle.util.BigIntegers;

import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;

/**
 * @description: 三未信安转换工具类
 * @author: feng zhen
 * @date: 2020/6/1 16:52
 **/
public class GMSSLSancConverUtils {
    /**
     * 转换三未信安公钥至 公钥
     *
     * @param sm2refPublicKey 三未信安公钥
     * @return
     * @throws Exception
     */
    public static PublicKey converSM2PublicKey(SM2refPublicKey sm2refPublicKey) throws Exception {
        return GMSSLX509Utils.convertSM2PublicKey(sm2refPublicKey.getX(), sm2refPublicKey.getY());
    }

    /**
     * 转换三未信安私钥至私钥
     *
     * @param sm2refPrivateKey 三未信安私钥
     * @return
     * @throws Exception
     */
    public static PrivateKey converSM2PrivateKey(SM2refPrivateKey sm2refPrivateKey) throws Exception {
        return GMSSLX509Utils.convertPrivateKeyEncode(sm2refPrivateKey.getD());
    }

    /**
     * 转换公钥至三未信安公钥
     *
     * @param publicKey SM2公钥
     * @return
     * @throws Exception
     */
    public static SM2refPublicKey converSM2RefPublicKey(PublicKey publicKey) {
        BCECPublicKey bcecPublicKey = (BCECPublicKey) publicKey;
        byte[] x = bcecPublicKey.getW().getAffineX().toByteArray();
        byte[] y = bcecPublicKey.getW().getAffineY().toByteArray();
        return new SM2refPublicKey(GMSSLByteArrayUtils.changeByteArrayLength(x, 32),
                GMSSLByteArrayUtils.changeByteArrayLength(y, 32));
    }

    /**
     * 转换私钥至三未信安私钥
     *
     * @param privateKey SM2私钥
     * @return
     * @throws Exception
     */
    public static SM2refPrivateKey converSM2RefPrivateKey(PrivateKey privateKey) throws Exception {
        BCECPrivateKey bcecPrivateKey = (BCECPrivateKey) privateKey;
        byte[] s = bcecPrivateKey.getD().toByteArray();
        return new SM2refPrivateKey(GMSSLByteArrayUtils.changeByteArrayLength(s, 32));
    }

    /**
     * 转换签名值值三未信安模式
     *
     * @param signature SM2签名
     * @return
     * @throws Exception
     */
    public static SM2refSignature converSM2RefSignature(byte[] signature) throws Exception {
        BigInteger[] bigIntegers = GMSSLX509Utils.derSignatureDecode(signature);
        assert bigIntegers != null;
        byte[] r = GMSSLByteArrayUtils.changeByteArrayLength(bigIntegers[0].toByteArray(), 32);
        byte[] s = GMSSLByteArrayUtils.changeByteArrayLength(bigIntegers[1].toByteArray(), 32);
        return new SM2refSignature(r, s);
    }

    /**
     * 转换签名值值SM2模式  做ASN1转换
     *
     * @param refSignature 三未信安密码机签出签名值
     * @return
     * @throws Exception
     */
    public static byte[] converSM2Signature(SM2refSignature refSignature) throws Exception {
        return GMSSLX509Utils.derSignatureEncode(refSignature.getR(), refSignature.getS());
    }

    /**
     * 转换三未信安密码机RSA公钥格式
     *
     * @param publicKey BCRSAPublicKey
     * @return
     * @throws Exception
     */
    public static IRSArefPublicKey converRSARefPublicKey(PublicKey publicKey) throws Exception {
        BCRSAPublicKey rsaPublicKey = (BCRSAPublicKey) publicKey;
        RSArefPublicKeyLite rsArefPublicKeyLite = new RSArefPublicKeyLite();
        rsArefPublicKeyLite.bits = rsaPublicKey.getModulus().bitLength();
        rsArefPublicKeyLite.e = GMSSLByteArrayUtils.changeByteArrayLength(rsaPublicKey.getPublicExponent().toByteArray(), 256);
        rsArefPublicKeyLite.m = GMSSLByteArrayUtils.changeByteArrayLength(rsaPublicKey.getModulus().toByteArray(), 256);
        return rsArefPublicKeyLite;
    }

    /**
     * 三未信安密码机RSA公钥格式转换普通BCRSAPublicKey
     *
     * @param arefPublicKey 三未信安密码机公钥
     * @return
     * @throws Exception
     */
    public static PublicKey converRSAPublicKey(IRSArefPublicKey arefPublicKey) throws Exception {
        return GMSSLRSAUtils.convertRSAPublicKey(arefPublicKey.getM());
    }

    /**
     * 三未信安密码机RSA公钥格式转换普通RSAPublicKey
     *
     * @param arefPrivateKey 三未信安密码机私钥
     * @return
     * @throws Exception
     */
    public static PrivateKey converRSAPrivateKey(IRSArefPrivateKey arefPrivateKey) throws Exception {
        return GMSSLRSAUtils.convertRSAPrivateKey(BigIntegers.fromUnsignedByteArray(arefPrivateKey.getM()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getE()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getD()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getPrime1()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getPrime2()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getPexp1()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getPexp2()),
                BigIntegers.fromUnsignedByteArray(arefPrivateKey.getCoef()));
    }
}
