package com.xdja.pki.gmssl.keystore.utils;

import com.xdja.pki.gmssl.core.utils.GMSSLCertPathUtils;
import com.xdja.pki.gmssl.core.utils.GMSSLX509Utils;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreBean;
import com.xdja.pki.gmssl.crypto.init.GMSSLHsmKeyStoreUtils;
import com.xdja.pki.gmssl.crypto.init.GMSSLPkiCryptoInit;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.x509.utils.GMSSLExtensionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.SecretKey;
import java.io.*;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

public class GMSSLKeyStoreUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    public static final String BKS_KEYSTORE_TYPE = GMSSLCertPathUtils.KEYSTORE_TYPE_BKS;
    public static final String PKCS12_KEYSTORE_TYPE = GMSSLCertPathUtils.KEYSTORE_TYPE_PKCS12;
    public static final String JKS_KEYSTORE_TYPE = GMSSLCertPathUtils.KEYSTORE_TYPE_JKS;

    private static Logger logger = LoggerFactory.getLogger(GMSSLExtensionUtils.class);

    /**
     * 构建BKS类型KeyStore
     *
     * @param password  keystore的密码
     * @param rootCert  根证书
     * @param signAlias 签名证书别名
     * @param signKey   签名证书私钥
     * @param signCert  签名证书
     * @param encAlias  加密证书别名
     * @param encKey    加密证书私钥
     * @param encCert   加密证书
     * @deprecated 请调用generateGMSSLKeyStoreWithBKS方法
     */
    @Deprecated
    public static KeyStore generateGMSSLKeyStore(
            String password, X509Certificate rootCert,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithBKS(password, rootCert, signAlias, signKey, signCert, encAlias, encKey, encCert);
    }

    /**
     * 构建BKS类型KeyStore
     *
     * @param password  keystore的密码
     * @param rootCerts 根证书链
     * @param signAlias 签名证书别名
     * @param signKey   签名证书私钥
     * @param signCert  签名证书
     * @param encAlias  加密证书别名
     * @param encKey    加密证书私钥
     * @param encCert   加密证书
     * @deprecated 请调用generateGMSSLKeyStoreWithBKS方法
     */
    @Deprecated
    public static KeyStore generateGMSSLKeyStore(
            String password, List<X509Certificate> rootCerts,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithType(password, rootCerts, signAlias, signKey, signCert, encAlias, encKey, encCert, BKS_KEYSTORE_TYPE);
    }


    /**
     * 构建JKS类型KeyStore
     *
     * @param password  keystore的密码
     * @param rootCert  根证书
     * @param signAlias 签名证书别名
     * @param signKey   签名证书私钥
     * @param signCert  签名证书
     * @param encAlias  加密证书别名
     * @param encKey    加密证书私钥
     * @param encCert   加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithJKS(
            String password, X509Certificate rootCert,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithType(password, rootCert, signAlias, signKey, signCert, encAlias, encKey, encCert, JKS_KEYSTORE_TYPE);
    }

    /**
     * 构建JKS类型KeyStore
     *
     * @param password  keystore的密码
     * @param rootCerts 信任证书链
     * @param signAlias 签名证书别名
     * @param signKey   签名证书私钥
     * @param signCert  签名证书
     * @param encAlias  加密证书别名
     * @param encKey    加密证书私钥
     * @param encCert   加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithJKS(
            String password, List<X509Certificate> rootCerts,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithType(password, rootCerts, signAlias, signKey, signCert, encAlias, encKey, encCert, JKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS类型KeyStore
     *
     * @param password  keystore的密码
     * @param rootCert  根证书
     * @param signAlias 签名证书别名
     * @param signKey   签名证书私钥
     * @param signCert  签名证书
     * @param encAlias  加密证书别名
     * @param encKey    加密证书私钥
     * @param encCert   加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithBKS(
            String password, X509Certificate rootCert,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithType(password, rootCert, signAlias, signKey, signCert, encAlias, encKey, encCert, BKS_KEYSTORE_TYPE);
    }

    public static KeyStore generateGMSSLKeyStoreWithPKCS12(
            String password, X509Certificate rootCert,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithType(password, rootCert, signAlias, signKey, signCert, encAlias, encKey, encCert, PKCS12_KEYSTORE_TYPE);
    }


    /**
     * 构建JKS类型KeyStore
     *
     * @param password           keystore的密码
     * @param rootCerts          信任证书链
     * @param signAlias          签名证书别名
     * @param privateKeyIndex    密钥索引
     * @param signCert           签名证书
     * @param encAlias           加密证书别名
     * @param privateKeyPassword 私钥访问控制码
     * @param encCert            加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithJKS(
            String password, List<X509Certificate> rootCerts,
            int privateKeyIndex, String privateKeyPassword,
            String signAlias, X509Certificate signCert,
            String encAlias, X509Certificate encCert
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(privateKeyIndex, privateKeyPassword);
        return generateGMSSLKeyStoreWithType(password, rootCerts, signAlias, sdfPrivateKey, signCert, encAlias, sdfPrivateKey, encCert, JKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS类型KeyStore
     *
     * @param password           keystore的密码
     * @param rootCert           根证书
     * @param signAlias          签名证书别名
     * @param privateKeyIndex    密钥索引
     * @param privateKeyPassword 私钥访问控制码
     * @param signCert           签名证书
     * @param encAlias           加密证书别名
     * @param encCert            加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithBKS(
            String password, X509Certificate rootCert,
            int privateKeyIndex, String privateKeyPassword,
            String signAlias, X509Certificate signCert,
            String encAlias, X509Certificate encCert
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(privateKeyIndex, privateKeyPassword);
        return generateGMSSLKeyStoreWithType(password, rootCert, signAlias, sdfPrivateKey, signCert, encAlias, sdfPrivateKey, encCert, BKS_KEYSTORE_TYPE);
    }


    /**
     * 构建BKS类型KeyStore
     *
     * @param password           keystore的密码
     * @param rootCerts          信任证书链
     * @param signAlias          签名证书别名
     * @param privateKeyIndex    密钥索引
     * @param privateKeyPassword 私钥访问控制码
     * @param signCert           签名证书
     * @param encAlias           加密证书别名
     * @param encCert            加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithBKS(
            String password, List<X509Certificate> rootCerts,
            int privateKeyIndex, String privateKeyPassword,
            String signAlias, X509Certificate signCert,
            String encAlias, X509Certificate encCert
    ) throws Exception {
        SdfPrivateKey sdfPrivateKey = new SdfPrivateKey(privateKeyIndex, privateKeyPassword);
        return generateGMSSLKeyStoreWithType(password, rootCerts, signAlias, sdfPrivateKey, signCert, encAlias, sdfPrivateKey, encCert, BKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS类型KeyStore
     *
     * @param password  keystore的密码
     * @param rootCerts 信任证书链
     * @param signAlias 签名证书别名
     * @param signKey   签名证书私钥
     * @param signCert  签名证书
     * @param encAlias  加密证书别名
     * @param encKey    加密证书私钥
     * @param encCert   加密证书
     */
    public static KeyStore generateGMSSLKeyStoreWithBKS(
            String password, List<X509Certificate> rootCerts,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert
    ) throws Exception {
        return generateGMSSLKeyStoreWithType(password, rootCerts, signAlias, signKey, signCert, encAlias, encKey, encCert, BKS_KEYSTORE_TYPE);
    }

    /**
     * 构建KeyStore
     *
     * @param password     keystore的密码
     * @param rootCert     根证书
     * @param signAlias    签名证书别名
     * @param signKey      签名证书私钥
     * @param signCert     签名证书
     * @param encAlias     加密证书别名
     * @param encKey       加密证书私钥
     * @param encCert      加密证书
     * @param keystoreType keyStore类型 BKS ？JKS
     */
    public static KeyStore generateGMSSLKeyStoreWithType(
            String password, X509Certificate rootCert,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert, String keystoreType
    ) throws Exception {
        List<X509Certificate> trustCerts = new ArrayList<>();
        trustCerts.add(rootCert);
        return generateGMSSLKeyStoreWithType(password, trustCerts, signAlias, signKey, signCert, encAlias, encKey, encCert, keystoreType);
    }


    /**
     * 构建KeyStore
     *
     * @param password     keystore的密码
     * @param trustCerts   信任证书链
     * @param signAlias    签名证书别名
     * @param signKey      签名证书私钥
     * @param signCert     签名证书
     * @param encAlias     加密证书别名
     * @param encKey       加密证书私钥
     * @param encCert      加密证书
     * @param keystoreType keyStore类型 BKS ？JKS
     */
    public static KeyStore generateGMSSLKeyStoreWithType(
            String password, List<X509Certificate> trustCerts,
            String signAlias, PrivateKey signKey, X509Certificate signCert,
            String encAlias, PrivateKey encKey, X509Certificate encCert, String keystoreType
    ) throws Exception {
        KeyStore keyStore = generateKeystore(keystoreType);
        keyStore.load(null, null);
        if (GMSSLPkiCryptoInit.isHsmServer() && signKey instanceof SdfPrivateKey) {
            SdfPrivateKey sdfPrivateKey = (SdfPrivateKey) signKey;
            GMSSLHsmKeyStoreBean signBean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), false);
            GMSSLHsmKeyStoreBean encBean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), true);
            keyStore.setKeyEntry(signAlias, signBean.getPrivateKey(), password.toCharArray(), generateCertArrays(trustCerts, signCert));
            keyStore.setKeyEntry(encAlias, encBean.getPrivateKey(), password.toCharArray(), generateCertArrays(trustCerts, encCert));
        } else {
            keyStore.setKeyEntry(signAlias, signKey, password.toCharArray(), generateCertArrays(trustCerts, signCert));
            keyStore.setKeyEntry(encAlias, encKey, password.toCharArray(), generateCertArrays(trustCerts, encCert));
        }
        return keyStore;
    }

    /**
     * 构建KeyStore
     *
     * @param password   keystore的密码
     * @param trustCerts 信任证书链
     * @param signAlias  服务器证书别名
     * @param signKey    服务器证书私钥
     * @param signCert   签名证书
     */
    public static KeyStore generateGMSSLKeyStoreWithJKS(
            String password, List<X509Certificate> trustCerts,
            String signAlias, PrivateKey signKey, X509Certificate signCert) throws Exception {
        KeyStore keyStore = generateKeystore(JKS_KEYSTORE_TYPE);
        keyStore.load(null, null);

        if (GMSSLPkiCryptoInit.isHsmServer() && signKey instanceof SdfPrivateKey) {
            SdfPrivateKey sdfPrivateKey = (SdfPrivateKey) signKey;
            GMSSLHsmKeyStoreBean signBean = GMSSLHsmKeyStoreUtils.getAsymKey(sdfPrivateKey.getIndex(), false);
            keyStore.setKeyEntry(signAlias, signBean.getPrivateKey(), password.toCharArray(), generateCertArrays(trustCerts, signCert));
        } else {
            keyStore.setKeyEntry(signAlias, signKey, password.toCharArray(), generateCertArrays(trustCerts, signCert));
        }
        return keyStore;
    }

    private static X509Certificate[] generateCertArrays(List<X509Certificate> trustCerts, X509Certificate cert) {
        X509Certificate[] certArrays = new X509Certificate[trustCerts.size() + 1];
        certArrays[0] = cert;
        for (int i = 1; i < certArrays.length; i++) {
            certArrays[i] = trustCerts.get(i - 1);
        }
        return certArrays;
    }

    /**
     * 构建信任证书KeyStore
     *
     * @param rootCert 根证书
     * @deprecated 请调用generateGMSSLTrustStoreWithBKS方法
     */
    @Deprecated
    public static KeyStore generateGMSSLTrustStore(X509Certificate rootCert) throws Exception {
        return generateGMSSLTrustStoreWithBKS(rootCert);
    }

    /**
     * 构建BKS类型信任证书KeyStore
     *
     * @param rootCert 根证书
     */
    public static KeyStore generateGMSSLTrustStoreWithBKS(X509Certificate rootCert) throws Exception {
        return generateGMSSLTrustStoreWithType(rootCert, BKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS类型信任证书KeyStore
     *
     * @param rootCert 根证书
     */
    public static KeyStore generateGMSSLTrustStoreWithJKS(X509Certificate rootCert) throws Exception {
        return generateGMSSLTrustStoreWithType(rootCert, JKS_KEYSTORE_TYPE);
    }

    /**
     * 构建信任证书KeyStore
     *
     * @param rootCert     根证书
     * @param keystoreType 构建的keyStore类型 BKS ? JKS
     */
    public static KeyStore generateGMSSLTrustStoreWithType(X509Certificate rootCert, String keystoreType) throws Exception {
        KeyStore keyStore = generateKeystore(keystoreType);
        keyStore.load(null, null);
        keyStore.setCertificateEntry("trust", rootCert);
        return keyStore;
    }

    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCerts 根证书
     * @deprecated 请调用 generateGMSSLTrustStoreWithBKS方法
     */
    @Deprecated
    public static KeyStore generateGMSSLTrustStore(X509Certificate[] rootCerts) throws Exception {

        return generateGMSSLTrustStoreWithBKS(rootCerts);
    }


    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCerts 根证书
     */
    public static KeyStore generateGMSSLTrustStoreWithBKS(X509Certificate[] rootCerts) throws Exception {
        return generateGMSSLTrustStoreWithType(rootCerts, BKS_KEYSTORE_TYPE);
    }

    /**
     * 构建JKS信任证书KeyStore
     *
     * @param rootCerts 根证书
     */
    public static KeyStore generateGMSSLTrustStoreWithJKS(X509Certificate[] rootCerts) throws Exception {
        return generateGMSSLTrustStoreWithType(rootCerts, JKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCerts    根证书
     * @param keystoreType 构建的keyStore类型 BKS ? JKS
     */
    public static KeyStore generateGMSSLTrustStoreWithType(X509Certificate[] rootCerts, String keystoreType) throws Exception {
        KeyStore keyStore = generateKeystore(keystoreType);
        keyStore.load(null, null);
        for (int i = 0; i < rootCerts.length; i++) {
            keyStore.setCertificateEntry("trust-" + i, rootCerts[i]);
        }
        return keyStore;
    }

    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCert 根证书
     * @deprecated 请调用 generateGMSSLTrustStoreWithBKS方法
     */
    @Deprecated
    public static KeyStore generateGMSSLTrustStore(List<X509Certificate> rootCert) throws Exception {
        return generateGMSSLTrustStoreWithBKS(rootCert);
    }

    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCert 根证书
     */
    public static KeyStore generateGMSSLTrustStoreWithBKS(List<X509Certificate> rootCert) throws Exception {
        return generateGMSSLTrustStoreWithType(rootCert, BKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCert 根证书
     */
    public static KeyStore generateGMSSLTrustStoreWithJKS(List<X509Certificate> rootCert) throws Exception {
        return generateGMSSLTrustStoreWithType(rootCert, JKS_KEYSTORE_TYPE);
    }

    /**
     * 构建BKS信任证书KeyStore
     *
     * @param rootCert     根证书
     * @param keystoreType 构建的keyStore类型 BKS ? JKS
     */
    public static KeyStore generateGMSSLTrustStoreWithType(List<X509Certificate> rootCert, String keystoreType) throws Exception {
        KeyStore keyStore = generateKeystore(keystoreType);
        keyStore.load(null, null);
        for (int i = 0; i < rootCert.size(); i++) {
            keyStore.setCertificateEntry("trust-" + i, rootCert.get(i));
        }
        return keyStore;
    }

    /**
     * 将KeyStore保存为文件
     *
     * @param keyStore 需要保存的KeyStore
     * @param password KeyStore的密码
     * @param path     保存的路径
     * @param name     保存的名称
     */
    public static void saveGMSSLKeyStore(
            KeyStore keyStore, String password, String path, String name
    ) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        saveGMSSLKeyStoreFullName(keyStore, password, path, name + ".keystore");
    }

    public static void saveGMSSLKeyStoreFullName(
            KeyStore keyStore, String password, String path, String fullName
    ) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        File file = new File(path);
        if (!file.exists()) {
            file.mkdirs();
        }
        saveGMSSLKeyStoreFullName(keyStore, password, path + File.separator + fullName);
    }

    public static void saveGMSSLKeyStoreFullName(
            KeyStore keyStore, String password, String fullPath
    ) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        FileOutputStream out = new FileOutputStream(fullPath);
        keyStore.store(out, password.toCharArray());
        out.close();
    }

    /**
     * 将KeyStore保存为pfx文件
     *
     * @param keyStore 需要保存的KeyStore
     * @param password KeyStore的密码
     * @param path     保存的路径
     * @param name     保存的名称
     */
    public static void saveGMSSLPfx(
            KeyStore keyStore, String password, String path, String name
    ) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        FileOutputStream out = new FileOutputStream(path + "/" + name + ".pfx");
        keyStore.store(out, password.toCharArray());
        out.close();
    }

    /**
     * 将KeyStore保存为pfx文件
     *
     * @param keyStore 需要保存的KeyStore
     * @param password KeyStore的密码
     * @param path     保存的路径
     */
    public static void saveGMSSLPfx(
            KeyStore keyStore, String password, String path
    ) throws KeyStoreException, NoSuchProviderException, CertificateException, IOException, NoSuchAlgorithmException, UnrecoverableKeyException {
        KeyStore write = KeyStore.getInstance("PKCS12", BouncyCastleProvider.PROVIDER_NAME);
        write.load(null, null);
        Enumeration<String> aliases = keyStore.aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (keyStore.getCertificate(alias) != null) {
                write.setCertificateEntry(alias, keyStore.getCertificate(alias));
            }
            if (keyStore.getKey(alias, password.toCharArray()) != null) {
                write.setKeyEntry(alias, keyStore.getKey(alias, password.toCharArray()), password.toCharArray(), keyStore.getCertificateChain(alias));
            }
            String name = alias + "_" + password;
            saveGMSSLPfx(write, password, path, name);
        }

    }


    /**
     * 将KeyStore保存为文件
     *
     * @param keyStore 需要保存的KeyStore
     * @param password KeyStore的密码
     * @param path     保存的路径
     * @param name     保存的名称
     */
    public static void saveGMSSLBKSKeyStore(
            KeyStore keyStore, String password, String path, String name
    ) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        FileOutputStream out = new FileOutputStream(path + "/" + name + ".bks");
        keyStore.store(out, password.toCharArray());
        out.close();
    }

    /**
     * 将KeyStore保存为文件
     *
     * @param keyStore 需要保存的KeyStore
     * @param password KeyStore的密码
     * @param path     保存的路径
     * @param name     保存的名称
     */
    public static void saveGMSSLJKSKeyStore(
            KeyStore keyStore, String password, String path, String name
    ) throws IOException, CertificateException, NoSuchAlgorithmException, KeyStoreException {
        FileOutputStream out = new FileOutputStream(path + "/" + name + ".jks");
        keyStore.store(out, password.toCharArray());
        out.close();
    }

    /**
     * 读取KeyStore文件并保存值另一个地方
     *
     * @param file     原KeyStore文件路径
     * @param password KeyStore的密码
     * @param path     保存的路径
     * @deprecated 请调用 saveGMSSLKeyStoreEntryWithBKS方法
     */
    @Deprecated
    public static void saveGMSSLKeyStoreEntry(String file, String password, String path) throws Exception {
        saveGMSSLKeyStoreEntryWithBKS(file, password, path);
    }

    /**
     * 读取KeyStore文件并保存值另一个地方
     *
     * @param file     原KeyStore文件路径
     * @param password KeyStore的密码
     * @param path     保存的路径
     */
    public static void saveGMSSLKeyStoreEntryWithBKS(String file, String password, String path) throws Exception {
        saveGMSSLKeyStoreEntryWithType(file, password, path, BKS_KEYSTORE_TYPE);
    }

    /**
     * 读取KeyStore文件并保存值另一个地方
     *
     * @param file     原KeyStore文件路径
     * @param password KeyStore的密码
     * @param path     保存的路径
     */
    public static void saveGMSSLKeyStoreEntryWithJKS(String file, String password, String path) throws Exception {
        saveGMSSLKeyStoreEntryWithType(file, password, path, JKS_KEYSTORE_TYPE);
    }

    /**
     * 读取KeyStore文件并保存值另一个地方
     *
     * @param file         原KeyStore文件路径
     * @param password     KeyStore的密码
     * @param path         保存的路径
     * @param keystoreType keyStore类型
     */
    public static void saveGMSSLKeyStoreEntryWithType(String file, String password, String path, String keystoreType) throws Exception {
        InputStream in = GMSSLX509Utils.readInputStreamFromPath(file);
        KeyStore keyStore = generateKeystore(keystoreType);
        keyStore.load(in, password.toCharArray());
        saveGMSSLKeyStore(keyStore, password, path);
    }

    /**
     * 读取BKS类型KeyStore文件并打印详细信息
     *
     * @param file     KeyStore文件路径
     * @param password KeyStore的密码
     * @deprecated 请调用printGMSSLKeyStoreWithBKS方法
     */
    @Deprecated
    public static void printGMSSLKeyStore(String file, String password) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException, IOException, NoSuchProviderException, CertificateException {
        printGMSSLKeyStoreWithBKS(file, password);
    }

    /**
     * 读取BKS类型KeyStore文件并打印详细信息
     *
     * @param file     KeyStore文件路径
     * @param password KeyStore的密码
     */
    public static void printGMSSLKeyStoreWithBKS(String file, String password) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException, IOException, NoSuchProviderException, CertificateException {
        printGMSSLKeyStoreWithType(file, password, BKS_KEYSTORE_TYPE);
    }

    /**
     * 读取JKS类型KeyStore文件并打印详细信息
     *
     * @param file     KeyStore文件路径
     * @param password KeyStore的密码
     */
    public static void printGMSSLKeyStoreWithJKS(String file, String password) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException, IOException, NoSuchProviderException, CertificateException {
        printGMSSLKeyStoreWithType(file, password, JKS_KEYSTORE_TYPE);
    }

    /**
     * 读取KeyStore文件并打印详细信息
     *
     * @param file         KeyStore文件路径
     * @param password     KeyStore的密码
     * @param keystoreType keystore类型
     */
    public static void printGMSSLKeyStoreWithType(String file, String password, String keystoreType) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException, IOException, NoSuchProviderException, CertificateException {
        InputStream in = GMSSLX509Utils.readInputStreamFromPath(file);
        KeyStore keyStore = generateKeystore(keystoreType);
        keyStore.load(in, password.toCharArray());
        printGMSSLKeyStore(keyStore, password);
    }

    /**
     * 读取KeyStore文件并打印详细信息
     *
     * @param keyStore KeyStore对象
     * @param password KeyStore的密码
     */
    public static void printGMSSLKeyStore(KeyStore keyStore, String password) throws KeyStoreException, UnrecoverableKeyException, NoSuchAlgorithmException {
        System.out.println("Keystore type: " + keyStore.getType());
        System.out.println("Keystore provider: " + keyStore.getProvider());
        System.out.println();
        System.out.println("Your keystore contains " + keyStore.size() + " entries");
        Enumeration<String> aliases = keyStore.aliases();
        while (aliases.hasMoreElements()) {
            System.out.println();
            String alias = aliases.nextElement();
            System.out.println("Alias name: " + alias);
            System.out.println("Creation date: " + keyStore.getCreationDate(alias));
            if (keyStore.isKeyEntry(alias)) {
                System.out.println("Entry type: PrivateKeyEntry");
                Key key = keyStore.getKey(alias, password.toCharArray());
                System.out.println("Key Algorithm: " + key.getAlgorithm());
                if (key instanceof SecretKey) {
                    SdfPrivateKey sdfPrivateKey = SdfPrivateKey.getInstance(key);
                    System.out.println(sdfPrivateKey);
                } else {
                    System.out.println(key);
                }
                Certificate[] certificateChain = keyStore.getCertificateChain(alias);
                System.out.println("Certificate chain length: " + certificateChain.length);
                for (int i = 1; i <= certificateChain.length; i++) {
                    System.out.println("Certificate[" + i + "]: ");
                    System.out.println(certificateChain[i - 1]);
                }
            } else if (keyStore.isCertificateEntry(alias)) {
                System.out.println("Entry type: CertificateEntry");
                Certificate certificate = keyStore.getCertificate(alias);
                System.out.println(certificate);
            } else {
                System.out.println("UnKnown type alias : " + alias);
            }
        }
    }

    /**
     * 保存KeyStore文件
     *
     * @param keyStore KeyStore对象
     * @param password KeyStore的密码
     * @param path     保存的路径
     */
    public static void saveGMSSLKeyStore(KeyStore keyStore, String password, String path) throws Exception {
        Enumeration<String> aliases = keyStore.aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (keyStore.isKeyEntry(alias)) {
                Key key = keyStore.getKey(alias, password.toCharArray());
                String pri = alias + "_pri";
                if (key instanceof SecretKey) {
                    SdfPrivateKey sdfPrivateKey = SdfPrivateKey.getInstance(key);
                    assert sdfPrivateKey != null;
                    GMSSLX509Utils.writeSdfPrivateKey(path, pri, sdfPrivateKey);
                } else {
                    GMSSLX509Utils.writePrivateKeyToPem(path, pri, (PrivateKey) key);
                }
                Certificate[] certificateChain = keyStore.getCertificateChain(alias);
                for (int i = 1; i <= certificateChain.length; i++) {
                    GMSSLX509Utils.writeCertificateToPem(path, alias + "_cert_" + i, (X509Certificate) certificateChain[i - 1]);
                }
            } else if (keyStore.isCertificateEntry(alias)) {
                Certificate certificate = keyStore.getCertificate(alias);
                GMSSLX509Utils.writeCertificateToPem(path, alias + "_cert", (X509Certificate) certificate);
            } else {
                logger.error("UnKnown type alias : " + alias);
            }
        }
    }

    private static KeyStore generateKeystore(String keystoreType) throws NoSuchProviderException, KeyStoreException {
        if (BKS_KEYSTORE_TYPE.equals(keystoreType) || PKCS12_KEYSTORE_TYPE.equals(keystoreType)) {
            return KeyStore.getInstance(keystoreType, BouncyCastleProvider.PROVIDER_NAME);
        } else {
            return KeyStore.getInstance(keystoreType);
        }
    }

    public static KeyStore readKeyStoreFromPath(String filename, char[] pw) throws Exception {
        KeyStore keyStore;
        try {
            keyStore = readKeyStoreFromPath(filename, pw, JKS_KEYSTORE_TYPE);
        } catch (Exception e) {
            keyStore = readKeyStoreFromPath(filename, pw, BKS_KEYSTORE_TYPE);
        }
        return keyStore;
    }

    public static KeyStore readKeyStoreFromPath(String filename, char[] pw, String keyStoreType) throws Exception {
        InputStream in = GMSSLX509Utils.readInputStreamFromPath(filename);
        return readKeyStoreFromStream(pw, keyStoreType, in);
    }

    public static KeyStore readKeyStoreFromBytes(char[] pw, String keyStoreType, byte[] in) throws Exception {
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(in);
        return readKeyStoreFromStream(pw, keyStoreType, byteArrayInputStream);
    }

    public static KeyStore readKeyStoreFromStream(char[] pw, String keyStoreType, InputStream in) throws Exception {
        KeyStore keyStore;
        if (BKS_KEYSTORE_TYPE.equals(keyStoreType)) {
            keyStore = KeyStore.getInstance(keyStoreType, BouncyCastleProvider.PROVIDER_NAME);
        } else {
            keyStore = KeyStore.getInstance(keyStoreType);
        }
        keyStore.load(in, pw);
        return keyStore;
    }

    public static Certificate readCertificateFromKeyStore(KeyStore keyStore, String alias) throws KeyStoreException {
        return keyStore.getCertificate(alias);
    }

    public static PrivateKey readPrivateKeyFromKeyStore(String filename, char[] pw, String alias) throws Exception {
        KeyStore keyStore = readKeyStoreFromPath(filename, pw);
        return readPrivateKeyFromKeyStore(keyStore, pw, alias);
    }

    public static PrivateKey readPrivateKeyFromKeyStore(KeyStore keyStore, char[] pw, String alias) throws Exception {
        return (PrivateKey) keyStore.getKey(alias, pw);
    }

}
