package com.xdja.pki.gmssl.operator.utils;

import com.xdja.pki.gmssl.crypto.sdf.SdfContentSigner;
import com.xdja.pki.gmssl.crypto.sdf.SdfCryptoType;
import com.xdja.pki.gmssl.crypto.sdf.SdfPrivateKey;
import com.xdja.pki.gmssl.sdf.SdfSDKException;
import com.xdja.pki.gmssl.x509.utils.bean.GMSSLSignatureAlgorithm;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.*;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

import java.security.PrivateKey;
import java.security.Security;

public class GMSSLContentSignerUtils {

    static {
        if (Security.getProvider(BouncyCastleProvider.PROVIDER_NAME) == null) {
            Security.addProvider(new BouncyCastleProvider());
        }
    }

    /************************************************************************************
     *                             ContentSigner 生成                                   *
     ************************************************************************************/

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param signatureAlgorithm 签名算法 GMSSLSignatureAlgorithm 枚举
     * @param privateKey         签发者私钥
     * @return ContentSigner 内容签发体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByBC(GMSSLSignatureAlgorithm signatureAlgorithm, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerByBC(signatureAlgorithm.getSigAlgName(), privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param algorithm  签名算法 GMSSLSignatureAlgorithm 枚举
     * @param privateKey 签发者私钥
     * @return ContentSigner 内容签发体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByBC(String algorithm, PrivateKey privateKey, boolean isWithParam) throws OperatorCreationException {
        if (isWithParam) {
            return generateContentSignerByBCWithParam(algorithm, privateKey);
        } else {
            return generateContentSignerByBC(algorithm, privateKey);
        }
    }

    public static ContentSigner generateContentSignerByBC(String algorithm, PrivateKey privateKey) throws OperatorCreationException {
        JcaContentSignerBuilder contentSignerBuilder = new JcaContentSignerBuilder(algorithm);
        return contentSignerBuilder.build(privateKey);
    }

    public static ContentSigner generateContentSignerByBCWithParam(String algorithm, PrivateKey privateKey) throws OperatorCreationException {
        GMSSLJcaContentSignerBuilder contentSignerBuilder = new GMSSLJcaContentSignerBuilder(algorithm);
        return contentSignerBuilder.build(privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @param algorithm  签名算法 GMSSLSignatureAlgorithm 枚举
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByYunhsm(String algorithm, PrivateKey privateKey, boolean isWithParam) throws OperatorCreationException {
        return generateContentSignerByYunhsm(algorithm, privateKey, isWithParam, 256);
    }

    public static ContentSigner generateContentSignerByYunhsm(String algorithm, PrivateKey privateKey, boolean isWithParam, int bits) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.YUNHSM, algorithm, privateKey, isWithParam, bits);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @param algorithm  签名算法 GMSSLSignatureAlgorithm 枚举
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByYunhsm(String algorithm, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.YUNHSM, algorithm, privateKey);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerByPcie(String algorithm, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.PCIE, algorithm, privateKey);
    }

    public static ContentSigner generateContentSignerByPcie(String algorithm, PrivateKey privateKey, boolean isWithParam) throws OperatorCreationException {
        return generateContentSignerBySDF(SdfCryptoType.PCIE, algorithm, privateKey, isWithParam);
    }

    /**
     * 生成 ContentSigner 内容签名实体类 支持 BC（BC软实现）/YUNHSM（密码机）/PCIE（PCI-E卡）
     *
     * @param privateKey 签发者私钥
     * @return ContentSigner 内容签发实体类
     * @throws OperatorCreationException 生成异常
     */
    public static ContentSigner generateContentSignerBySDF(SdfCryptoType sdfCryptoType, String algorithm, PrivateKey privateKey, boolean isWithParam) throws OperatorCreationException {
        return generateContentSignerBySDF(sdfCryptoType, algorithm, privateKey, isWithParam, 256);
    }

    public static ContentSigner generateContentSignerBySDF(SdfCryptoType sdfCryptoType, String algorithm, PrivateKey privateKey, boolean isWithParam, int bits) throws OperatorCreationException {
        AlgorithmIdentifier algorithmIdentifier = GMSSLSignatureAlgorithm.convertAlgorithmIdentifier(algorithm, isWithParam);
        if (!(privateKey instanceof SdfPrivateKey)) {
            throw new OperatorCreationException(" un support private key type " + privateKey.getClass());
        } else {
            try {
                return new SdfContentSigner(sdfCryptoType, algorithmIdentifier, (SdfPrivateKey) privateKey, bits);
            } catch (SdfSDKException e) {
                throw new OperatorCreationException(e.getMessage());
            }
        }
    }

    public static ContentSigner generateContentSignerBySDF(SdfCryptoType sdfCryptoType, String algorithm, PrivateKey privateKey) throws OperatorCreationException {
        return generateContentSignerBySDF(sdfCryptoType, algorithm, privateKey, false);
    }

    public static ContentSigner generateContentSignerBySDFHSM(SdfCryptoType sdfCryptoType, String algorithm, PrivateKey privateKey) throws OperatorCreationException {
        AlgorithmIdentifier algorithmIdentifier = GMSSLSignatureAlgorithm.convertAlgorithmIdentifier(algorithm, false);
        if (!(privateKey instanceof SdfPrivateKey)) {
            throw new OperatorCreationException(" un support private key type " + privateKey.getClass());
        } else {
            try {
                return new SdfContentSigner(sdfCryptoType, algorithmIdentifier, (SdfPrivateKey) privateKey, false);
            } catch (SdfSDKException e) {
                throw new OperatorCreationException(e.getMessage());
            }
        }
    }
}
